package com.vaadin.copilot.plugins.themeeditor.handlers;

import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.vaadin.copilot.plugins.themeeditor.ThemeEditorCommand;
import com.vaadin.copilot.plugins.themeeditor.messages.BaseResponse;
import com.vaadin.copilot.plugins.themeeditor.messages.ClassNameRequest;
import com.vaadin.copilot.plugins.themeeditor.messages.ClassNameResponse;
import com.vaadin.copilot.plugins.themeeditor.utils.HasThemeModifier;
import com.vaadin.copilot.plugins.themeeditor.utils.MessageHandler;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.shared.util.SharedUtil;

import org.apache.commons.io.FilenameUtils;
import tools.jackson.databind.JsonNode;

/**
 * Returns the class name of given component. If class name exists in theme
 * editor, it returns the existing one otherwise suggests a new one.
 */
public class ClassNameHandler implements MessageHandler {
    private static final Pattern SUGGEST_CLASS_NAME_PATTERN = Pattern.compile("(?!^)(?=[A-Z][a-z])");
    private final HasThemeModifier hasThemeModifier;

    public ClassNameHandler(HasThemeModifier hasThemeModifier) {
        this.hasThemeModifier = hasThemeModifier;
    }

    @Override
    public BaseResponse handle(JsonNode data) {
        List<String> existingClassNames = hasThemeModifier.getThemeModifier().getExistingClassNames();
        ClassNameRequest request = JacksonUtils.readToObject(data, ClassNameRequest.class);
        return findExistingClassName(existingClassNames, request)
                .orElseGet(() -> suggestClassName(existingClassNames, request));
    }

    private BaseResponse suggestClassName(List<String> existingClassNames, ClassNameRequest request) {
        String baseName = SharedUtil
                .upperCamelCaseToDashSeparatedLowerCase(FilenameUtils.getBaseName(request.getFileName())) + '-'
                + SharedUtil.upperCamelCaseToDashSeparatedLowerCase(request.getComponentDisplayName());
        baseName = baseName.replace("@", ""); // @ is not valid in selectors and
        // used in FS router file names
        String name = baseName;
        String selectorName = request.getComponentName() + "." + name;
        int i = 1;
        while (existingClassNames.contains(selectorName)) {
            name = baseName + "-" + i;
            selectorName = appendComponentTag(request.getComponentName(), name);
            i++;
        }
        String dashedName = dashify(name);
        return new ClassNameResponse(dashedName, true);
    }

    private Optional<BaseResponse> findExistingClassName(List<String> existingClassNames, ClassNameRequest request) {
        for (String componentClass : request.getClassList()) {
            if (existingClassNames.contains(appendComponentTag(request.getComponentName(), componentClass))) {
                return Optional.of(new ClassNameResponse(componentClass, false));
            }
        }
        return Optional.empty();
    }

    private String dashify(String text) {
        if (text == null) {
            return null;
        }
        final String subst = "-";
        final Matcher matcher = SUGGEST_CLASS_NAME_PATTERN.matcher(text);
        return matcher.replaceAll(subst).toLowerCase(Locale.ENGLISH).replaceAll("-+", "-");
    }

    private String appendComponentTag(String tag, String str) {
        return tag + "." + str;
    }

    @Override
    public String getCommandName() {
        return ThemeEditorCommand.GET_CLASS_NAME.getValue();
    }
}
