package com.vaadin.copilot.plugins.testgeneration;

import java.io.File;
import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.FlowUtil;
import com.vaadin.copilot.JavaSourcePathDetector;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.ai.AICommunicationUtil;
import com.vaadin.copilot.ai.AIConstants;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.flow.component.Tag;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.router.Route;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.ProKey;
import com.vaadin.uitest.TestCodeGenerator;
import com.vaadin.uitest.ai.utils.PromptUtils;
import com.vaadin.uitest.model.TestFramework;
import com.vaadin.uitest.model.UiRoute;

import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ObjectNode;

public class GenerateTestsHandler extends CopilotCommand {

    private static final String GENERATE_TESTS_COMMAND = "generate-tests";
    private static final String RESPONSE_ERROR_KEY = "error";
    private static final String RESPONSE_TEST_GENERATION_RESULT_SUMMARY_KEY = "testGenerationResultSummary";
    private static final String RESPONSE_TEST_GENERATION_GENERATED_FILE_KEY = "testGenerationResultFile";

    private static final Logger LOGGER = LoggerFactory.getLogger(GenerateTestsHandler.class);

    private final TestGenerationServerClient client = new TestGenerationServerClient();

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (!GENERATE_TESTS_COMMAND.equals(command)) {
            return false;
        }

        UserInfoServerClient.throwIfAIUsageDisabled();
        ProKey proKey = getProKey();
        String machineId = getMachineId();
        if (proKey == null && machineId == null) {
            AICommunicationUtil.promptTextCannotCall(data, devToolsInterface);
            return true;
        }

        UiRoute route = getRoute(data, devToolsInterface);
        if (route == null) {
            throw new IllegalArgumentException(
                    "Route cannot be processed, as reference here is the received data: \n" + data.toPrettyString());
        }

        ObjectNode respData = JacksonUtils.createObjectNode();
        respData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());
        respData.put(RESPONSE_TEST_GENERATION_GENERATED_FILE_KEY, route.getTestfile());
        LOGGER.info("Generating '{}' tests for route: '/{}', file '{}'", route.getFramework(),
                route.getRoute() == null ? "" : route.getRoute(), route.getTestfile());
        try {
            // Call copilot server to run AI request
            String generatedTestSource = client.generateTests(proKey, machineId, route, null);

            // Save the generated test to the file-system
            TestCodeGenerator.writeUiTest(route, generatedTestSource);

            // Call the generator to update pom and package files
            String summary = TestCodeGenerator.addTestDependencies(route,
                    getProjectFileManager().getProjectRoot().getPath(), getTestFolder(route).toString())
                            ? "Test generated and dependencies updated."
                            : "Test generation successful.";
            respData.put(RESPONSE_TEST_GENERATION_RESULT_SUMMARY_KEY, summary);

        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (Exception e) {
            LOGGER.error("Failed to generate tests", e);
            respData.put(RESPONSE_ERROR_KEY, "Failed to generate tests. See the server log for more details");
        }

        // return the response with the test generated
        devToolsInterface.send(command + "-response", respData);
        return true;
    }

    private UiRoute getRoute(JsonNode data, DevToolsInterface devToolsInterface) {
        // The list of hilla filenames coming in the "sources" data property
        List<UiRoute> sources = new ArrayList<>(getHillaSources(data));
        // The list of flow sources computed in server based on the "uiid" data
        // property
        try {
            sources.addAll(getJavaSources(data));
        } catch (IOException e) {
            LOGGER.error("Error reading requested project Flow Java files", e);
            devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, JacksonUtils.createObjectNode());
            return null;
        }

        if (sources.size() == 0) {
            LOGGER.error("Error incorrect data received from client: No sources.");
            devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, JacksonUtils.createObjectNode());
            return null;
        }

        sources.forEach(s -> s.setSource(PromptUtils.cleanJavaCode(s.getSource())));

        // The url of the current view
        String path = getPath(data);
        if (path == null) {
            LOGGER.error("Error incorrect data received from client: No route.");
            devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, JacksonUtils.createObjectNode());
            return null;
        }

        // Get the the view that matches the route. If not found, get
        // the last source in the list
        UiRoute view = sources.stream().filter(s -> s.getRoute() != null && s.getRoute().equals(path)).findFirst()
                .orElse(sources.get(sources.size() - 1));

        // Now adjust the path of the selected view with the route from ui
        view.setRoute(path);
        // We send in the prompt all known sources used in the view
        String allSources = sources.stream()
                .flatMap(s -> Stream.of(
                        "//Filename: " + getProjectFileManager()
                                .getProjectRelativeName(getProjectFileManager().getFileForClass(s.getClassName())),
                        s.getSource()))
                .collect(Collectors.joining("\n      "));
        view.setSource(allSources);

        view.setHtml(getHtml(data));
        view.setBaseUrl(getBaseUrl(data));
        TestFramework fw = "flow".equals(view.getFramework()) ? TestFramework.PLAYWRIGHT_JAVA
                : TestFramework.PLAYWRIGHT_NODE;
        Path javaTestDir = getTestFolder(view);
        view.computeTestName(javaTestDir.toFile(), fw);

        // Assure that computed test file is relative to the project root
        File testFile = new File(view.getTestfile());
        if (testFile.isAbsolute()) {
            view.setTestfile(getProjectFileManager().getProjectRelativeName(testFile));
        }

        view.setProjectRoot(getProjectFileManager().getProjectRoot().getAbsolutePath());
        return view;
    }

    private static Path getTestFolder(UiRoute view) {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        File viewFile = new File(view.getFile());
        File referenceFile = viewFile.exists() ? viewFile : projectFileManager.getFileForClass(view.getClassName());
        Optional<JavaSourcePathDetector.ModuleInfo> module = projectFileManager.findModule(referenceFile);
        Optional<Path> javaTestDir = module.flatMap(JavaSourcePathDetector.ModuleInfo::getOrGuessTestFolder);
        if (javaTestDir.isEmpty()) {
            throw new IllegalStateException("Unable to determine the test folder for the module");
        }
        return javaTestDir.get();
    }

    private String getPath(JsonNode data) {
        if (!data.has("path")) {
            return null;
        }
        return data.get("path").asString();
    }

    private String getBaseUrl(JsonNode data) {
        if (!data.has("base")) {
            return null;
        }
        return data.get("base").asString();
    }

    private String getHtml(JsonNode data) {
        if (!data.has("html")) {
            return null;
        }
        return PromptUtils.cleanHtml(data.get("html").asString());
    }

    private List<UiRoute> getHillaSources(JsonNode data) {
        Map<String, String> hillaSourceFiles = AICommunicationUtil.getHillaSourceFiles(data);

        return hillaSourceFiles.entrySet().stream().map(e -> {
            UiRoute route = new UiRoute();
            route.setFile(e.getKey());
            route.setClassName(FilenameUtils.getBaseName(e.getKey()));
            route.setSource(e.getValue());
            return route;
        }).toList();
    }

    private List<UiRoute> getJavaSources(JsonNode data) throws IOException {
        if (!data.has("uiid")) {
            return Collections.emptyList();
        }

        List<UiRoute> sources = new ArrayList<>();

        for (Map.Entry<ComponentTracker.Location, File> entry : FlowUtil
                .findActiveJavaFiles(getVaadinSession(), data.get("uiid").asInt()).entrySet()) {
            ComponentTracker.Location location = entry.getKey();
            File javaFile = entry.getValue();

            UiRoute route = new UiRoute();
            route.setFile(location.filename());
            route.setClassName(location.className());
            route.setRoute(getAnnotationValue(location.className(), Route.class));
            route.setTagName(getAnnotationValue(location.className(), Tag.class));
            route.setSource(getProjectFileManager().readFile(javaFile.getPath()));
            sources.add(route);
        }
        return sources;
    }

    public String getAnnotationValue(String className, Class<? extends Annotation> annotationClass) {
        try {
            // Load the class dynamically
            Class<?> clazz = Class.forName(className);

            // Traverse the class hierarchy
            while (clazz != null) {
                // Check if the class has the specified annotation
                if (clazz.isAnnotationPresent(annotationClass)) {
                    // Get the annotation
                    Annotation annotation = clazz.getAnnotation(annotationClass);
                    // Use reflection to get the value method of the annotation
                    Method valueMethod = annotationClass.getMethod("value");
                    // Invoke the value method to get the annotation value
                    return (String) valueMethod.invoke(annotation);
                }
                // Move to the superclass
                clazz = clazz.getSuperclass();
            }
        } catch (Exception e) {
            LOGGER.error("Error getting annotation value: {} - Class: {} - Annotation: {}", e.getMessage(), className,
                    annotationClass.getName());
        }
        return null;
    }

    private static ProKey getProKey() {
        return LocalProKey.get();
    }

    String getMachineId() {
        return MachineId.get();
    }
}
