package com.vaadin.copilot.exception.report;

import java.io.File;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.vaadin.copilot.javarewriter.ComponentAttachInfo;
import com.vaadin.copilot.javarewriter.ComponentCreateInfo;
import com.vaadin.copilot.javarewriter.ComponentInfo;

import com.github.javaparser.ast.CompilationUnit;
import tools.jackson.databind.JsonNode;

/**
 * A builder-style class used to construct an {@link ExceptionReport} instance.
 * <p>
 * This class collects various pieces of information relevant to an exception
 * report, including a title, component details, relevant data pairs, and
 * component node context. It serves as a flexible intermediary to assemble all
 * necessary parts before creating a finalized {@link ExceptionReport}.
 */

public class ExceptionReportCreator {
    private String title;
    private final List<ComponentInfo> componentInfoList = new ArrayList<>();
    private final List<ExceptionReportRelevantPairData> relevantPairs = new ArrayList<>();
    private final List<ExceptionReportRelevantComponentNode> relevantComponentNodes = new ArrayList<>();

    private final Set<String> addedFilePaths = new HashSet<>();

    /**
     * Method that adds given pair into a list.
     *
     * @param pair
     *            Pair that is relevant to the action.
     */
    public void addRelevantPair(ExceptionReportRelevantPairData pair) {
        relevantPairs.add(pair);
    }

    /**
     * Method that adds relevant nodes into a list
     *
     * @param exceptionReportRelevantComponentNode
     *            Node to be added
     */
    public void addRelevantComponentNode(ExceptionReportRelevantComponentNode exceptionReportRelevantComponentNode) {
        relevantComponentNodes.add(exceptionReportRelevantComponentNode);
    }

    /**
     * A utility method to add relevant component node for the report. Component
     * node id and ui Id are obtained from component JSON with {@code uiId} and
     * {@code nodeId} keys
     *
     * @param relevance
     *            a string that represents the component with the action.
     * @param component
     *            JSON object that should contain {@code nodeId} and {@code uiId}
     */
    public void addRelevantComponentNode(String relevance, JsonNode component) {
        ExceptionReportRelevantComponentNode exceptionReportRelevantComponentNode = new ExceptionReportRelevantComponentNode();
        exceptionReportRelevantComponentNode.setNodeId(component.get("nodeId").asInt());
        exceptionReportRelevantComponentNode.setUiId(component.get("uiId").asInt());
        exceptionReportRelevantComponentNode.setRelevance(relevance);
        addRelevantComponentNode(exceptionReportRelevantComponentNode);
    }

    /**
     * Creates exception report from the information that is provided. Eliminates
     * irrelevant info from the source that is provided by {@link ComponentInfo} and
     * generates a simplified class source code for the report.
     *
     * @return Exception report
     */
    public ExceptionReport create() {
        ExceptionReport report = new ExceptionReport();
        report.setTitle(title);
        report.setRelevantPairs(relevantPairs);
        report.setNodes(relevantComponentNodes);
        for (ComponentInfo componentInfo : componentInfoList) {
            List<ExceptionReportFileItem> relevantContent = findRelevantContent(componentInfo);
            relevantContent.forEach(report::addItem);
        }
        return report;
    }

    private List<ExceptionReportFileItem> findRelevantContent(ComponentInfo componentInfo) {
        List<ExceptionReportFileItem> fileItems = new ArrayList<>();
        Optional<ComponentCreateInfo> componentCreateInfoOptional = componentInfo.componentCreateInfoOptional();
        if (componentCreateInfoOptional.isPresent()) {
            ComponentCreateInfo componentCreateInfo = componentCreateInfoOptional.get();
            File sourceFile = componentCreateInfo.getJavaSource().getFile();
            computeIfFileAbsent(sourceFile, () -> {
                CompilationUnit compilationUnit = componentCreateInfo.getJavaSource().getCompilationUnit();
                compilationUnit.getPackageDeclaration().ifPresent(compilationUnit::remove);
                removeIrrelevantImports(compilationUnit);

                fileItems.add(new ExceptionReportFileItem(sourceFile, compilationUnit.toString()));
                addedFilePaths.add(sourceFile.getAbsolutePath());
            });

        }
        Optional<ComponentAttachInfo> componentAttachInfo = componentInfo.componentAttachInfoOptional();
        if (componentAttachInfo.isPresent() && !componentInfo.createAndAttachLocationsAreInSameFile()) {
            File sourceFile = componentAttachInfo.get().getJavaSource().getFile();
            computeIfFileAbsent(sourceFile, () -> {
                CompilationUnit compilationUnit = componentAttachInfo.get().getJavaSource().getCompilationUnit();
                compilationUnit.getPackageDeclaration().ifPresent(compilationUnit::remove);
                removeIrrelevantImports(compilationUnit);
                fileItems.add(new ExceptionReportFileItem(sourceFile, compilationUnit.toString()));
            });
        }
        return fileItems;
    }

    private void computeIfFileAbsent(File file, Runnable runnable) {
        if (!addedFilePaths.contains(file.getAbsolutePath())) {
            runnable.run();
        }
        addedFilePaths.add(file.getAbsolutePath());
    }

    private void removeIrrelevantImports(CompilationUnit compilationUnit) {
        compilationUnit.getImports().removeIf(importDeclaration -> {
            if (importDeclaration.getName().toString().startsWith("com.vaadin")
                    || importDeclaration.getName().toString().startsWith("org.vaadin")) {
                return true;
            }
            return false;
        });
    }

    /**
     * Adds single {@link ComponentInfo} to the list of component information that
     * are used to extract source code
     *
     * @param componentInfo
     *            the component information to be added
     */
    public void addComponentInfo(ComponentInfo componentInfo) {
        componentInfoList.add(componentInfo);
    }

    /**
     * Adds multiple {@link ComponentInfo} objects to the list of component
     * information that are used to extract source code
     *
     * @param componentInfos
     *            a list of component information objects to be added
     */
    public void addComponentInfo(List<ComponentInfo> componentInfos) {
        componentInfoList.addAll(componentInfos);
    }

    public String getTitle() {
        return title;
    }

    /**
     * Sets title of the report. This will be used in GH as issue title.
     *
     * @param title
     *            title that represents what exception is about
     */
    public void setTitle(String title) {
        this.title = title;
    }
}
