package com.vaadin.copilot.ai;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.TimeoutException;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.communication.CopilotServerResponseCode;
import com.vaadin.copilot.userinfo.AIUsageDisabledException;
import com.vaadin.flow.internal.JacksonUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ObjectNode;

public class AICommunicationUtil {

    /**
     * Method handling the case when the AI cannot be called, because the proKey or
     * machineId is missing.
     *
     * @param data
     *            - the data object, we are getting from the client/browser
     * @param devToolsInterface
     *            - the devtools interface we use to communicate with the
     *            client/browser
     */
    public static void promptTextCannotCall(JsonNode data, DevToolsInterface devToolsInterface) {
        getLogger().error("Cannot call AI without a proKey or machineId present.");
        ObjectNode responseData = JacksonUtils.createObjectNode();
        if (data.has(CopilotCommand.KEY_REQ_ID)) {
            responseData.put(CopilotCommand.KEY_REQ_ID, data.get(CopilotCommand.KEY_REQ_ID).asString());
        }
        responseData.put("error", true);
        responseData.put("code", CopilotServerResponseCode.COPILOT_AUTHORISATION_FAILED.getCode());
        devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
    }

    public static Map<String, String> getHillaSourceFiles(JsonNode data) {
        Map<String, String> sources = new LinkedHashMap<>();

        // Hilla source files;
        if (data.hasNonNull("sources")) {
            for (JsonNode source : data.withArray("sources")) {
                getLogger().debug("Hilla file: {}", source.get("filename").asString());
                sources.put(source.get("filename").asString(), source.get("projectFile").asString());
            }
        }
        return sources;
    }

    public static void handlingExceptionsAndNotifyDevtoolsInterface(JsonNode dataJson,
            DevToolsInterface devToolsInterface, Throwable e) {
        ObjectNode responseData = JacksonUtils.createObjectNode();
        if (dataJson.has(CopilotCommand.KEY_REQ_ID)) {
            responseData.put(CopilotCommand.KEY_REQ_ID, dataJson.get(CopilotCommand.KEY_REQ_ID).asString());
        }
        if (e.getCause() instanceof TimeoutException) {
            responseData.put("code", AIConstants.COPILOT_CLIENT_TIMEOUT_ERROR);
        } else if (e instanceof AIUsageDisabledException) {
            responseData.put("code", CopilotServerResponseCode.AI_DISABLED.getCode());
        } else {
            responseData.put("code", AIConstants.COPILOT_INTERNAL_ERROR);
        }
        responseData.put("message", "Error querying copilot server");
        getLogger().error("Error querying copilot server", e);
        devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(AICommunicationUtil.class);
    }
}
