package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.exception.RouteDuplicatedException;
import com.vaadin.copilot.exception.report.ExceptionReportCreator;
import com.vaadin.copilot.exception.report.ExceptionReportRelevantPairData;
import com.vaadin.copilot.javarewriter.JavaSource;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasElement;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.router.RouterLayout;
import com.vaadin.flow.router.internal.AbstractRouteRegistry;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.SessionRouteRegistry;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.server.startup.ApplicationRouteRegistry;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ArrayNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * Provides server side route information to the client.
 */
public class RouteHandler extends CopilotCommand {
    private static final String NODE_ID_JSON_KEY = "nodeId";
    private static final String UI_ID_JSON_KEY = "uiId";
    private static final String QUALIFIED_CLASS_NAME_JSON_KEY = "qualifiedClassName";
    private static final String FILE_PATH_JSON_KEY = "filePath";

    private final RouteCreator routeCreator;
    private final NewRouteTemplateHandler templateFinder;
    private final SourceSyncChecker sourceSyncChecker;

    public RouteHandler(SourceSyncChecker sourceSyncChecker) {
        this.routeCreator = new RouteCreator(getVaadinSession());
        this.templateFinder = new NewRouteTemplateHandler();
        this.sourceSyncChecker = sourceSyncChecker;
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-routes")) {
            ObjectNode returnData = JacksonUtils.createObjectNode();
            returnData.put("reqId", data.get("reqId").asString());
            boolean securityEnabled = isViewSecurityEnabled();
            returnData.put("securityEnabled", securityEnabled);
            try {
                ArrayNode serverRoutes = RouteHandler.getServerRoutes(getVaadinSession()).stream().filter(
                        routeData -> !sourceSyncChecker.isViewDeleted(routeData.getNavigationTarget().getName()))
                        .map(routeData -> {
                            ObjectNode route = JacksonUtils.createObjectNode();
                            Class<? extends Component> target = routeData.getNavigationTarget();
                            route.put("path", addInitialSlash(routeData.getTemplate()));
                            route.put("navigationTarget", target.getName());
                            if (securityEnabled) {
                                try {
                                    AccessRequirement req = AccessRequirementUtil.getAccessRequirement(target, null);
                                    route.set("accessRequirement", JacksonUtils.beanToJson(req));
                                } catch (Exception e) {
                                    getLogger().error("Unable to determine access requirement", e);
                                }
                            }
                            route.put(QUALIFIED_CLASS_NAME_JSON_KEY, routeData.getNavigationTarget().getName());
                            route.put("filename", getProjectFileManager()
                                    .getFileForClass(routeData.getNavigationTarget()).getAbsolutePath());
                            return route;
                        }).collect(JacksonUtils.asArray());
                returnData.set("routes", serverRoutes);
                devToolsInterface.send("server-routes", returnData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Error getting routes", e);
            }
            return true;

        } else if (command.equals("add-route")) {
            ObjectNode returnData = JacksonUtils.createObjectNode();
            returnData.put("reqId", data.get("reqId").asString());

            String title = data.get("title").asString();
            String route = data.get("route").asString();
            String className = data.get("className").asString();
            if (route.startsWith("/")) {
                route = route.substring(1);
            }
            String template = data.get("template").asString();
            String templateData = data.has("templateData") ? data.get("templateData").asString() : null;

            String framework = data.get("framework").asString();
            AccessRequirement.Type accessControl = AccessRequirement.Type.valueOf(data.get("accesscontrol").asString());
            AccessRequirement accessRequirement;
            if (accessControl == AccessRequirement.Type.ROLES_ALLOWED) {
                accessRequirement = new AccessRequirement(accessControl,
                        JacksonUtils.stream(data.withArray("roles")).map(JsonNode::asText).toArray(String[]::new));
            } else {
                accessRequirement = new AccessRequirement(accessControl);
            }
            Path folderPath = RouteCreator.getNewFilePath(framework, className);
            boolean addToMenu = data.has("addToMenu") ? data.get("addToMenu").asBoolean() : false;

            try {
                if ("hilla".equals(framework)) {
                    routeCreator.createHillaView(title, route, folderPath, accessRequirement, template, templateData,
                            addToMenu);
                } else if ("flow".equals(framework)) {
                    Class<? extends HasElement> layoutClass = null;
                    if (data.has("uiId")) {
                        Optional<RouteData> routeData = FlowUtil.getRouteData(getVaadinSession(),
                                data.get("uiId").asInt());
                        if (routeData.isPresent()) {
                            layoutClass = routeData.get().getParentLayout();
                        }

                    }

                    routeCreator.createFlowView(title, route, folderPath, accessRequirement, layoutClass, template,
                            templateData, addToMenu);
                } else {
                    throw new IllegalArgumentException("Unknown framework: " + framework);
                }
                devToolsInterface.send(command + "-response", returnData);
            } catch (RouteDuplicatedException e) {
                returnData.put("errorDuplicatedRoute", true);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Route already exists", e);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Error adding route", e);
            }

            return true;
        } else if (command.equals("get-new-route-templates")) {
            ObjectNode response = JacksonUtils.createObjectNode();
            try {
                response.put("reqId", data.get("reqId").asString());
                String framework = data.get("framework").asString();
                List<String> templateNames = templateFinder.getTemplateNames(framework);
                ArrayNode array = JacksonUtils.listToJson(templateNames);
                response.set("templates", array);

                devToolsInterface.send(command + "-response", response);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, response, "Error loading template views", e);
            }

            return true;
        } else if (command.equals("update-route-access")) {
            ObjectNode response = JacksonUtils.createObjectNode();
            response.put("reqId", data.get("reqId").asString());

            String path = data.get("path").asString().replaceFirst("^/", "");
            AccessRequirement accessRequirement = getAccessRequirement(data);
            List<RouteData> serverRoutes = RouteHandler.getServerRoutes(getVaadinSession());
            RouteData route = serverRoutes.stream().filter(routeData -> routeData.getTemplate().equals(path)).findAny()
                    .orElseThrow(() -> new IllegalArgumentException("Route not found: " + path));

            try {
                ProjectFileManager projectFileManager = ProjectFileManager.get();
                File routeFile = projectFileManager.getFileForClass(route.getNavigationTarget());
                updateRouteAccessControl(routeFile, route.getTemplate(), accessRequirement);
                devToolsInterface.send(command + "-response", response);
                return true;
            } catch (IOException e) {
                throw new CopilotException("Error updating route access control", e);
            }
        } else if (command.equals("get-new-route-package-name")) {
            String framework = data.get("framework").asString();
            String packageNameRelative = RouteCreator.getNewClassPackageNameRelative(framework, getVaadinSession());

            ObjectNode response = JacksonUtils.createObjectNode();
            response.put("reqId", data.get("reqId").asString());
            response.put("path", packageNameRelative);
            devToolsInterface.send(command + "-response", response);
            return true;
        } else if (command.equals("delete-route")) {
            ObjectNode returnData = JacksonUtils.createObjectNode();
            returnData.put("reqId", data.get("reqId").asString());
            ExceptionReportCreator exceptionReportCreator = new ExceptionReportCreator();
            exceptionReportCreator.setTitle("Unable to delete route");
            exceptionReportCreator
                    .addRelevantPair(new ExceptionReportRelevantPairData("Request Data", data.toString()));
            try {
                deleteRouteHandler(data, devToolsInterface, command, returnData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Error deleting file", e,
                        exceptionReportCreator);
            }
            return true;
        }
        return false;

    }

    private void deleteRouteHandler(JsonNode data, DevToolsInterface devToolsInterface, String command,
            ObjectNode returnData) throws IOException {
        File fileToDelete;
        if (data.hasNonNull(NODE_ID_JSON_KEY) && data.hasNonNull(UI_ID_JSON_KEY)) {

            Optional<Component> component;
            try {
                getVaadinSession().lock();
                Element element = getVaadinSession().findElement(data.get(UI_ID_JSON_KEY).asInt(),
                        data.get(NODE_ID_JSON_KEY).asInt());
                component = element.getComponent();
            } finally {
                getVaadinSession().unlock();
            }

            if (component.isEmpty()) {
                throw new CopilotException(
                        "Unable to find component for node id " + data.get(NODE_ID_JSON_KEY).asInt());
            }
            fileToDelete = ProjectFileManager.get().getFileForClass(component.get().getClass());
            ProjectFileManager.get().deleteFile(fileToDelete);
            sourceSyncChecker.addDeletedView(component.get());
        } else if (data.hasNonNull(QUALIFIED_CLASS_NAME_JSON_KEY)) {
            fileToDelete = ProjectFileManager.get().getFileForClass(data.get(QUALIFIED_CLASS_NAME_JSON_KEY).asString());
            if (fileToDelete == null || !fileToDelete.exists()) {
                throw new IllegalArgumentException("Unable to find file for qualifiedClassName "
                        + data.get(QUALIFIED_CLASS_NAME_JSON_KEY).asString());
            }

            ProjectFileManager.get().deleteFile(fileToDelete);
            sourceSyncChecker.addDeletedView(data.get(QUALIFIED_CLASS_NAME_JSON_KEY).asString());
        } else if (data.hasNonNull(FILE_PATH_JSON_KEY)) {
            fileToDelete = new File(data.get(FILE_PATH_JSON_KEY).asString());
            if (!fileToDelete.exists()) {
                throw new IllegalArgumentException(
                        "Unable to find file for path " + data.get(FILE_PATH_JSON_KEY).asString());
            }
            ProjectFileManager.get().deleteFile(fileToDelete);
        }
        devToolsInterface.send(command + "-response", returnData);
    }

    static Optional<AccessRequirement.Type> getRouteAccessControl(File routeFile, String routePath) throws IOException {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        if (!routeFile.exists()) {
            throw new IllegalArgumentException("The source file for the route is not in this project: " + routeFile);
        }
        String source = projectFileManager.readFile(routeFile);
        JavaSource javaSource = new JavaSource(routeFile, source, true, true);
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        ClassOrInterfaceDeclaration routeClass = AccessRequirementUtil.findRouteClass(compilationUnit, routePath);
        return AccessRequirementUtil.getAccessAnnotation(routeClass);
    }

    static void updateRouteAccessControl(File routeFile, String routePath, AccessRequirement accessRequirement)
            throws IOException {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        if (!routeFile.exists()) {
            throw new IllegalArgumentException("The source file for the route is not in this project: " + routeFile);
        }
        String source = projectFileManager.readFile(routeFile);
        JavaSource javaSource = new JavaSource(routeFile, source, true, true);
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        ClassOrInterfaceDeclaration routeClass = AccessRequirementUtil.findRouteClass(compilationUnit, routePath);
        AccessRequirementUtil.setAccessAnnotation(routeClass, accessRequirement);
        projectFileManager.writeFile(routeFile, "Update access control", javaSource.getResult());
    }

    static Optional<AccessRequirement.Type> getLayoutAccessControl(File layoutFile) throws IOException {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        if (!layoutFile.exists()) {
            throw new IllegalArgumentException("The source file for the layout is not in this project: " + layoutFile);
        }
        String source = projectFileManager.readFile(layoutFile);
        JavaSource javaSource = new JavaSource(layoutFile, source, true, true);
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        ClassOrInterfaceDeclaration layoutClass = AccessRequirementUtil.findLayoutClass(compilationUnit);
        return AccessRequirementUtil.getAccessAnnotation(layoutClass);
    }

    static void updateLayoutAccessControl(File layoutFile, AccessRequirement accessRequirement) throws IOException {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        if (!layoutFile.exists()) {
            throw new IllegalArgumentException("The source file for the layout is not in this project: " + layoutFile);
        }
        String source = projectFileManager.readFile(layoutFile);
        JavaSource javaSource = new JavaSource(layoutFile, source, true, true);
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        ClassOrInterfaceDeclaration layoutClass = AccessRequirementUtil.findLayoutClass(compilationUnit);
        AccessRequirementUtil.setAccessAnnotation(layoutClass, accessRequirement);
        projectFileManager.writeFile(layoutFile, "Update access control", javaSource.getResult());
    }

    private static AccessRequirement getAccessRequirement(JsonNode data) {
        AccessRequirement.Type accessControl = AccessRequirement.Type.valueOf(data.get("accesscontrol").asString());
        if (accessControl == AccessRequirement.Type.ROLES_ALLOWED) {
            return new AccessRequirement(accessControl,
                    JacksonUtils.stream(data.withArray("roles")).map(JsonNode::asText).toArray(String[]::new));
        } else {
            return new AccessRequirement(accessControl);
        }

    }

    private boolean isViewSecurityEnabled() {
        return SpringBridge.isSpringAvailable(getVaadinContext())
                && SpringBridge.isViewSecurityEnabled(getVaadinContext());
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(RouteHandler.class);
    }

    private String addInitialSlash(String template) {
        if (!template.startsWith("/")) {
            return "/" + template;
        }
        return template;
    }

    public static List<RouteData> getServerRoutes(VaadinSession vaadinSession) {
        try {
            vaadinSession.lock();
            return vaadinSession.getService().getRouter().getRegistry().getRegisteredRoutes();
        } catch (Exception e) {
            getLogger().error("Error getting server routes", e);
        } finally {
            vaadinSession.unlock();
        }
        return new ArrayList<>();
    }

    public static Set<Class<?>> getServerAutoLayouts(VaadinSession vaadinSession) {
        try {
            vaadinSession.lock();
            RouteRegistry sessionRegistry = SessionRouteRegistry.getSessionRegistry(vaadinSession);
            RouteRegistry appRegistry = ApplicationRouteRegistry.getInstance(vaadinSession.getService().getContext());
            Set<Class<?>> autoLayouts = new HashSet<>();
            autoLayouts.addAll(getServerAutoLayouts(sessionRegistry));
            autoLayouts.addAll(getServerAutoLayouts(appRegistry));
            return autoLayouts;
        } catch (Exception e) {
            getLogger().error("Error getting server auto layouts", e);
        } finally {
            vaadinSession.unlock();
        }
        return Set.of();
    }

    private static Collection<Class<?>> getServerAutoLayouts(RouteRegistry registry)
            throws NoSuchMethodException, InvocationTargetException, IllegalAccessException {
        if (registry instanceof AbstractRouteRegistry abstractRouteRegistry) {
            Method getLayoutsMethod = AbstractRouteRegistry.class.getDeclaredMethod("getLayouts");
            getLayoutsMethod.setAccessible(true);
            return (Collection<Class<?>>) getLayoutsMethod.invoke(abstractRouteRegistry);
        }
        return List.of();
    }

    /**
     * Returns the unique layout classes that are used in given route list
     *
     * @param vaadinSession
     *            Vaadin session to access router
     * @param routeDataList
     *            List of routes
     * @return the unique classes of Layouts
     */
    public static Set<Class<? extends RouterLayout>> getLayoutClasses(VaadinSession vaadinSession,
            List<RouteData> routeDataList) {
        Set<Class<? extends RouterLayout>> layoutClasses = new HashSet<>();
        try {
            vaadinSession.lock();
            RouteRegistry registry = vaadinSession.getService().getRouter().getRegistry();
            for (RouteData routeData : routeDataList) {
                Class<? extends RouterLayout> template = registry.getLayout(routeData.getTemplate());
                if (template != null) {
                    layoutClasses.add(template);
                }
                if (routeData.getParentLayouts() != null) {
                    layoutClasses.addAll(routeData.getParentLayouts());
                }

            }
        } catch (Exception e) {
            getLogger().error("Error getting server routes", e);
        } finally {
            vaadinSession.unlock();
        }
        return layoutClasses;
    }
}
