package com.vaadin.copilot;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.stream.Stream;

import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasElement;
import com.vaadin.flow.component.UI;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.dom.Element;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.Command;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.VaadinSession;

public class FlowUtil {

    private FlowUtil() {
        // Utils only
    }

    /**
     * Finds all Java files that are used to create the current component tree for
     * the given ui.
     *
     * @param uiId
     *            the id for the UI, inside the given session
     * @return a map of the locations and the corresponding Java files
     */
    public static Map<ComponentTracker.Location, File> findActiveJavaFiles(VaadinSession session, int uiId) {
        try {
            session.lock();
            UI ui = session.getUIById(uiId);
            Map<Component, ComponentTypeAndSourceLocation> allComponents = findAllComponentSourceLocations(session, ui);
            Set<String> processedClasses = new HashSet<>();
            LinkedHashMap<ComponentTracker.Location, File> locations = new LinkedHashMap<>();
            for (ComponentTypeAndSourceLocation componentTypeAndSourceLocation : allComponents.values()) {
                for (ComponentTracker.Location location : getLocations(componentTypeAndSourceLocation)) {
                    if (processedClasses.contains(location.className())) {
                        continue;
                    }

                    File javaFile = ProjectFileManager.get().getFileForClass(location.className());
                    // The component locator tracks the Java class and guesses the
                    // Java file, so it's possible the Java file is not in the
                    // project
                    if (javaFile.exists()) {
                        processedClasses.add(location.className());
                        locations.put(location, javaFile);
                    }
                }
            }
            return locations;
        } finally {
            session.unlock();
        }
    }

    private static List<ComponentTracker.Location> getLocations(
            ComponentTypeAndSourceLocation componentTypeAndSourceLocation) {
        return Stream.concat(componentTypeAndSourceLocation.createLocationInProject().stream(),
                componentTypeAndSourceLocation.attachLocationInProject().stream()).toList();
    }

    public static Map<Component, ComponentTypeAndSourceLocation> findAllComponents(VaadinSession session, int uiId) {
        try {
            session.lock();
            UI ui = session.getUIById(uiId);
            return findAllComponentSourceLocations(session, ui);
        } finally {
            session.unlock();
        }
    }

    private static Map<Component, ComponentTypeAndSourceLocation> findAllComponentSourceLocations(VaadinSession session,
            UI ui) {
        List<Component> componentList = new ArrayList<>();
        addComponents(ui, componentList);

        Map<Component, ComponentTypeAndSourceLocation> locations = new HashMap<>();
        ComponentSourceFinder sourceFinder = new ComponentSourceFinder(session);

        for (Component component : componentList) {
            ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component,
                    false);
            locations.put(component, typeAndSourceLocation);
        }
        return locations;
    }

    /**
     * Returns the view chain for the given UI.
     *
     * @param session
     *            the vaadin session
     * @param uiId
     *            the id for the UI, inside the given session
     * @return a list of the view classes currently being shown in the UI
     */
    public static List<Class<? extends HasElement>> getViewChain(VaadinSession session, int uiId) {
        try {
            session.lock();
            UI ui = session.getUIById(uiId);
            if (ui == null) {
                return Collections.emptyList();
            }
            return (List) ui.getActiveRouterTargetsChain().stream().map(Object::getClass).toList();
        } finally {
            session.unlock();
        }
    }

    /**
     * The method traverses up the component hierarchy, starting from the immediate
     * parent of the given component, and applies the provided {@link Predicate} to
     * each ancestor.
     *
     * <p>
     * The session is locked during traversal
     * </p>
     *
     * @param session
     *            the {@link VaadinSession} to which the component belongs; must not
     *            be {@code null}.
     * @param component
     *            the {@link Component} whose ancestors are to be tested; must not
     *            be {@code null}.
     * @param testPredicate
     *            the {@link Predicate} used to test each ancestor component; must
     *            not be {@code null}.
     * @return {@code true} if any ancestor component satisfies the given predicate,
     *         {@code false} otherwise.
     */
    public static boolean testAncestors(VaadinSession session, Component component,
            Predicate<Component> testPredicate) {
        try {
            session.lock();
            Component current = component.getParent().orElse(null);
            while (current != null) {
                boolean test = testPredicate.test(current);
                if (test) {
                    return true;
                }
                current = current.getParent().orElse(null);
            }
        } finally {
            session.unlock();
        }
        return false;
    }

    private static void addComponents(Component component, List<Component> componentList) {
        componentList.add(component);
        component.getChildren().forEach(c -> addComponents(c, componentList));
    }

    public static Integer getNodeId(Component component) {
        return component.getElement().getNode().getId();
    }

    public static Optional<RouteData> getRouteData(VaadinSession session, int uiId) {
        try {
            session.lock();
            List<Class<? extends HasElement>> viewChain = getViewChain(session, uiId);
            if (viewChain.isEmpty()) {
                return Optional.empty();
            }

            Class<? extends HasElement> routeClass = viewChain.get(0);
            RouteRegistry registry = session.getService().getRouter().getRegistry();
            List<RouteData> data = registry.getRegisteredRoutes().stream()
                    .filter(routeData -> routeData.getNavigationTarget() == routeClass).toList();
            if (data.isEmpty()) {
                return Optional.empty();
            }
            return Optional.of(data.get(0));
        } finally {
            session.unlock();
        }
    }

    /**
     * Finds the Component instance with given <code>nodeId</code> in UI with
     * <code>uiId</code> via {@link VaadinSession#accessSynchronously(Command)}
     *
     * @param vaadinSession
     *            Vaadin session to access UI elements
     * @param nodeId
     *            Node id
     * @param uiId
     *            ui id
     * @return Component or {@link Optional#empty()}
     */
    public static Optional<Component> findComponentByNodeIdAndUiId(VaadinSession vaadinSession, int nodeId, int uiId) {
        AtomicReference<Component> componentAtomicReference = new AtomicReference<>(null);
        vaadinSession.accessSynchronously(() -> {
            Element element = vaadinSession.findElement(uiId, nodeId);
            Optional<Component> componentOptional = element.getComponent();
            componentOptional.ifPresent(componentAtomicReference::set);
        });
        if (componentAtomicReference.get() == null) {
            return Optional.empty();
        }
        return Optional.of(componentAtomicReference.get());
    }

}
