package com.vaadin.copilot.testbenchgenerator;

import java.util.ArrayList;
import java.util.List;

import com.vaadin.copilot.testbenchgenerator.assertion.AssertionScenarioItem;
import com.vaadin.copilot.testbenchgenerator.events.PageChangeEvent;
import com.vaadin.copilot.testbenchgenerator.events.TargetEvent;
import com.vaadin.flow.internal.JacksonUtils;

import tools.jackson.core.JacksonException;
import tools.jackson.core.type.TypeReference;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.ObjectMapper;
import tools.jackson.databind.node.ArrayNode;

/**
 * Represents the scenario generated by the test bench record plugin. It
 * contains all the items, imports etc... basically everything needed to create
 * a test
 */
public class TestBenchScenario {
    private List<TestBenchScenarioItem> items = new ArrayList<>();
    private List<String> imports = new ArrayList<>();
    private String qualifiedClassName;
    private String methodName;

    public List<TestBenchScenarioItem> getItems() {
        return items;
    }

    public List<String> getImports() {
        return imports;
    }

    public String getQualifiedClassName() {
        return qualifiedClassName;
    }

    public String getMethodName() {
        return methodName;
    }

    /**
     * Converts the given JSON to Scenario.
     *
     * @param json
     *            JSON sent from the UI
     * @return the scenario object
     * @throws IllegalArgumentException
     *             when JSON is not formatted well, or a required info is missing
     */
    public static TestBenchScenario from(JsonNode json) {
        ObjectMapper mapper = JacksonUtils.getMapper();
        TestBenchScenario request = new TestBenchScenario();
        if (json.has("imports")) {
            try {
                request.imports = mapper.readValue(json.get("imports").toString(), new TypeReference<>() {
                });
            } catch (Exception e) {
                throw new IllegalArgumentException("Unable to parse imports JSON", e);
            }
        }
        if (json.hasNonNull("qualifiedClassName")) {
            request.qualifiedClassName = json.get("qualifiedClassName").asString();
        }
        if (json.hasNonNull("methodName")) {
            request.methodName = json.get("methodName").asString();
        }
        if (!json.hasNonNull("items")) {
            throw new IllegalArgumentException("Unable to parse scenarios JSON");
        }
        ArrayNode items = (ArrayNode) json.get("items");
        items.forEach(scenarioItemJson -> {
            TestBenchScenarioItemType type = TestBenchScenarioItemType
                    .fromString(scenarioItemJson.get("type").asString());
            if (type == null) {
                throw new IllegalArgumentException("Unable to parse Scenario without type");
            }
            if (type == TestBenchScenarioItemType.targetEvent) {
                try {
                    TargetEvent targetEvent = mapper.readValue(scenarioItemJson.toString(), TargetEvent.class);
                    request.items.add(targetEvent);
                } catch (JacksonException e) {
                    throw new IllegalArgumentException("Unable to parse target event", e);
                }
            } else if (type == TestBenchScenarioItemType.pageChangeEvent) {
                try {
                    PageChangeEvent pageChangeEvent = mapper.readValue(scenarioItemJson.toString(),
                            PageChangeEvent.class);
                    request.items.add(pageChangeEvent);
                } catch (JacksonException e) {
                    throw new IllegalArgumentException("Unable to parse page change event", e);
                }
            } else if (type == TestBenchScenarioItemType.assertion) {
                String assertionTypeStr = scenarioItemJson.get("assertionType").asString();
                Class<? extends TestBenchScenarioItem> scenarioItemClass = AssertionScenarioItem
                        .getScenarioClass(assertionTypeStr);
                try {
                    request.items.add(mapper.readValue(scenarioItemJson.toString(), scenarioItemClass));
                } catch (JacksonException e) {
                    throw new IllegalArgumentException("Unable to parse page change event", e);
                }
            }
        });

        return request;
    }
}
