package com.vaadin.copilot.ai;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.ServerInfo;
import com.vaadin.copilot.ComponentSourceFinder;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.CopilotJacksonUtils;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.CopilotVersion;
import com.vaadin.copilot.FlowUtil;
import com.vaadin.copilot.MachineConfiguration;
import com.vaadin.copilot.Util;
import com.vaadin.copilot.communication.CopilotServerRequest;
import com.vaadin.copilot.communication.CopilotServerResponse;
import com.vaadin.copilot.communication.CopilotServerResponseCode;
import com.vaadin.copilot.communication.StreamResponse;
import com.vaadin.copilot.communication.StreamResponseEnum;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.ProKey;

import io.netty.buffer.ByteBufAllocator;
import io.netty.handler.codec.http.HttpMethod;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import reactor.core.Disposable;
import reactor.core.publisher.Mono;
import reactor.netty.ByteBufMono;
import reactor.netty.http.client.HttpClient;
import tools.jackson.core.JacksonException;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.json.JsonMapper;
import tools.jackson.databind.node.ArrayNode;
import tools.jackson.databind.node.ObjectNode;
import tools.jackson.databind.node.StringNode;

/**
 * Command handler for AI related operations
 */
public class AICommandHandler extends CopilotCommand {
    private static final String THE_FOLLOWING_ITEM_IS_SELECTED = "the following item is selected:";
    private final Map<String, String> serverInfoVersions;
    private ComponentSourceFinder componentSourceFinder;
    private final CopilotServerClient copilotServerClient = new CopilotServerClient();

    private static Logger getLogger() {
        return LoggerFactory.getLogger(AICommandHandler.class);
    }

    public AICommandHandler() {
        ServerInfo serverInfo = new ServerInfo();
        this.componentSourceFinder = new ComponentSourceFinder(getVaadinSession());
        serverInfoVersions = serverInfo.getVersions().stream()
                .collect(Collectors.toMap(ServerInfo.NameAndVersion::name, ServerInfo.NameAndVersion::version));
    }

    private final Map<String, Disposable> disposables = new HashMap<>();

    /**
     * Handles the message with exception handling, and if there was a caught
     * exception then it will notify the client/browser.
     *
     * @param command
     *            - the command, we are getting from the client/browser (can be
     *            prompt-text or prompt-cancel), which we handle here
     * @param data
     *            - the data object, we are getting from the client/browser
     * @param devToolsInterface
     *            - the devtools interface we use to communicate with the
     *            client/browser
     * @return - true if the message was handled by this handler, false otherwise
     */
    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        try {
            return handleMessageWithException(command, data, devToolsInterface);
        } catch (Exception e) {
            AICommunicationUtil.handlingExceptionsAndNotifyDevtoolsInterface(data, devToolsInterface, e);
            return true;
        }
    }

    /**
     * Handles the message.
     *
     * @param command
     *            - the command, we are getting from the client/browser (can be
     *            prompt-text or prompt-cancel), which we handle here
     * @param data
     *            - the data object, we are getting from the client/browser
     * @param devToolsInterface
     *            - the devtools interface we use to communicate with the
     *            client/browser
     * @return - true if the message was handled by this handler, false otherwise
     */
    public boolean handleMessageWithException(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("prompt-text")) {
            UserInfoServerClient.throwIfAIUsageDisabled();
            Map<String, String> metadata = new HashMap<>(serverInfoVersions);
            ProKey proKey = getProKey();
            String machineId = getMachineId();
            if (proKey == null && machineId == null) {
                AICommunicationUtil.promptTextCannotCall(data, devToolsInterface);
                return true;
            }

            if (proKey != null) {
                metadata.put(AIConstants.PRO_KEY_KEY, proKey.toJson());
            }
            if (machineId != null) {
                metadata.put(AIConstants.MACHINE_ID_KEY, machineId);
            }
            metadata.put(AIConstants.VERSION_KEY, CopilotVersion.getVersion());
            metadata.put(AIConstants.AI_PROVIDER_KEY, MachineConfiguration.get().getAiProvider().toString());
            String prompt = data.get("text").asString();
            // Hilla source files:
            Map<String, String> sources = AICommunicationUtil.getHillaSourceFiles(data);

            // Java source files:
            if (data.has("uiid")) {
                // handling selections:
                if (data.has("selections") && !data.withArray("selections").isEmpty()) {
                    ArrayNode componentsJson = data.withArray("selections");
                    getLogger().debug("Selections: {}", data.withArray("selections"));
                    List<ComponentTypeAndSourceLocation> selectedComponents = new ArrayList<>();
                    for (int i = 0; i < componentsJson.size(); i++) {
                        ComponentTypeAndSourceLocation root = componentSourceFinder
                                .findTypeAndSourceLocation(componentsJson.get(i), true);
                        selectedComponents.add(root);
                    }
                    getLogger().debug("Selected components: {}", selectedComponents);

                    try {
                        // Adding drill down component source
                        if (data.get("activeDrillDownFilePath") != null) {
                            String activeDrillDownFilePath = data.get("activeDrillDownFilePath").asString();
                            File activeDrillDownFile = new File(activeDrillDownFilePath);
                            if (activeDrillDownFile.exists()) {
                                sources.put(activeDrillDownFile.getAbsolutePath(),
                                        getProjectFileManager().readFile(activeDrillDownFile.getPath()));
                            }
                        }
                        List<ComponentTracker.Location> projectLocations = selectedComponents.stream()
                                .map(ComponentTypeAndSourceLocation::createLocationInProject)
                                .filter(Optional::isPresent).map(Optional::get).toList();

                        List<File> files = projectLocations.stream()
                                .map(projectLocation -> getProjectFileManager().getSourceFile(projectLocation))
                                .distinct().toList();
                        for (File javaFile : files) {
                            String source = getProjectFileManager().readFile(javaFile);
                            List<Integer> lineNumbers = projectLocations.stream()
                                    .filter(projectLocation -> javaFile
                                            .equals(getProjectFileManager().getSourceFile(projectLocation)))
                                    .map(ComponentTracker.Location::lineNumber).toList();
                            String sourceWithSelections = Util.insertLines(source, lineNumbers,
                                    "// " + THE_FOLLOWING_ITEM_IS_SELECTED);
                            sources.put(javaFile.getAbsolutePath(), sourceWithSelections);
                        }
                    } catch (IOException ee) {
                        getLogger().error("Error reading requested project Flow Java files", ee);
                        devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED,
                                JacksonUtils.createObjectNode());
                        return true;
                    }
                } else {
                    try {
                        // Adding drill down component source
                        if (data.get("activeDrillDownFilePath") != null) {
                            String activeDrillDownFilePath = data.get("activeDrillDownFilePath").asString();
                            File activeDrillDownFile = new File(activeDrillDownFilePath);
                            if (activeDrillDownFile.exists()) {
                                sources.put(activeDrillDownFile.getAbsolutePath(),
                                        getProjectFileManager().readFile(activeDrillDownFile.getPath()));
                            }
                        } else {
                            // Adding all Java source files
                            sources.putAll(getJavaSourceMap(data.get("uiid").asInt()));
                        }
                    } catch (IOException e) {
                        getLogger().error("Error reading requested project Flow Java files", e);
                        devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED,
                                JacksonUtils.createObjectNode());
                        return true;
                    }
                }
            }

            // Make all sources relative to the project, so we do not send
            // information about where the project is located
            Map<String, String> relativeSources = new HashMap<>();

            for (String filename : sources.keySet()) {
                try {
                    relativeSources.put(getProjectFileManager().makeRelative(filename), sources.get(filename));
                } catch (IOException e) {
                    ObjectNode responseData = JacksonUtils.createObjectNode();
                    if (data.has(KEY_REQ_ID)) {
                        responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());
                    }
                    responseData.put("code", AIConstants.COPILOT_INTERNAL_ERROR);
                    responseData.put("message", "Error processing the files");
                    getLogger().error("Error making file relative to project", e);
                    devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
                    return true;
                }
            }

            try {
                CopilotServerRequest req = new CopilotServerRequest(prompt, relativeSources, metadata);
                ObjectNode responseData = JacksonUtils.createObjectNode();
                responseData.put(KEY_CANCEL_REQ_ID, (data.has(KEY_REQ_ID) ? data.get(KEY_REQ_ID).asString() : ""));
                getLogger().debug("Request Registered in client: " + responseData.get(KEY_CANCEL_REQ_ID).asString());
                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_REQUEST_ID, responseData);
                queryCopilotServer(req, devToolsInterface, data);
                return true;
            } catch (Exception e) {
                AICommunicationUtil.handlingExceptionsAndNotifyDevtoolsInterface(data, devToolsInterface, e);
                return true;
            }
        } else if (command.equals("prompt-cancel")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            if (data.has(KEY_CANCEL_REQ_ID) && data.has(KEY_REQ_ID)) {
                responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());
                responseData.put(KEY_CANCEL_REQ_ID, data.get(KEY_CANCEL_REQ_ID).asString());
            } else {
                getLogger().error("Impossible to cancel without request and cancel request Ids");
                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_CANCEL_FAILED, responseData);
                return true;
            }
            getLogger().debug("Trying to cancel Disposable object with id: {} - object: {}",
                    data.get(KEY_REQ_ID).asString(), disposables.get(data.get(KEY_REQ_ID).asString()));

            if (disposables.get(data.get(KEY_CANCEL_REQ_ID).asString()) != null) {
                responseData.put("message", "Request cancelled successfully");
                Disposable disposable = disposables.get(data.get(KEY_CANCEL_REQ_ID).asString());
                disposable.dispose();
                disposables.remove(data.get(KEY_CANCEL_REQ_ID).asString());
                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_CANCEL_OK, responseData);
            } else {
                responseData.put("message", "Error cancelling the request with id: " + data.get(KEY_REQ_ID).asString());
                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_CANCEL_FAILED, responseData);
            }
            return true;
        }
        return false;
    }

    private Map<String, String> getJavaSourceMap(int uiId) throws IOException {
        Map<String, String> sources = new HashMap<>();
        for (Map.Entry<ComponentTracker.Location, File> entry : FlowUtil.findActiveJavaFiles(getVaadinSession(), uiId)
                .entrySet()) {
            ArrayList<String> javaFileNames = new ArrayList<>();
            File javaFile = entry.getValue();
            javaFileNames.add(javaFile.getName());
            sources.put(javaFile.getAbsolutePath(), getProjectFileManager().readFile(javaFile.getPath()));
            getLogger().debug("Java filenames: {}", javaFileNames);
        }
        return sources;
    }

    private void queryCopilotServer(CopilotServerRequest req, DevToolsInterface devToolsInterface, JsonNode dataJson) {

        URI queryUriStream = copilotServerClient.getQueryURI("stream");
        String json = CopilotJacksonUtils.writeValueAsString(req);
        if (Copilot.isDevelopmentMode()) {
            getLogger().debug("Querying copilot server at {} using {}", queryUriStream, json);
        }

        JsonMapper objectMapper = new JsonMapper(); // Jackson's JSON parser
        AtomicReference<StringBuilder> jsonBuffer = new AtomicReference<>(new StringBuilder());

        AtomicBoolean completedSuccessfully = new AtomicBoolean(false);

        Disposable disposable = HttpClient.create()
                .responseTimeout(Duration.of(AIConstants.CLIENT_MAX_TIMEOUT, ChronoUnit.SECONDS))
                .headers(headers -> headers.set("Content-Type", "application/json; charset=utf-8"))
                .request(HttpMethod.POST)
                .send(ByteBufMono.fromString(Mono.just(json), StandardCharsets.UTF_8, ByteBufAllocator.DEFAULT))
                .uri(queryUriStream).responseContent().asString(StandardCharsets.UTF_8)
                .doOnError(e -> AICommunicationUtil
                        .handlingExceptionsAndNotifyDevtoolsInterface(dataJson, devToolsInterface, e))
                .subscribe(chunk -> handleIncomingDataChunk(chunk, // the new raw chunk from the server
                        dataJson, devToolsInterface, objectMapper, jsonBuffer, completedSuccessfully, req.sources()),
                        error -> getLogger().error("Error: " + error), () -> {
                            // onComplete
                            if (!completedSuccessfully.get()) {
                                ObjectNode responseData = JacksonUtils.createObjectNode();
                                responseData.put(KEY_REQ_ID, dataJson.get(KEY_REQ_ID).asString());
                                getLogger().error("Stream did not complete successfully.");
                                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED,
                                        responseData);
                            } else {
                                disposables.remove(dataJson.get(KEY_REQ_ID).asString());
                                getLogger().debug("Stream completed successfully.");
                            }
                        });
        disposables.put(dataJson.get(KEY_REQ_ID).asString(), disposable);
        getLogger().debug("Disposable object created for prompt {} - object: {} - RequestId: {}", json, disposable,
                dataJson.get(KEY_REQ_ID).asString());
    }

    void handleIncomingDataChunk(String newChunk, JsonNode dataJson, DevToolsInterface devToolsInterface,
            JsonMapper objectMapper, AtomicReference<StringBuilder> jsonBuffer, AtomicBoolean completedSuccessfully,
            Map<String, String> sources) {
        // 1) Clean and accumulate the chunk
        String chunk = newChunk.trim();
        if (chunk.isEmpty()) {
            return; // no new data
        }

        // Some messages start with "data:", others have multiple "data:" lines within
        // One approach is to ensure each "data:" line is handled separately:
        // e.g. split by newline if your the server sets a newline after each `data:...`
        // or directly handle if chunk has multiple data: lines

        // We'll simply accumulate everything. We'll do a second pass to split out
        // "data:" if needed.
        jsonBuffer.get().append(chunk);

        // 2) Attempt to parse as many JSON objects as possible from the buffer
        List<String> completeJsonNodes = extractCompleteJsonMessages(jsonBuffer.get());

        for (String jsonObj : completeJsonNodes) {
            // 3) If "data:" prefix remains, remove it
            String actualJson = jsonObj.startsWith("data:") ? jsonObj.substring("data:".length()).trim()
                    : jsonObj.trim();

            // 4) Try to parse and handle
            try {
                Optional<StreamResponse> parsedOpt = tryParseJson(objectMapper, actualJson);
                if (parsedOpt.isPresent()) {
                    StreamResponse response = parsedOpt.get();
                    handleParsedStreamResponse(response, dataJson, devToolsInterface, completedSuccessfully, sources);
                }
            } catch (Exception e) {
                getLogger().error("Error handling AI response", e);

                // we send a "failed" message and not keeping the buffer
                ObjectNode responseData = JacksonUtils.createObjectNode();
                responseData.put(KEY_REQ_ID, dataJson.get(KEY_REQ_ID).asString());
                devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
            }
        }
    }

    private void handleParsedStreamResponse(StreamResponse parsedStreamResponse, JsonNode dataJson,
            DevToolsInterface devToolsInterface, AtomicBoolean completedSuccessfully, Map<String, String> sources)
            throws IOException {
        ObjectNode responseData = JacksonUtils.createObjectNode();
        responseData.put("status", parsedStreamResponse.status().getMessage());
        getLogger().debug("Parsed JSON: " + parsedStreamResponse);
        devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_STATUS, responseData);

        completedSuccessfully.set(true);

        if (dataJson.has(KEY_REQ_ID)) {
            responseData.put(KEY_REQ_ID, dataJson.get(KEY_REQ_ID).asString());
        }
        if (parsedStreamResponse.code() < 0) {
            responseData.put("code", parsedStreamResponse.code());
            responseData.put("message", parsedStreamResponse.message());
            devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_FAILED, responseData);
        } else if (parsedStreamResponse.status() == StreamResponseEnum.POST_PROCESS) {
            responseData.put("message", parsedStreamResponse.message());
            responseData.set("changes", parsedStreamResponse.changes().keySet().stream().map(StringNode::valueOf)
                    .collect(JacksonUtils.asArray()));
            getLogger().debug("PostProcess finished");
            devToolsInterface.send(Copilot.PREFIX + AIConstants.MESSAGE_PROMPT_OK, responseData);

            CopilotServerResponse response = new CopilotServerResponse(CopilotServerResponseCode.HILLA_REACT.getCode(),
                    parsedStreamResponse.status().getMessage(), parsedStreamResponse.changes());
            handleQueryResponse(response, sources);
        }
    }

    private List<String> extractCompleteJsonMessages(StringBuilder buffer) {
        List<String> extracted = new ArrayList<>();
        int startIndex = 0;
        while (true) {
            // naive approach:
            // find the first '{' after any "data:" prefix, then find the matching '}'
            // with naive brace counting.
            // (Or just look for next "data:" if we know each data is single-line)

            // If we're certain each JSON is prefixed with "data:", we can locate it:
            int dataIndex = buffer.indexOf("data:", startIndex);
            if (dataIndex == -1) {
                // no more "data:" prefixes found
                break;
            }

            // from dataIndex, find the next chunk that forms valid JSON
            // e.g. naive brace counting
            int openingBrace = buffer.indexOf("{", dataIndex);
            if (openingBrace == -1) {
                // incomplete JSON
                break;
            }

            // Now we do a naive brace match from openingBrace forward
            int braceCount = 0;
            int i = openingBrace;
            int endOfJson = -1;
            for (; i < buffer.length(); i++) {
                char c = buffer.charAt(i);
                if (c == '{') {
                    braceCount++;
                } else if (c == '}') {
                    braceCount--;
                    if (braceCount == 0) {
                        endOfJson = i; // matched the final brace
                        break;
                    }
                }
            }

            if (endOfJson == -1) {
                // We didn't find a matching closing brace => incomplete
                break;
            }

            // We found a complete JSON from dataIndex up to endOfJson
            String oneJsonMsg = buffer.substring(dataIndex, endOfJson + 1);
            extracted.add(oneJsonMsg);

            // Move startIndex forward
            startIndex = endOfJson + 1;
        }

        // Remove the extracted pieces from the buffer
        if (!extracted.isEmpty()) {
            buffer.delete(0, startIndex);
        }

        return extracted;
    }

    /**
     * Utility method to attempt parsing JSON from the given string, returns - null
     * if parsing fails or JSON is incomplete
     *
     * @param objectMapper
     *            - Jackson's ObjectMapper
     * @param data
     *            - JSON string
     * @return - Optional of parsed StreamResponse, or empty if parsing failed
     */
    private Optional<StreamResponse> tryParseJson(JsonMapper objectMapper, String data) {
        try {
            // ObjectMapper attempts to parse the data, may throw if incomplete
            // or invalid
            return Optional.ofNullable(objectMapper.readValue(data, StreamResponse.class));
        } catch (JacksonException e) {
            return Optional.empty(); // Return empty to indicate parsing was
            // unsuccessful or
            // data is incomplete
        }
    }

    private void handleQueryResponse(CopilotServerResponse response, Map<String, String> sources) throws IOException {
        if (response.code() == CopilotServerResponseCode.ERROR.getCode()
                || response.code() == CopilotServerResponseCode.ERROR_REQUEST.getCode()) {
            getLogger().error("Copilot server returned error because an internal error."
                    + " The reason could be a malformed request or a timeout.");
            return;
        } else if (response.code() == CopilotServerResponseCode.NOTHING.getCode()) {
            getLogger().debug("Copilot server returned no changes");
            return;
        } else if (response.code() == CopilotServerResponseCode.HILLA_REACT.getCode()) {
            getLogger().debug("Copilot server returned Hilla/React changes");
        } else if (response.code() == CopilotServerResponseCode.FLOW.getCode()) {
            getLogger().debug("Copilot server returned Flow changes");
        } else if (response.code() == CopilotServerResponseCode.LOCAL.getCode()) {
            getLogger().debug("Copilot server returned Local changes");
        } else if (response.code() < 0) {
            getLogger().debug("Copilot server returned Internal error code: {}", response.code());
            return;
        } else {
            getLogger().debug("Copilot server returned unknown response code: {}", response.code());
            return;
        }

        for (Map.Entry<String, String> change : response.changes().entrySet()) {
            boolean fileIncludedInRequest = sources.containsKey(change.getKey());
            try {
                // if the file is not included in the request, we need to check if it
                // already exists in the project to allow new views or new components
                // but not to overwrite existing files that the AI does not know about
                if (!fileIncludedInRequest && getProjectFileManager().getAbsolutePath(change.getKey()).exists()) {
                    getLogger().warn("Ignoring changes to file that was not included in the AI request: {}",
                            change.getKey());
                    continue;
                }
                String source = change.getValue().replaceAll("// " + THE_FOLLOWING_ITEM_IS_SELECTED, "");
                getProjectFileManager().writeFile(getProjectFileManager().makeAbsolute(change.getKey()),
                        AIConstants.COPILOT_AI_FILE_UPDATE_UNDO_LABEL, source);
            } catch (IOException e) {
                throw new IOException(
                        "Unable to write file (" + change.getKey() + ") with data from copilot server response", e);
            }
        }
    }

    ProKey getProKey() {
        return LocalProKey.get();
    }

    String getMachineId() {
        return MachineId.get();
    }

    public Map<String, Disposable> getDisposables() {
        return disposables;
    }
}
