package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.userinfo.UserInfo;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.experimental.FeatureFlags;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.pro.licensechecker.BuildType;
import com.vaadin.pro.licensechecker.LicenseChecker;
import com.vaadin.pro.licensechecker.LicenseException;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.ProKey;
import com.vaadin.pro.licensechecker.Product;
import com.vaadin.pro.licensechecker.VaadinComIntegration;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * Handles getting information about the user (pro key based) and communicating
 * with the server about feature access.
 */
public class UserInfoHandler extends CopilotCommand {

    public static final String PRODUCT_VAADIN_COPILOT = "vaadin-copilot";
    public static final int LOGIN_TIMEOUT_SECONDS = 15 * 60;

    private final UserInfoServerClient client = new UserInfoServerClient();

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-user-info")) {
            ObjectNode userInfo = JacksonUtils.createObjectNode();
            UserInfo info = UserInfoServerClient.getUserInfoWithLocalProKey();
            if (info != null) {
                userInfo.put("validLicense", hasValidLicense());
                userInfo.put("firstName", info.firstname());
                userInfo.put("lastName", info.lastname());
                userInfo.put("email", info.email());
                userInfo.put("portraitUrl", info.portraitUrl());
                userInfo.put("vaadiner", info.vaadiner());
                userInfo.put("copilotProjectCannotLeaveLocalhost", info.copilotProjectCannotLeaveLocalhost());
            } else {
                userInfo.put("vaadiner", false);
            }

            userInfo.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());
            userInfo.put("copilotExperimentalFeatureFlag",
                    FeatureFlags.get(getVaadinContext()).isEnabled(FeatureFlags.COPILOT_EXPERIMENTAL));
            devToolsInterface.send("copilot-user-info", userInfo);
            return true;
        } else if (command.equals("log-in")) {
            ProKey proKey = LocalProKey.get();
            ObjectNode response = JacksonUtils.createObjectNode();
            response.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());
            if (proKey == null) {
                // Create a url so we can download the pro key. We don't really
                // care if the check succeeds or fails
                // as the client side will recheck the license after the login
                LicenseChecker.checkLicenseAsync(PRODUCT_VAADIN_COPILOT, CopilotVersion.getVersion(),
                        BuildType.DEVELOPMENT, new LicenseChecker.Callback() {
                            @Override
                            public void ok() {
                                devToolsInterface.send("copilot-prokey-received", JacksonUtils.createObjectNode());
                            }

                            @Override
                            public void failed(Exception e) {
                                getLogger().error("Unable to fetch pro key", e);
                            }
                        }, loginUrl -> {
                            // This is where we always should end up as there is
                            // no pro key
                            response.put("loginUrl", loginUrl);
                            response.put("reason", "no-pro-key");
                            devToolsInterface.send("log-in-resp", response);
                        }, LOGIN_TIMEOUT_SECONDS);
            } else {
                // There is a pro key, so probably the user does not have an
                // appropriate subscription. Use the default URL
                response.put("loginUrl", VaadinComIntegration.constructUrl(null,
                        new Product(PRODUCT_VAADIN_COPILOT, CopilotVersion.getVersion())));
                response.put("reason", "no-subscription");
                devToolsInterface.send("log-in-resp", response);
            }
            return true;
        }
        return false;
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

    private boolean hasValidLicense() {
        try {
            LicenseChecker.checkLicense(PRODUCT_VAADIN_COPILOT, CopilotVersion.getVersion(), BuildType.DEVELOPMENT,
                    null);
            return true;
        } catch (LicenseException e) {
            if (!ignoreWhenDoingBackgroundCheck(e)) {
                getLogger().error("Problem checking the subscription status", e);
            }
        }

        return false;
    }

    private boolean ignoreWhenDoingBackgroundCheck(LicenseException e) {
        String msg = e.getMessage();
        if (msg.contains("The provided license key does not allow usage")) {
            // There is a key but it does not allow access
            return true;
        }
        // Offline
        return msg.contains("could not be reached and no offline key was found");
    }
}
