package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.flow.internal.JacksonUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ObjectNode;

public class ProjectFileHandler extends CopilotCommand {

    private static final String COPILOT_UNDO_LABEL = CopilotIDEPlugin.undoLabel("File Update");
    private static final String COPILOT_DELETE_LABEL = CopilotIDEPlugin.undoLabel("File Delete");
    private static final Logger log = LoggerFactory.getLogger(ProjectFileHandler.class);

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-project-file")) {
            String filename = data.get("filename").asString();
            String reqId = data.get(KEY_REQ_ID).asString();

            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, reqId);
            try {
                responseData.put("content", getProjectFileManager().readFile(filename));
                devToolsInterface.send(Copilot.PREFIX + "project-file", responseData);
            } catch (IOException e) {
                getLogger().debug("Unable to read project file {}", filename, e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData, "Unable to read file", e);
            }
            return true;
        } else if (command.equals("write-project-file")) {
            String filename = data.get("filename").asString();
            String reqId = data.get(KEY_REQ_ID).asString();
            String content = data.get("content").asString();
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, reqId);
            try {
                getProjectFileManager().writeFile(filename, COPILOT_UNDO_LABEL, content);
                devToolsInterface.send(Copilot.PREFIX + "project-file-written", responseData);
            } catch (IOException e) {
                getLogger().debug("Unable to write project file {}", filename, e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData, "Unable to write file", e);
            }
            return true;
        } else if (command.equals("write-project-resource-binary")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());

            Optional<Path> resourceFolder = Util.findCurrentModule(getVaadinSession(), data.get("currentView"))
                    .flatMap(JavaSourcePathDetector.ModuleInfo::getOrGuessResourceFolder);
            String filename = data.get("filename").asString();
            String base64 = data.get("base64Content").asString();
            boolean renameIfExists = data.get("renameIfExists").asBoolean();
            try {
                if (resourceFolder.isEmpty()) {
                    throw new IOException("Unable to determine resource folder for the current view");
                }
                File written = getProjectFileManager().writeFileBase64(
                        resourceFolder.get().resolve(filename).toString(), COPILOT_UNDO_LABEL, base64, renameIfExists);
                responseData.put("filename", written.getName());
                devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            } catch (IOException e) {
                responseData.put("error", true);
                devToolsInterface.send(Copilot.PREFIX + "project-file-written", responseData);
                getLogger().error("Unable to write project file {}", filename, e);
            }
            return true;
        } else if (command.equals("get-project-root-path")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());
            File projectRoot = ProjectFileManager.get().getProjectRoot();
            responseData.put("projectRootPath", projectRoot.getAbsolutePath());
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        }

        return false;
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
