package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

import com.vaadin.flow.internal.JacksonUtils;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.ObjectMapper;
import tools.jackson.databind.node.ObjectNode;

/**
 * Utility class to create JSON response from Configuration files and save
 * configuration data from request
 */
public final class ConfigurationFileUtil {
    private ConfigurationFileUtil() {

    }

    /**
     * Gets response that contains configuration JSON.
     *
     * @param configFile
     *            file to read config
     * @return JSON object with <code>conf</code> key that has Configuration data
     */
    public static JsonNode getConfigResponse(File configFile) {
        ObjectNode response = JacksonUtils.createObjectNode();
        try {
            if (configFile.exists()) {
                String json = FileUtils.readFileToString(configFile, StandardCharsets.UTF_8);
                getLogger().debug("Reading configuration: {}", json);
                response.set("conf", JacksonUtils.readTree(json));
            } else {
                response.set("conf", JacksonUtils.readTree("{}"));
            }
        } catch (Exception e) {
            getLogger().error("Failed to read configuration from {}", configFile.toPath(), e);
            response.put("error", true);
        }
        return response;
    }

    /**
     * Saves the configuration into given file by extracting configuration data from
     * request
     *
     * @param configFile
     *            file to store configuration
     * @param request
     *            request that has <code>conf</code> object
     */
    public static void saveConfigFromRequest(File configFile, JsonNode request) {
        try {
            String json = request.get("conf").asString();
            ObjectNode jsonObject = new ObjectMapper().readValue(json, ObjectNode.class);
            String indented = new ObjectMapper().writerWithDefaultPrettyPrinter().writeValueAsString(jsonObject);
            FileUtils.write(configFile, indented, StandardCharsets.UTF_8);
            getLogger().debug("Writing configuration: {}", indented);
        } catch (IOException e) {
            getLogger().error("Failed to write configuration to {}", configFile.toPath(), e);
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ConfigurationFileUtil.class);
    }
}
