package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ide.IdeUtils;
import com.vaadin.copilot.javarewriter.JavaModifier;
import com.vaadin.flow.component.dependency.CssImport;
import com.vaadin.flow.component.page.AppShellConfigurator;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.theme.Theme;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import tools.jackson.databind.JsonNode;
import tools.jackson.databind.node.ObjectNode;

/**
 * Helps to initialize a Spring Boot Vaadin application with no views by adding
 * either a Flow or Hilla view + a theme.
 */
public class ApplicationInitializer extends CopilotCommand {
    public static final String SUCCESS_KEY = "success";
    public static final String REASON_KEY = "reason";
    public static final String INIT_APP_RESULT = "copilot-init-app-result";
    public static final String REFRESH_KEY = "refresh";

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("init-app")) {
            String framework = data.get("framework").asString();
            try {
                initApp(framework, devToolsInterface, data.get(KEY_REQ_ID).asString());
            } catch (IOException e) {
                getLogger().error("Unable to initialize project", e);
            }
            return true;
        } else if (command.equals("create-theme")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asString());
            try {
                createTheme();
                devToolsInterface.send(command + "-resp", responseData);
            } catch (IOException e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command + "-resp", responseData,
                        "Error creating theme", e);
            }
        }
        return false;
    }

    private synchronized void createTheme() throws IOException {

        if (!SpringBridge.isSpringAvailable(getVaadinContext())) {
            throw new IOException("This only works for Spring Boot projects");
        }
        Map<File, String> toWrite = new HashMap<>();

        Class<?> applicationClass = SpringBridge.getApplicationClass(getVaadinContext());
        File applicationClassFile = ProjectFileManager.get().getFileForClass(applicationClass);
        KotlinUtil.throwIfKotlin(applicationClassFile);
        addStylesCss(applicationClass, toWrite);
        for (Map.Entry<File, String> entry : toWrite.entrySet()) {
            getProjectFileManager().writeFile(entry.getKey(), "Create theme", entry.getValue());
        }
    }

    private synchronized void initApp(String framework, DevToolsInterface devToolsInterface, String reqId)
            throws IOException {
        ObjectNode responseData = JacksonUtils.createObjectNode();
        responseData.put(KEY_REQ_ID, reqId);

        if (!SpringBridge.isSpringAvailable(getVaadinContext())) {
            responseData.put(SUCCESS_KEY, false);
            responseData.put(REASON_KEY, "This only works for Spring Boot projects");
            devToolsInterface.send(INIT_APP_RESULT, responseData);
            return;
        }
        Map<File, String> toWrite = new HashMap<>();

        Class<?> applicationClass = SpringBridge.getApplicationClass(getVaadinContext());
        addStylesCss(applicationClass, toWrite);
        File toOpen;
        if (framework.equals("flow")) {
            Map<? extends File, String> flowView = addFlowView(applicationClass);
            toWrite.putAll(flowView);
            toOpen = flowView.keySet().iterator().next();
        } else if (framework.equals("hilla")) {
            Map<File, String> hillaView = addHillaView();
            toWrite.putAll(hillaView);
            toOpen = hillaView.keySet().iterator().next();
        } else {
            responseData.put(SUCCESS_KEY, false);
            responseData.put(REASON_KEY, "Unknown framework " + framework);
            devToolsInterface.send(INIT_APP_RESULT, responseData);
            return;
        }

        for (Map.Entry<File, String> entry : toWrite.entrySet()) {
            getProjectFileManager().writeFile(entry.getKey(), "Project generation", entry.getValue());
        }

        if (toOpen != null) {
            try {
                Thread.sleep(1000);
                IdeUtils.openFile(toOpen, 1);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        responseData.put(SUCCESS_KEY, true);
        responseData.put(REFRESH_KEY, true);
        devToolsInterface.send(INIT_APP_RESULT, responseData);
    }

    private void addStylesCss(Class<?> applicationClass, Map<File, String> toWrite) throws IOException {
        String stylesCss = "styles.css";
        Theme themeAnnotation = applicationClass.getAnnotation(Theme.class);
        // add import only when no theme or existing styles.css import
        if (themeAnnotation == null || themeAnnotation.value().isEmpty()) {
            CssImport[] cssImportAnnotation = applicationClass.getAnnotationsByType(CssImport.class);
            if (Stream.of(cssImportAnnotation).noneMatch(cssImport -> stylesCss.equals(cssImport.value()))) {
                toWrite.putAll(new JavaModifier().modify(applicationClass, operations -> {
                    if (!operations.hasClassAnnotation(CssImport.class)) {
                        // The class check above is not enough if somebody clicks multiple time on the
                        // 'create' link
                        operations.addClassAnnotation(CssImport.class, stylesCss);
                        operations.addInterface(AppShellConfigurator.class);
                    }
                }));
                toWrite.putAll(createStylesCss());
            }
        }

    }

    private Map<File, String> addHillaView() {
        File frontendFolder = getProjectFileManager().getFrontendFolder();
        File viewFile = new File(new File(frontendFolder, "views"), "@index.tsx");
        String viewTemplate = getViewTemplate(viewFile);

        return Collections.singletonMap(viewFile, viewTemplate);
    }

    private Map<? extends File, String> addFlowView(Class<?> applicationClass) {
        File applicationFile = getProjectFileManager().getFileForClass(applicationClass);
        String basePackage = applicationClass.getPackage().getName();
        File viewFolder = new File(applicationFile.getParentFile(), "views");
        File viewFile = new File(viewFolder, "HomeView.java");
        String relativeViewName = getProjectFileManager().getProjectRelativeName(viewFile);
        String viewPackage = basePackage + ".views";

        String content = String.format("""
                package %s;

                import com.vaadin.flow.component.html.H1;
                import com.vaadin.flow.component.html.Paragraph;
                import com.vaadin.flow.component.orderedlayout.VerticalLayout;
                import com.vaadin.flow.router.Route;

                @Route("")
                public class HomeView extends VerticalLayout {

                    public HomeView() {

                        add(new H1("Welcome to your new application"));
                        add(new Paragraph("This is the home view"));

                        add(new Paragraph("You can edit this view in %s"));

                    }
                }
                """, viewPackage, relativeViewName.replace("\\", "\\\\"));

        return Collections.singletonMap(viewFile, content);
    }

    private String getViewTemplate(File viewFile) {
        String relativeViewName = getProjectFileManager().getProjectRelativeName(viewFile);
        return String.format("""
                export default function HomeView() {
                  return (
                    <div>
                      <h1>Welcome to your new application</h1>
                      <p>This is the home view.</p>
                      <p>
                        You can edit this view in <code>%s</code> or by
                        activating Copilot by clicking the icon in the lower right corner
                      </p>
                    </div>
                  );
                }

                """, relativeViewName);
    }

    private Map<File, String> createStylesCss() {
        File stylesCss = getProjectFileManager().getStylesCss();
        if (!stylesCss.exists()) {
            return Collections.singletonMap(stylesCss, "");
        }
        return Collections.emptyMap();
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ApplicationInitializer.class);
    }
}
