package com.vaadin.copilot.testbenchgenerator;

import java.util.Objects;

/**
 * This class represents a possible selector of a target element. The proper
 * selector should be picked based on the priority e.g.
 * selectedAttributePairByUser has higher priority than id because it has been
 * set explicitly by user. The property should be as follows:
 * selectedAttributePairByUser, label, text, firstElementInDocument, id,
 * dataTestId, className, cssPath
 *
 * @param className
 *            class name of the element. Might be null
 * @param id
 *            id of the element. Might be null.
 * @param label
 *            label of the element. Might be null.
 * @param text
 *            text of the element. Might be null.
 * @param firstElementInDocument
 *            indicator that the element is the first element in the document
 *            with same tag
 * @param dataTestId
 *            value of <code>data-test-id</code> if present. Null otherwise.
 * @param cssPath
 *            CSS Path in the document. Note that using this might not produce
 *            correct output always.
 * @param selectedAttributePairByUser
 *            Attribute selected by the user.
 */
public record PossibleSelectors(String className, String id, String label, String text, Boolean firstElementInDocument,
        String dataTestId, String cssPath, SelectedAttributePairByUser selectedAttributePairByUser) {

    /**
     * Explicitly selected attribute by the user.
     * 
     * @param attrName
     *            Attribute name that will be used in withAttribute() as key
     * @param attrValue
     *            Attribute value that will be used in withAttribute() as value
     * @param comparisonType
     *            exact or contains.
     */
    public record SelectedAttributePairByUser(String attrName, String attrValue, String comparisonType) {
        @Override
        public boolean equals(Object o) {
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            SelectedAttributePairByUser that = (SelectedAttributePairByUser) o;
            return Objects.equals(attrName, that.attrName) && Objects.equals(attrValue, that.attrValue)
                    && Objects.equals(comparisonType, that.comparisonType);
        }

        @Override
        public int hashCode() {
            return Objects.hash(attrName, attrValue, comparisonType);
        }
    }

    @Override
    public boolean equals(Object o) {
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        PossibleSelectors that = (PossibleSelectors) o;
        return Objects.equals(id, that.id) && Objects.equals(text, that.text) && Objects.equals(label, that.label)
                && Objects.equals(cssPath, that.cssPath) && Objects.equals(className, that.className)
                && Objects.equals(dataTestId, that.dataTestId)
                && Objects.equals(firstElementInDocument, that.firstElementInDocument)
                && Objects.equals(selectedAttributePairByUser, that.selectedAttributePairByUser);
    }

    @Override
    public int hashCode() {
        return Objects.hash(className, id, label, text, firstElementInDocument, dataTestId, cssPath,
                selectedAttributePairByUser);
    }
}
