package com.vaadin.copilot.ide;

import static com.vaadin.copilot.plugins.themeeditor.ThemeModifier.THEME_EDITOR_CSS;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.SpringBridge;
import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

public class IdePluginCommandHandler extends CopilotCommand {

    private final CopilotIDEPlugin plugin = CopilotIDEPlugin.getInstance();

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        try {
            if (command.equals("plugin-undo")) {
                Optional<Boolean> maybePerformed = handlePluginUndoCommand(data)
                        .map(r -> r.get("performed").asBoolean());
                ObjectNode responseData = JacksonUtils.createObjectNode();
                responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());
                responseData.put("performed", maybePerformed.orElse(true));
                devToolsInterface.send("plugin-undo-response", responseData);
                return true;
            }

            if (command.equals("plugin-redo")) {
                Optional<Boolean> maybePerformed = handlePluginRedoCommand(data)
                        .map(r -> r.get("performed").asBoolean());
                ObjectNode responseData = JacksonUtils.createObjectNode();
                responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());
                responseData.put("performed", maybePerformed.orElse(true));
                devToolsInterface.send("plugin-redo-response", responseData);
                return true;
            }

            if (command.equals("plugin-restart-application")) {
                handlePluginRestartApplicationCommand();
                return true;
            }

            if (command.equals("plugin-refresh") && plugin.supports(CopilotIDEPlugin.Commands.REFRESH)) {
                handlePluginRefreshCommand();
                return true;
            }

        } catch (IOException e) {
            throw new UnsupportedOperationException("Copilot Plugin operation failed", e);
        }
        return false;
    }

    private Optional<JsonNode> handlePluginUndoCommand(JsonNode data) throws IOException {
        if (plugin.isActive() && plugin.supports(CopilotIDEPlugin.Commands.UNDO)) {
            List<String> files = listFiles(data);
            return plugin.undo(files);
        }
        return Optional.empty();
    }

    private Optional<JsonNode> handlePluginRedoCommand(JsonNode data) throws IOException {
        if (plugin.isActive() && plugin.supports(CopilotIDEPlugin.Commands.REDO)) {
            List<String> files = listFiles(data);
            return plugin.redo(files);
        }
        return Optional.empty();
    }

    private void handlePluginRestartApplicationCommand() {
        if (plugin.isActive() && plugin.supports(CopilotIDEPlugin.Commands.RESTART_APPLICATION)) {
            plugin.restartApplication(getMainClass());
        }
    }

    private void handlePluginRefreshCommand() throws IOException {
        if (plugin.isActive()) {
            plugin.refresh();
        }
    }

    private List<String> listFiles(JsonNode data) {
        List<String> files = new ArrayList<>();
        files.addAll(JacksonUtils.stream(data.withArray("files")).map(JsonNode::asText).map(this::filter).toList());
        return files;
    }

    private String filter(String filename) {
        if (filename.equals("$THEME_EDITOR_CSS")) {
            return new File(getProjectFileManager().getThemeFolder(), THEME_EDITOR_CSS).toString();
        }
        return filename;
    }

    private String getMainClass() {
        Class<?> appClass = SpringBridge.getApplicationClass(getVaadinContext());
        return appClass != null ? appClass.getName() : null;
    }
}
