package com.vaadin.copilot;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.WeakHashMap;
import java.util.function.Supplier;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.base.devserver.DevToolsMessageHandler;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;

import com.fasterxml.jackson.databind.JsonNode;

import com.github.javaparser.ParserConfiguration;

/**
 * This is the entry point for copilot.
 *
 * <p>
 * It is a singleton with the main purpose to forward messages to the correct
 * CopilotSession instance, based on which session the message is related to.
 */
public class Copilot implements DevToolsMessageHandler {

    public static final String PREFIX = "copilot-";
    public static final ParserConfiguration.LanguageLevel LANGUAGE_LEVEL = ParserConfiguration.LanguageLevel.JAVA_21;

    public static final String ISSUE_TRACKER = "https://github.com/vaadin/copilot/issues";
    public static final Supplier<IllegalStateException> sourceNotFound = () -> new IllegalStateException(
            "The component instance is not created in the project sources");

    private static Boolean enable = null;

    private final Map<VaadinSession, CopilotSession> copilotSessions = Collections.synchronizedMap(new WeakHashMap<>());

    @Override
    public void handleConnect(DevToolsInterface devToolsInterface) {
        VaadinSession vaadinSession = VaadinSession.getCurrent();
        if (vaadinSession == null || !isEnabled(vaadinSession)) {
            return;
        }
        copilotSessions.computeIfAbsent(vaadinSession, newSession -> {
            try {
                return new CopilotSession(newSession, devToolsInterface);
            } catch (IOException e) {
                throw new CopilotException(e);
            }
        }).handleConnect(devToolsInterface);
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        VaadinSession vaadinSession = VaadinSession.getCurrent();
        if (vaadinSession == null || !isEnabled(vaadinSession)) {
            return false;
        }

        if (!command.startsWith(Copilot.PREFIX)) {
            return false;
        }
        command = command.substring(Copilot.PREFIX.length());

        CopilotSession copilotSession = copilotSessions.get(vaadinSession);
        if (copilotSession == null) {
            // This should not happen, but if it does - what can we do?
            throw new CopilotException("No CopilotSession found for the current VaadinSession");
        }
        copilotSession.handleMessage(command, data, devToolsInterface);
        return true;
    }

    private static boolean isEnabled(VaadinSession vaadinSession) {
        if (enable == null) {
            enable = CopilotStatus.isEnabled(getContext(vaadinSession));
        }
        return enable;
    }

    static VaadinServletContext getContext(VaadinSession vaadinSession) {
        return (VaadinServletContext) vaadinSession.getService().getContext();
    }

    /**
     * Check if copilot is running in development mode i.e. is being developed.
     * <p>
     * Unrelated to Vaadin's production mode.
     *
     * @return true if copilot is running in development mode
     */
    public static boolean isDevelopmentMode() {
        return Boolean.getBoolean("copilot.development");
    }
}
