package com.vaadin.copilot.ide;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;

import com.vaadin.base.devserver.NamedDaemonThreadFactory;

import com.fasterxml.jackson.databind.node.ArrayNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A scheduler implementation that sends a request to IDE plugin in an interval
 * and shares the response with consumers.
 * <p>
 * Scheduler won't be created in the following conditions:
 * <ul>
 * <li>When plugin is not active</li>
 * <li>When plugin does not support
 * {@link com.vaadin.copilot.ide.CopilotIDEPlugin.Commands#HEARTBEAT}</li>
 * </ul>
 * </p>
 */
public class IDEHeartbeatScheduler {
    private static final String COMPILATION_ERROR_JSON_KEY = "hasCompilationError";
    private static final String FILES_CONTAIN_COMPILATION_ERROR_JSON_KEY = "filesContainCompilationError";

    private final ScheduledExecutorService scheduledExecutorService;
    private boolean started = false;
    private final Set<Consumer<IDEHeartbeatResponse>> consumers = ConcurrentHashMap.newKeySet();

    private static IDEHeartbeatScheduler instance;

    private IDEHeartbeatScheduler() {
        scheduledExecutorService = Executors
                .newSingleThreadScheduledExecutor(new NamedDaemonThreadFactory("copilot-compilation-status-checker"));
    }

    /**
     * Gets the singleton instance of {@link IDEHeartbeatScheduler}.
     * <p>
     * This method is synchronized to ensure thread-safety
     *
     * @return the instance
     */
    public static synchronized IDEHeartbeatScheduler getInstance() {
        if (instance == null) {
            instance = new IDEHeartbeatScheduler();
        }
        return instance;
    }

    /**
     * Starts scheduler if not started. This method considers state of IDE Plugin
     * activation, also works only if IDE Plugin supports HEARTBEAT command. Once it
     * is started, an internal flag is set so it cannot be started again. This
     * method is synchronized to ensure thread-safety
     */
    public synchronized void startIfNotStarted() {
        if (this.started) {
            return;
        }
        if (!CopilotIDEPlugin.getInstance().isActive()) {
            return;
        }
        if (!CopilotIDEPlugin.getInstance().supports(CopilotIDEPlugin.Commands.HEARTBEAT)) {
            return;
        }
        scheduledExecutorService.scheduleAtFixedRate(this::sendHeartbeatRequestAndCallConsumers, 5, 5,
                TimeUnit.SECONDS);
        this.started = true;
    }

    /**
     * Saves the consumer into a set so that gets heartbeat responses from the IDE
     */
    public synchronized void addConsumer(Consumer<IDEHeartbeatResponse> consumer) {
        this.consumers.add(consumer);
    }

    private void sendHeartbeatRequestAndCallConsumers() {
        CopilotIDEPlugin.getInstance().heartbeat().map(jsonObject -> {
            Boolean compilationError = null;
            if (jsonObject.has(COMPILATION_ERROR_JSON_KEY) && jsonObject.get(COMPILATION_ERROR_JSON_KEY).isBoolean()) {
                compilationError = jsonObject.get(COMPILATION_ERROR_JSON_KEY).asBoolean();
            }
            List<String> filePaths = new ArrayList<>();
            if (jsonObject.has(FILES_CONTAIN_COMPILATION_ERROR_JSON_KEY)
                    && jsonObject.get(FILES_CONTAIN_COMPILATION_ERROR_JSON_KEY).isArray()) {
                ArrayNode array = jsonObject.withArray(FILES_CONTAIN_COMPILATION_ERROR_JSON_KEY);
                for (int i = 0; i < array.size(); i++) {
                    filePaths.add(array.get(i).asText());
                }
            }
            return new IDEHeartbeatResponse(compilationError, filePaths);
        }).ifPresent(ideHeartbeatResponse -> consumers.forEach(consumer -> {
            try {
                consumer.accept(ideHeartbeatResponse);
            } catch (Exception e) {
                getLogger().debug("Error sending IDE heartbeat response", e);
            }
        }));
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(IDEHeartbeatScheduler.class);
    }
}
