package com.vaadin.copilot;

import java.io.File;
import java.nio.file.Path;
import java.util.List;

import com.vaadin.copilot.exception.KotlinNotSupportedException;

public class KotlinUtil {
    private KotlinUtil() {
    }

    /**
     * Checks file extension is kotlin extensions .e.g kt, kts and throws
     * {@link KotlinNotSupportedException} when file has one of the extensions
     *
     * @param file
     *            File to check
     */
    public static void throwIfKotlin(File file) {
        if (file == null) {
            return;
        }
        if (isKotlin(file)) {
            throw new KotlinNotSupportedException();
        }
    }

    /**
     * Checks file extension is kotlin extensions .e.g kt, kts
     *
     * @param file
     *            File to check
     * @return true if kotlin file, false otherwise.
     */
    public static boolean isKotlin(File file) {
        return Util.hasExtension(file.getName(), "kt", "kts");
    }

    /**
     * Checks whether project source folders include Kotlin sources.
     *
     * @param projectFileManager
     *            Project file manager to inspect
     * @return true if Kotlin sources are detected, false otherwise.
     */
    public static boolean isKotlinProject(ProjectFileManager projectFileManager) {
        if (projectFileManager == null) {
            return false;
        }
        List<Path> sourceFolders = projectFileManager.getSourceFolders();
        if (sourceFolders == null || sourceFolders.isEmpty()) {
            return false;
        }
        return sourceFolders.stream().anyMatch(KotlinUtil::isKotlinSourcePath);
    }

    static boolean isKotlinSourcePath(Path path) {
        if (path == null) {
            return false;
        }
        for (Path part : path) {
            if ("kotlin".equalsIgnoreCase(part.toString())) {
                return true;
            }
        }
        return false;
    }

}
