package com.vaadin.copilot;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpression;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.List;
import java.util.Optional;
import java.util.stream.IntStream;

import com.vaadin.copilot.exception.CopilotException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

/**
 * Pom File rewriter handles modifications for the given pom file by copying
 * original file to temporary file. When {@link #save()} method is called,
 * original file would be replaced with temporary file.
 * <p>
 * Rewriter file takes two steps. SAX parser traverses through all elements to
 * gather line numbers, and they are used to replace nodes using String
 * manipulation to keep file formatting preserved
 * </p>
 */
public class PomFileRewriter {
    private static final String LINE_NUM_ATTRIBUTE_NAME = "lineNumber";
    private final Document document;
    private final Path originalFile;

    private final List<ReplacementChanges> replacementChanges = new ArrayList<>();
    private final List<AddChanges> addChanges = new ArrayList<>();

    /**
     * Creates a rewriter for the given pom.xml file.
     * 
     * @param file
     *            the pom.xml file to be modified
     * @throws IOException
     *             if the file cannot be read
     * @throws SAXException
     *             if a parsing error occurred
     */
    public PomFileRewriter(Path file) throws IOException, SAXException {
        this.originalFile = file;
        this.document = readXML(new FileInputStream(file.toFile()));
    }

    /**
     * Finds dependency under both dependencies and dependencyManagement tags by
     * groupId and artifactId
     *
     * @param groupId
     *            Group ID to look for e.g. com.vaadin
     * @param artifactId
     *            Artifact ID to look for e.g. flow-server
     * @return Dependency if found, <code>null</code> otherwise
     */
    public Dependency findDependencyByGroupIdAndArtifactId(String groupId, String artifactId) {
        String expression = String.format(
                "//dependencies/dependency[groupId='%s' and artifactId='%s'] | //dependencyManagement/dependencies/dependency[groupId='%s' and artifactId='%s']",
                groupId, artifactId, groupId, artifactId);
        Optional<NodeList> nodesUsingXPath = findNodesUsingXPath(expression);
        if (nodesUsingXPath.isEmpty()) {
            return null;
        }
        NodeList nodeList = nodesUsingXPath.get();
        if (nodeList.getLength() == 0) {
            return null;
        }
        if (nodeList.item(0).getNodeType() != Node.ELEMENT_NODE) {
            return null;
        }
        Element element = (Element) nodeList.item(0);
        Node versionNode = element.getElementsByTagName("version").item(0);
        if (versionNode == null) {
            return null;
        }
        return new Dependency(this, artifactId, groupId, versionNode.getTextContent(), getLineNumber(versionNode));
    }

    /**
     * Finds the dependency with the given groupId and artifactId.
     * 
     * Does not try to resolve the version unless it is part of the same
     * &lt;dependency&gt; tag.
     *
     * @param groupId
     *            Group ID to look for e.g. com.vaadin
     * @param artifactId
     *            Artifact ID to look for e.g. flow-server
     * @return An {@link Optional} containing the dependency if found, or an empty
     *         {@link Optional} if not found.
     */
    public Optional<Dependency> findDependency(String groupId, String artifactId) {
        String expression = String.format("//dependencies/dependency[groupId='%s' and artifactId='%s']", groupId,
                artifactId);
        Optional<Element> dependencyElement = findNodesUsingXPath(expression).flatMap(PomFileRewriter::getFirstElement);
        if (dependencyElement.isEmpty()) {
            return Optional.empty();
        }
        Optional<Element> versionElement = dependencyElement.map(element -> element.getElementsByTagName("version"))
                .flatMap(PomFileRewriter::getFirstElement);

        String version = versionElement.map(Node::getTextContent).orElse(null);
        int lineNumber = dependencyElement.map(this::getLineNumber).orElse(-1);
        return Optional.of(new Dependency(this, artifactId, groupId, version, lineNumber));
    }

    private static Optional<Element> getFirstElement(NodeList nodeList) {
        for (int i = 0; i < nodeList.getLength(); i++) {
            Node node = nodeList.item(i);
            if (node.getNodeType() == Node.ELEMENT_NODE) {
                return Optional.of((Element) node);
            }
        }
        return Optional.empty();
    }

    /**
     * Checks repository exists in repositories tag by id and url
     *
     * @param id
     *            id of the repository
     * @param url
     *            url of the repository
     * @return <code>true</code> if found, <code>false</code> otherwise.
     */
    public boolean hasRepositoryByIdAndUrl(String id, String url) {
        String expression = String.format("//project/repositories/repository[id='%s' and url='%s']", id, url);
        return getFirstNodeInExpressionIfElementType(expression).isPresent();
    }

    /**
     * Checks repository exists in repositories tag by id
     *
     * @param id
     *            id of the repository
     * @return <code>true</code> if found, <code>false</code> otherwise.
     */
    public boolean hasRepositoryById(String id) {
        String expression = String.format("//project/repositories/repository[id='%s']", id);
        return getFirstNodeInExpressionIfElementType(expression).isPresent();
    }

    /**
     * Checks repository exists in repositories tag by url
     *
     * @param url
     *            url of the repository
     * @return <code>true</code> if found, <code>false</code> otherwise.
     */
    public boolean hasRepositoryByUrl(String url) {
        String expression = String.format("//project/repositories/repository[url='%s']", url);
        return getFirstNodeInExpressionIfElementType(expression).isPresent();
    }

    /**
     * Checks repositories tag present in project as a child.
     *
     * @return Repository node if found, {@link Optional#empty()} otherwise.
     */
    public Optional<Element> getRepositoriesElement() {
        return getFirstNodeInExpressionIfElementType("//project/repositories");
    }

    /**
     * Adds repository to given pom.xml if it does not exist in the repositories.
     * Repositories tag is added at the end of the file when it is not present.
     *
     * @param id
     *            id attribute of the repository
     * @param url
     *            url attribute of the repository
     */
    public void addRepository(String id, String url) {
        if (hasRepositoryByUrl(url) || hasRepositoryByUrl(toggleTrailingSlash(url))) {
            return;
        }

        if (hasRepositoryById(id)) {
            getLogger().warn(
                    "pom.xml already has a repository with id {} but it does not refer to {}. This might not work as expected.",
                    id, url);
            return;
        }

        Optional<Element> repositoriesElementOptional = getRepositoriesElement();
        if (repositoriesElementOptional.isEmpty()) {
            String textToAdd = String.format("""
                    \t<repositories>
                    \t    %s
                    \t</repositories>""", getRepositoryNodeText(id, url));
            addChanges.add(new AddChanges(textToAdd));
        } else {
            Element repositories = repositoriesElementOptional.get();
            int lineNumber = getLineNumber(repositories);
            addChanges.add(new AddChanges(getRepositoryNodeText(id, url), lineNumber));
        }
    }

    private String getRepositoryNodeText(String id, String url) {
        return String.format("""
                <repository>
                            <id>%s</id>
                            <url>%s</url>
                        </repository>""", id, url);
    }

    /**
     * Checks pluginRepositories tag present in project as a child.
     *
     * @return PluginRepository node if found, {@link Optional#empty()} otherwise.
     */
    public Optional<Element> getPluginRepositoriesElement() {
        return getFirstNodeInExpressionIfElementType("//project/pluginRepositories");
    }

    /**
     * Adds pluginRepository to given pom.xml if it does not exist in the
     * pluginRepositories. PluginRepositories tag is added at the end of the file
     * when it is not present.
     *
     * @param id
     *            id attribute of the plugin repository
     * @param url
     *            url attribute of the plugin repository
     */
    public void addPluginRepository(String id, String url) {
        if (hasPluginRepositoryByUrl(url) || hasPluginRepositoryByUrl(toggleTrailingSlash(url))) {
            return;
        }

        if (hasPluginRepositoryById(id)) {
            getLogger().warn(
                    "pom.xml already has a plugin repository with id {} but it does not refer to {}. This might not work as expected.",
                    id, url);
            return;
        }
        Optional<Element> pluginRepositoriesElementOptional = getPluginRepositoriesElement();
        if (pluginRepositoriesElementOptional.isEmpty()) {
            String textToAdd = String.format("""
                    \t<pluginRepositories>
                    \t    %s
                    \t</pluginRepositories>""", getPluginRepositoryNodeText(id, url));
            addChanges.add(new AddChanges(textToAdd));
        } else {
            Element pluginRepositories = pluginRepositoriesElementOptional.get();
            int lineNumber = getLineNumber(pluginRepositories);
            addChanges.add(new AddChanges(getPluginRepositoryNodeText(id, url), lineNumber));
        }
    }

    private String toggleTrailingSlash(String url) {
        if (url.endsWith("/")) {
            return url.substring(0, url.length() - 1);
        } else {
            return url + "/";
        }
    }

    private String getPluginRepositoryNodeText(String id, String url) {
        return String.format("""
                <pluginRepository>
                            <id>%s</id>
                            <url>%s</url>
                        </pluginRepository>""", id, url);
    }

    /**
     * Checks repository exists in repositories tag by id and url
     *
     * @param id
     *            id of the repository
     * @param url
     *            url of the repository
     * @return <code>true</code> if found, <code>false</code> otherwise.
     */
    public boolean hasPluginRepositoryByIdAndUrl(String id, String url) {
        String expression = String.format("//project/pluginRepositories/pluginRepository[id='%s' and url='%s']", id,
                url);
        return getFirstNodeInExpressionIfElementType(expression).isPresent();
    }

    /**
     * Checks repository exists in repositories tag by id and url
     *
     * @param id
     *            id of the repository
     * @return <code>true</code> if found, <code>false</code> otherwise.
     */
    public boolean hasPluginRepositoryById(String id) {
        String expression = String.format("//project/pluginRepositories/pluginRepository[id='%s']", id);
        return getFirstNodeInExpressionIfElementType(expression).isPresent();
    }

    /**
     * Checks repository exists in repositories tag by id and url
     *
     * @param url
     *            url of the repository
     * @return <code>true</code> if found, <code>false</code> otherwise.
     */
    public boolean hasPluginRepositoryByUrl(String url) {
        String expression = String.format("//project/pluginRepositories/pluginRepository[url='%s']", url);
        return getFirstNodeInExpressionIfElementType(expression).isPresent();
    }

    /**
     * Returns the first node from the given expression if type is element.
     *
     * @param expression
     *            XPath expression
     * @return <code>true</code> if found, <code>false</code> when first element is
     *         absent or not an element node
     */
    private Optional<Element> getFirstNodeInExpressionIfElementType(String expression) {
        Optional<NodeList> nodesUsingXPath = findNodesUsingXPath(expression);
        if (nodesUsingXPath.isEmpty()) {
            return Optional.empty();
        }
        NodeList nodeList = nodesUsingXPath.get();
        if (nodeList.getLength() == 0) {
            return Optional.empty();
        }

        if (nodeList.item(0).getNodeType() == Node.ELEMENT_NODE) {
            return Optional.of((Element) nodeList.item(0));
        }
        return Optional.empty();
    }

    /**
     * Checks if given pom file has <code>parent</code> tag.
     *
     * @return <code>true</code> when parent node is present under Project tag,
     *         <code>false</code> otherwise.
     */
    public boolean hasParentPom() {
        return getFirstNodeInExpressionIfElementType("//project/parent").isPresent();
    }

    private Optional<NodeList> findNodesUsingXPath(String expression) {
        try {
            XPathFactory xPathFactory = XPathFactory.newInstance();
            XPath xPath = xPathFactory.newXPath();
            XPathExpression xPathExpression = xPath.compile(expression);
            NodeList nodes = (NodeList) xPathExpression.evaluate(document, XPathConstants.NODESET);
            return Optional.ofNullable(nodes);
        } catch (XPathExpressionException e) {
            getLogger().error("Could not evaluate XPath: {}", expression, e);
            return Optional.empty();
        }
    }

    /**
     * Checks given <code>propertyKey</code> is a property under properties tag.
     *
     * @param propertyKey
     *            Property key to search for. It should be exactly the same as it is
     *            in pom.xml
     * @return Property when found, <code>null</code> otherwise.
     */
    public Property findPropertyByKey(String propertyKey) {
        Optional<NodeList> propertiesNodeListOptional = findNodesUsingXPath("//project/properties");
        if (propertiesNodeListOptional.isEmpty()) {
            return null;
        }
        NodeList nodeList = propertiesNodeListOptional.get();
        if (nodeList.getLength() == 0) {
            return null;
        }
        Node propertiesElement = nodeList.item(0);
        NodeList childNodes = propertiesElement.getChildNodes();
        for (int j = 0; j < childNodes.getLength(); j++) {
            if (childNodes.item(j).getNodeType() == Node.ELEMENT_NODE) {
                Element childElement = (Element) childNodes.item(j);
                if (childElement.getTagName().equals(propertyKey)) {
                    return new Property(this, propertyKey, childElement.getTextContent(), getLineNumber(childElement));
                }
            }
        }
        return null;
    }

    /**
     * Apply all modifications to a temporary file and then replaces it with the
     * real one.
     *
     * @throws IOException
     *             when a file operation fails.
     */
    public void save() throws IOException {
        String newFileContent = applyModificationsAndNewFileContent();
        if (newFileContent == null) {
            // No changes
            return;
        }
        Path tempFilePath = null;
        try {
            tempFilePath = Files.createTempFile("pom", ".xml");
            Files.copy(this.originalFile, tempFilePath, StandardCopyOption.REPLACE_EXISTING);
            Files.writeString(tempFilePath, newFileContent);
            Files.copy(tempFilePath, this.originalFile, StandardCopyOption.REPLACE_EXISTING);
        } finally {
            if (tempFilePath != null) {
                try {
                    Files.deleteIfExists(tempFilePath);
                } catch (IOException e) {
                    getLogger().warn("Could not delete temporary file: {}", tempFilePath, e);
                }
            }
        }
    }

    /**
     * Applies all modifications to the original file and returns the new file
     * content or null if no changes were made.
     * 
     * @return new file content or null if no changes were made
     * @throws IOException
     *             if something goes wrong
     */
    private String applyModificationsAndNewFileContent() throws IOException {
        List<String> lines = Files.readAllLines(this.originalFile);
        boolean hasChanges = false;
        for (ReplacementChanges replacementChange : replacementChanges) {
            String line = lines.get(replacementChange.lineNumber);
            if (!line.contains(replacementChange.previousText)) {
                throw new CopilotException(
                        "Could not find " + replacementChange.previousText + " at " + replacementChange.lineNumber);
            }
            line = line.replace(replacementChange.previousText, replacementChange.newText);
            lines.set(replacementChange.lineNumber, line);
            hasChanges = true;
        }
        for (AddChanges addChange : addChanges) {
            if (addChange.lineNumberAfter == -1) {
                int projectEndTag = IntStream.range(0, lines.size())
                        .filter(lineIndex -> lines.get(lineIndex).contains("</project>")).findFirst()
                        .orElseThrow(() -> new CopilotException("Pom.xml is corrupted."));
                lines.add(projectEndTag, addChange.newText);
            } else {
                lines.add(addChange.lineNumberAfter + 1, addChange.newText);
            }
            hasChanges = true;
        }
        if (!hasChanges) {
            return null;
        }

        return String.join("\n", lines);
    }

    /**
     * Saves replacement for version tag of given dependency.
     *
     * @param dependency
     *            dependency that will be updated
     * @param newVersion
     *            new version for the dependency
     */
    public void updateDependencyVersion(Dependency dependency, String newVersion) {
        replacementChanges.add(new ReplacementChanges(dependency.version, newVersion, dependency.versionLineNumber));
    }

    /**
     * Saves replacement for the text content of given property
     *
     * @param property
     *            property that will be updated
     * @param newValue
     *            new text content of the given property
     */
    public void updateProperty(Property property, String newValue) {
        replacementChanges.add(new ReplacementChanges(property.propertyValue, newValue, property.lineNumber));
    }

    private int getLineNumber(Node node) {
        return Integer.parseInt((String) node.getUserData(LINE_NUM_ATTRIBUTE_NAME));
    }

    private Document readXML(InputStream is) throws IOException, SAXException {
        final Document doc;
        SAXParser parser;
        try {
            SAXParserFactory factory = SAXParserFactory.newDefaultInstance();
            parser = factory.newSAXParser();
            DocumentBuilderFactory docBuilderFactory = DocumentBuilderFactory.newInstance();
            DocumentBuilder docBuilder = docBuilderFactory.newDocumentBuilder();
            doc = docBuilder.newDocument();
        } catch (ParserConfigurationException e) {
            throw new CopilotException("Can't create SAX parser / DOM builder.", e);
        }

        final Deque<Element> elementStack = new ArrayDeque<>();
        final StringBuilder textBuffer = new StringBuilder();

        // handler to set line numbers for elements
        DefaultHandler handler = new DefaultHandler() {
            private Locator locator;

            @Override
            public void setDocumentLocator(Locator locator) {
                this.locator = locator;
            }

            @Override
            public void startElement(String uri, String localName, String qName, Attributes attributes) {
                addTextIfNeeded();
                Element el = doc.createElement(qName);
                for (int i = 0; i < attributes.getLength(); i++)
                    el.setAttribute(attributes.getQName(i), attributes.getValue(i));
                el.setUserData(LINE_NUM_ATTRIBUTE_NAME, String.valueOf(locator.getLineNumber() - 1), null);
                elementStack.push(el);
            }

            @Override
            public void endElement(String uri, String localName, String qName) {
                addTextIfNeeded();
                Element closedEl = elementStack.pop();
                if (elementStack.isEmpty()) { // Is this the root element?
                    doc.appendChild(closedEl);
                } else {
                    Element parentEl = elementStack.peek();
                    parentEl.appendChild(closedEl);
                }
            }

            @Override
            public void characters(char[] ch, int start, int length) {
                textBuffer.append(ch, start, length);
            }

            // Outputs text accumulated under the current node
            private void addTextIfNeeded() {
                if (!textBuffer.isEmpty()) {
                    Element el = elementStack.peek();
                    Node textNode = doc.createTextNode(textBuffer.toString());
                    assert el != null;
                    el.appendChild(textNode);
                    textBuffer.delete(0, textBuffer.length());
                }
            }
        };
        parser.parse(is, handler);

        return doc;
    }

    /**
     * Adds dependency to the pom.xml file, if needed.
     * <p>
     * Returns true if the dependency was added, <code>false</code> if it already
     * existed.
     * 
     * @param groupId
     *            the groupId of the dependency
     * @param artifactId
     *            the artifactId of the dependency
     * @param version
     *            the version of the dependency or <code>null</code> if not
     *            specified
     * @return <code>true</code> if the dependency was added, <code>false</code> if
     *         it already existed
     */
    public boolean addDependency(String groupId, String artifactId, String version) {
        if (findDependency(groupId, artifactId).isPresent()) {
            return false;
        }
        Optional<Element> dependenciesSection = getFirstNodeInExpressionIfElementType("//project/dependencies");

        String newDependency = String.format("""
                <dependency>
                    <groupId>%s</groupId>
                    <artifactId>%s</artifactId>
                """, groupId, artifactId);
        if (version != null) {
            newDependency += String.format("""
                    <version>%s</version>
                    """, version);
        }
        newDependency += """
                </dependency>""";

        if (dependenciesSection.isEmpty()) {
            addChanges.add(new AddChanges(String.format("""
                                        <dependencies>
                    %s
                                        </dependencies>""", newDependency)));
        } else {
            int lineNumber = getLineNumber(dependenciesSection.get());
            addChanges.add(new AddChanges(newDependency, lineNumber));
        }
        return true;
    }

    /**
     * Property that presents in the pom.xml
     *
     * @param fileRewriter
     *            PomFileRewriter where property is present
     * @param propertyKey
     *            property key
     * @param propertyValue
     *            property value
     * @param lineNumber
     *            Line number of the property
     */
    public record Property(PomFileRewriter fileRewriter, String propertyKey, String propertyValue, int lineNumber) {
    }

    /**
     * Dependency that presents in the pom.xml
     *
     * @param fileRewriter
     *            PomFileRewriter where dependency is present
     * @param artifactId
     *            ArtifactId
     * @param groupId
     *            GroupId
     * @param version
     *            Version
     * @param versionLineNumber
     *            Line number of version node of dependency in file
     */
    public record Dependency(PomFileRewriter fileRewriter, String artifactId, String groupId, String version,
            int versionLineNumber) {
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }

    /**
     * Contains texts that will be replaced before save action.
     *
     * @param previousText
     * @param newText
     * @param lineNumber
     */
    private record ReplacementChanges(String previousText, String newText, int lineNumber) {
    }

    private record AddChanges(String newText, int lineNumberAfter) {
        // -1 means at the end of file
        public AddChanges(String newText) {
            this(newText, -1);
        }
    }
}
