package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.copilot.JavaReflectionUtil;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.SpringBridge;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.flow.server.VaadinServletContext;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.Name;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.SingleMemberAnnotationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.helger.collection.commons.ICommonsList;
import com.helger.css.decl.CSSImportRule;
import com.helger.css.decl.CascadingStyleSheet;
import com.helger.css.reader.CSSReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LumoRewriterUtil {
    private static final String addClassName = "addClassName";
    private static final String addClassNames = "addClassNames";
    private static final String setClassName = "setClassName";
    private static final List<String> classNameMethods = List.of(addClassName, addClassNames, setClassName);

    private static final String LUMO_UTILITY_CSS_FILE = "lumo/utility.css";

    private static final String COPILOT_STYLES_MODIFY_UNDO_LABEL = CopilotIDEPlugin.undoLabel("Styles Modify");

    private LumoRewriterUtil() {
    }

    /**
     * Map that stores string class name as key,lumo utility variable as value.
     * Example: <br>
     * <code>"p-0"</code> -> <code>"LumoUtility.Padding.NONE"</code><br>
     * <code>"p-m"</code> -> <code>"LumoUtility.Padding.MEDIUM"</code>
     */
    private static final Map<String, String> classNameLumoUtilityClassCacheMap = new HashMap<>();

    /**
     * Creates addClassNames call for the given owner using value. When value is
     * plain string, it searches lumo utility replacements and replaces plain string
     * class names with LumoUtility classes as possible.<br>
     * <br>
     * If there is no change, it returns null <code>"gap-m p-m"</code> -> <code>
     * LumoUtility.Gap.MEDIUM, LumoUtility.Padding.MEDIUM</code> <br>
     * <code>"foo gap-m"</code> -> <code>"foo", LumoUtility.Gap.MEDIUM </code>
     *
     * @param owner
     *            owner of the adding class related call.
     * @param value
     *            Class name arg value. If it is not string, returns null.
     * @param cu
     *            Complication unit to add lumo utility import if required
     * @return methodCallExpr that is owner scoped addClassNames. e.g. <br>
     *         <code>fooBtn.addClassNames(LumoUtility.Position.ABSOLUTE);</code>
     */
    public static Expression createAddClassNameExprUsingLumoVariables(Expression owner, Object value,
            CompilationUnit cu) {
        if (!(value instanceof String)) {
            return null;
        }
        NodeList<Expression> classNameArgExpressions = new NodeList<>();
        String[] classNames = ((String) value).split("\\s+");
        boolean addedAny = false;
        for (String className : classNames) {
            String lumoVariable = getLumoVariable(className);
            if (lumoVariable != null) {
                addedAny = true;
                classNameArgExpressions.add(new NameExpr(lumoVariable));
            } else {
                classNameArgExpressions.add(new StringLiteralExpr(className));
            }
        }
        if (!addedAny) {
            return null;
        }
        LumoRewriterUtil.addLumoUtilityImport(cu);
        return new MethodCallExpr(owner, addClassNames, classNameArgExpressions);
    }

    /**
     * Creates addClassNames(...) statement with given arguments if there is none.
     * Adds arguments to existing one.
     *
     * @param component
     *            the component to add class name
     * @param arguments
     *            class name arguments. Most likely StringLiteral or FieldAccessExpr
     *            generated for LumoUtility.
     * @return true if added successfully, false otherwise.
     */
    public static boolean addClassNameWithArgs(ComponentInfo component, List<Expression> arguments) {
        MethodCallExpr methodClassExpr;
        if (component.routeConstructor() != null) {
            methodClassExpr = new MethodCallExpr();
        } else {
            if (component.isAnonymousComponent()) {
                JavaRewriter.ExtractInlineVariableResult extractInlineVariableResult = JavaRewriterUtil
                        .extractInlineVariableToLocalVariable(component);
                if (extractInlineVariableResult == null) {
                    return false;
                }
                methodClassExpr = new MethodCallExpr(new NameExpr(extractInlineVariableResult.newVariableName()),
                        addClassName);
                arguments.forEach(methodClassExpr::addArgument);
                extractInlineVariableResult.blockStmt().addStatement(extractInlineVariableResult.index() + 1,
                        methodClassExpr);
                return true;
            }
            String varName = component.getCreateInfoOrThrow().getLocalVariableName() != null
                    ? component.getCreateInfoOrThrow().getLocalVariableName()
                    : component.getCreateInfoOrThrow().getFieldName();
            methodClassExpr = new MethodCallExpr(new NameExpr(varName), addClassName);
        }
        arguments.forEach(methodClassExpr::addArgument);
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(component);

        List<MethodCallExpr> classNameMethodCalls = methodCallStatements.stream()
                .filter(f -> classNameMethods.contains(f.getNameAsString())).toList();
        // consider only classes after setClassName is set
        for (int i = classNameMethodCalls.size() - 1; i >= 0; i--) {
            if (classNameMethodCalls.get(i).getNameAsString().equals(setClassName)) {
                classNameMethodCalls = classNameMethodCalls.subList(i + 1, classNameMethodCalls.size());
                break;
            }
        }

        Optional<MethodCallExpr> addClassNamesOptional = classNameMethodCalls.stream()
                .filter(f -> f.getNameAsString().equals(addClassNames)).findAny();
        // add argument to existing one.
        if (addClassNamesOptional.isPresent()) {
            arguments.forEach(arg -> addClassNamesOptional.get().addArgument(arg));
            return true;
        } else {
            // creates new addClassNames statement
            return JavaRewriterUtil.addAfterLastFunctionCall(methodCallStatements, addClassNames,
                    arguments.toArray(Expression[]::new)) != null;
        }
    }

    /**
     * Searches addClassName, addClassNames, setClassName methods for given
     * component and then remove given lumo utility class names. e.g. if self-start
     * is given for removal, <b>LumoUtility.AlignSelf.START</b> and
     * <b>"self-start"</b> are removed from args.
     *
     * @param component
     *            the component to remove class names
     * @param lumoUtilityClassNames
     *            Utility class names such as "align-items", "gap-m"
     */
    public static void removeClassNameArgs(ComponentInfo component, String... lumoUtilityClassNames) {
        List<String> classNameDefinitionMethodNames = List.of(addClassName, addClassNames, setClassName);
        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(component);
        List<MethodCallExpr> classNameDefinitionMethods = methodCallStatements.stream()
                .filter(f -> classNameDefinitionMethodNames.contains(f.getNameAsString())).toList();
        List<Expression> argsWillBeRemoved = new ArrayList<>();
        for (String lumoUtilityClassName : lumoUtilityClassNames) {
            argsWillBeRemoved.addAll(getPossibleLumoUtilityMethodArgExpressions(lumoUtilityClassName));
        }
        JavaRewriterUtil.removeArgumentCalls(classNameDefinitionMethods, argsWillBeRemoved, true);
    }

    public static void addLumoUtilityImport(CompilationUnit compilationUnit) {
        JavaRewriterUtil.addImport(compilationUnit, "com.vaadin.flow.theme.lumo.LumoUtility");
    }

    /**
     * @param lumoInnerClassName
     *            Inner class name of LumoUtility class. e.g. AlignSelf, AlignItems
     *            etc...
     * @return the list of expressions
     */
    private static List<Expression> getPossibleLumoUtilityMethodArgExpressions(String lumoInnerClassName) {
        BiMap<String, String> utilityClasses = getLumoFieldsNameValueMap(lumoInnerClassName);
        List<Expression> lumoRelatedUtilityClasses = new ArrayList<>();
        lumoRelatedUtilityClasses.addAll(utilityClasses.keySet().stream()
                .map(arg -> new FieldAccessExpr(new NameExpr("LumoUtility." + lumoInnerClassName), arg)).toList());
        lumoRelatedUtilityClasses.addAll(utilityClasses.values().stream().map(StringLiteralExpr::new).toList());
        return lumoRelatedUtilityClasses;
    }

    /**
     * Converts given class names to LumoUtility field access expressions. e.g.
     * items-center becomes AlignItems.CENTER
     *
     * @param lumoInnerClassName
     *            Inner class name of LumoUtility class. e.g. AlignSelf, AlignItems
     *            etc...
     * @param classNames
     *            html class names
     * @return list of expressions.
     */
    public static List<Expression> getLumoMethodArgExpressions(String lumoInnerClassName, List<String> classNames) {
        BiMap<String, String> classKeyValueMap = getLumoFieldsNameValueMap(lumoInnerClassName);
        return classNames.stream().map(lumoClassValue -> classKeyValueMap.inverse().get(lumoClassValue))
                .filter(Objects::nonNull)
                .map(arg -> new FieldAccessExpr(new NameExpr("LumoUtility." + lumoInnerClassName), arg))
                .map(k -> (Expression) k).toList();
    }

    /**
     * Converts given class names of given lumo inner class name lists and return
     * all.
     *
     * @param lumoInnerClassNames
     *            List of lumo inner class names e.g. Gap, Gap.Row, Gap.Column
     * @param classNames
     *            class names to look up in given lumo classes
     * @return List of expression to add to a method as arguments
     */
    public static List<Expression> getLumoMethodArgExpressions(List<String> lumoInnerClassNames,
            List<String> classNames) {
        List<Expression> expressions = new ArrayList<>();
        for (String lumoInnerClassName : lumoInnerClassNames) {
            List<Expression> lumoMethodArgExpressions = getLumoMethodArgExpressions(lumoInnerClassName, classNames);
            expressions.addAll(lumoMethodArgExpressions);
        }
        return expressions;
    }

    /**
     * Using reflection to get variables from given Lumo class Key -> field name,
     * value -> field value.
     *
     * @return bidirectional map of class variables.
     */
    private static BiMap<String, String> getLumoFieldsNameValueMap(String innerClassName) {
        try {
            BiMap<String, String> biMap = HashBiMap.create();
            Class<?> lumoUtilityClazz = JavaReflectionUtil
                    .getClass("com.vaadin.flow.theme.lumo.LumoUtility$" + innerClassName);
            for (Field declaredField : lumoUtilityClazz.getDeclaredFields()) {
                String name = declaredField.getName();
                String value = (String) declaredField.get(null);
                biMap.put(name, value);
            }
            return biMap;
        } catch (IllegalAccessException e) {
            throw new IllegalArgumentException("There is no lumo utility field named " + innerClassName);
        }
    }

    private static List<MethodCallExpr> getThemeAddCalls(List<MethodCallExpr> methodCallStatements) {
        return methodCallStatements.stream().filter(methodCallExpr -> methodCallExpr.getScope().isPresent())
                .filter(methodCallExpr -> methodCallExpr.getScope().get().toString().contains("getThemeList"))
                .filter(methodCallExpr -> methodCallExpr.getNameAsString().equals("add")
                        || methodCallExpr.getNameAsString().equals("addAll"))
                .collect(Collectors.toList());
    }

    private static String getLumoVariable(String className) {
        if (classNameLumoUtilityClassCacheMap.containsKey(className)) {
            return classNameLumoUtilityClassCacheMap.get(className);
        }
        Class<?> lumoUtilityClass = JavaReflectionUtil.getClass("com.vaadin.flow.theme.lumo.LumoUtility");
        Class<?>[] classes = lumoUtilityClass.getClasses();
        for (Class<?> clazz : classes) {
            Field[] fields = clazz.getFields();
            for (Field field : fields) {
                String name = field.getName();
                try {
                    Object classVarValue = field.get(null);
                    if (classVarValue instanceof String && classVarValue.equals(className)) {
                        String lumoVariableName = lumoUtilityClass.getSimpleName() + "." + clazz.getSimpleName() + "."
                                + name;
                        classNameLumoUtilityClassCacheMap.put(className, lumoVariableName);
                        return lumoVariableName;
                    }
                } catch (IllegalAccessException e) {
                    getLogger().debug("Could not access LumoUtility variable {}", name, e);
                    return null;
                }
            }
        }
        return null;
    }

    /**
     * Check if @StyleSheet("styles.css") exists on main app class -> yes,
     * put @import "lumo/utility.css" at the top Check
     * if @StyleSheet(Lumo.STYLESHEET) exists on main app -> yes,
     * put @StyleSheet(Lumo.UTILITY_STYLESHEET) below it Otherwise Lumo is loaded in
     * custom way, so just create styles.css add it to main app class
     * using @StyleSheet("styles.css") and add @import "lumo/utility.css" inside.
     * Same as ThemeEditor is doing.
     *
     * @param vaadinServletContext
     *            Context to find the main class of Spring Applications
     * @throws IOException
     *             is thrown when file operation fails
     */
    public static void addLumoUtilityStylesheetIfNotExists(VaadinServletContext vaadinServletContext)
            throws IOException {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        File stylesCss = projectFileManager.getStylesCss();
        if (checkLumoUtilityStyleSheetAnnotation(vaadinServletContext)) {
            return;
        }

        if (Files.exists(stylesCss.toPath()) && checkStyleSheetAnnotation(vaadinServletContext)) {
            updateStyleSheetFile(stylesCss);
            return;
        }
        if (checkLumoStyleSheetAnnotation(vaadinServletContext)) {
            addLumoUtilityAnnotation(vaadinServletContext);
            return;
        }
        // ensuring styles.css file and @Stylesheet("styles.css") exist
        boolean fileAndAnnotationAdded = addStylesAnnotationAndCreateStylesCssFileIfNotExists(vaadinServletContext);
        if (!fileAndAnnotationAdded) {
            throw new CopilotException("Could not add styles css file");
        }
        // updating the file
        updateStyleSheetFile(stylesCss);
    }

    private static void updateStyleSheetFile(File stylesCss) {
        CascadingStyleSheet styleSheet = CSSReader.readFromFile(stylesCss, StandardCharsets.UTF_8);
        if (checkUtilityImportStatementPresentInStylesCss(styleSheet)) {
            return;
        }
        addUtilityImportStatementIntoStylesCss(ProjectFileManager.get(), stylesCss);
    }

    private static boolean checkUtilityImportStatementPresentInStylesCss(CascadingStyleSheet styleSheet) {
        if (styleSheet == null) {
            return false;
        }
        CSSImportRule utilityImportRule = new CSSImportRule(LUMO_UTILITY_CSS_FILE);
        ICommonsList<CSSImportRule> allImportRules = styleSheet.getAllImportRules();
        for (CSSImportRule cssImportRule : allImportRules) {
            if (cssImportRule.equals(utilityImportRule)) {
                return true;
            }
        }
        return false;
    }

    private static void addUtilityImportStatementIntoStylesCss(ProjectFileManager fileManager, File styleStyleSheet) {
        try {
            List<String> lines = fileManager.readLines(styleStyleSheet);
            lines.addFirst("@import \"" + LUMO_UTILITY_CSS_FILE + "\";");
            String content = lines.stream().collect(Collectors.joining(System.lineSeparator()));
            fileManager.writeFile(styleStyleSheet, COPILOT_STYLES_MODIFY_UNDO_LABEL, content);
        } catch (IOException e) {
            throw new CopilotException("Unable to add Lumo Utility stylesheet", e);
        }

    }

    private static boolean checkLumoUtilityStyleSheetAnnotation(VaadinServletContext context) {
        boolean springAvailable = SpringBridge.isSpringAvailable(context);
        if (!springAvailable) {
            return false;
        }
        Class<?> applicationClass = SpringBridge.getApplicationClass(context);
        List<String> styleSheetAnnotationValues = JavaReflectionUtil.getStyleSheetAnnotationValues(applicationClass);
        return styleSheetAnnotationValues.contains(LUMO_UTILITY_CSS_FILE);
    }

    private static boolean checkLumoStyleSheetAnnotation(VaadinServletContext context) {
        boolean springAvailable = SpringBridge.isSpringAvailable(context);
        if (!springAvailable) {
            return false;
        }
        Class<?> applicationClass = SpringBridge.getApplicationClass(context);
        List<String> styleSheetAnnotationValues = JavaReflectionUtil.getStyleSheetAnnotationValues(applicationClass);
        return styleSheetAnnotationValues.contains("lumo/lumo.css");
    }

    private static void addLumoUtilityAnnotation(VaadinServletContext context) throws IOException {
        ApplicationClassInfo applicationClassDeclaration = getApplicationClassDeclaration(context);
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = applicationClassDeclaration
                .classOrInterfaceDeclaration();

        classOrInterfaceDeclaration.addAnnotation(new SingleMemberAnnotationExpr(new Name("StyleSheet"),
                new FieldAccessExpr(new NameExpr("Lumo"), "UTILITY_STYLESHEET")));
        ProjectFileManager.get().writeFile(applicationClassDeclaration.file(), COPILOT_STYLES_MODIFY_UNDO_LABEL,
                applicationClassDeclaration.compilationUnit.toString());
    }

    private static boolean checkStyleSheetAnnotation(VaadinServletContext context) {
        boolean springAvailable = SpringBridge.isSpringAvailable(context);
        if (!springAvailable) {
            return false;
        }
        Class<?> applicationClass = SpringBridge.getApplicationClass(context);
        List<String> styleSheetAnnotationValues = JavaReflectionUtil.getStyleSheetAnnotationValues(applicationClass);
        return styleSheetAnnotationValues.contains("styles.css");
    }

    private static boolean addStylesAnnotationAndCreateStylesCssFileIfNotExists(VaadinServletContext context)
            throws IOException {
        boolean springAvailable = SpringBridge.isSpringAvailable(context);
        if (!springAvailable) {
            return false;
        }
        ApplicationClassInfo applicationClassDeclaration = getApplicationClassDeclaration(context);
        ClassOrInterfaceDeclaration classOrInterfaceDeclaration = applicationClassDeclaration
                .classOrInterfaceDeclaration();
        classOrInterfaceDeclaration.addAnnotation(
                new SingleMemberAnnotationExpr(new Name("StyleSheet"), new StringLiteralExpr("styles.css")));
        File stylesCss = ProjectFileManager.get().getStylesCss();
        if (stylesCss == null || !stylesCss.exists()) {
            ProjectFileManager.get().writeFile(stylesCss, COPILOT_STYLES_MODIFY_UNDO_LABEL, "");
        }
        applicationClassDeclaration.compilationUnit.addImport("com.vaadin.flow.component.dependency.StyleSheet");
        ProjectFileManager.get().writeFile(applicationClassDeclaration.file, COPILOT_STYLES_MODIFY_UNDO_LABEL,
                applicationClassDeclaration.compilationUnit.toString());
        return true;
    }

    private static ApplicationClassInfo getApplicationClassDeclaration(VaadinServletContext context)
            throws IOException {
        boolean springAvailable = SpringBridge.isSpringAvailable(context);
        if (!springAvailable) {
            throw new CopilotException("Unable to add Lumo Utility annotation");
        }
        Class<?> applicationClass = SpringBridge.getApplicationClass(context);
        File fileForClass = ProjectFileManager.get().getFileForClass(applicationClass);
        if (fileForClass == null || !fileForClass.exists()) {
            throw new CopilotException("Unable to read file for class " + applicationClass);
        }
        JavaSource javaSource = new JavaSource(fileForClass, ProjectFileManager.get().readFile(fileForClass));
        CompilationUnit compilationUnit = javaSource.getCompilationUnit().addImport("com.vaadin.flow.theme.lumo.Lumo");
        Optional<ClassOrInterfaceDeclaration> classByName = compilationUnit
                .getClassByName(applicationClass.getSimpleName());
        if (classByName.isEmpty()) {
            throw new CopilotException("Unable to get class by name " + applicationClass.getSimpleName());
        }
        return new ApplicationClassInfo(compilationUnit, classByName.get(), fileForClass);
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(LumoRewriterUtil.class);
    }

    private record ApplicationClassInfo(CompilationUnit compilationUnit,
            ClassOrInterfaceDeclaration classOrInterfaceDeclaration, File file) {
    }
}
