package com.vaadin.copilot;

import jakarta.annotation.security.DenyAll;
import jakarta.annotation.security.PermitAll;
import jakarta.annotation.security.RolesAllowed;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.net.URI;
import java.nio.file.FileSystem;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.vaadin.copilot.ai.UIFromImage;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.javarewriter.JavaSource;
import com.vaadin.flow.server.auth.AnonymousAllowed;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.expr.ArrayInitializerExpr;
import com.github.javaparser.ast.expr.MarkerAnnotationExpr;
import com.github.javaparser.ast.expr.MemberValuePair;
import com.github.javaparser.ast.expr.Name;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.NormalAnnotationExpr;
import com.github.javaparser.ast.expr.SingleMemberAnnotationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles listing, supplying templates of new views that will be created by New
 * Route Panel.
 */
public class NewRouteTemplateHandler {
    private static final String PRETTIER_IGNORE_REGEX = "//\\s*prettier-ignore";
    private static final String HILLA = "hilla";
    private static final String FLOW = "flow";

    private static final String TEMPLATE_VIEWS_FOLDER_NAME = "com/vaadin/copilot/sampleviews/";
    private static final Map<String, List<TemplateView>> views = new HashMap<>();
    private static boolean loaded = false;
    private static final String MENU_ANNOTATION_IMPORT = "com.vaadin.flow.router.Menu";

    static {
        views.put(FLOW, new ArrayList<>());
        views.put(HILLA, new ArrayList<>());
    }

    private void loadTemplates() {
        if (loaded) {
            return;
        }
        URI uri = null;
        try {
            uri = getClass().getClassLoader().getResource(TEMPLATE_VIEWS_FOLDER_NAME).toURI();
            // runs when resources are in file system
            Path path = Paths.get(uri);
            loadTemplateInPath(path);
        } catch (FileSystemNotFoundException ignored) {
            // runs when resources are bundled in a JAR
            loadTemplatesLocatedInJarBundle(uri);
        } catch (Exception e) {
            throw new CopilotException("Could not load sample views", e);
        }
    }

    private void loadTemplatesLocatedInJarBundle(URI uri) {
        try (FileSystem fileSystem = FileSystems.newFileSystem(uri, new HashMap<>())) {
            Path path = fileSystem.getPath(TEMPLATE_VIEWS_FOLDER_NAME);
            loadTemplateInPath(path);
        } catch (IOException e) {
            throw new CopilotException("Could not load sample views", e);
        }
    }

    private void loadTemplateInPath(Path path) throws IOException {
        try (Stream<Path> stream = Files.walk(path)) {
            stream.filter(Files::isRegularFile).forEach(viewFilePath -> {
                String viewFile = viewFilePath.toString();
                String label = Util.getBaseName(viewFile);
                String framework = null;
                if (viewFile.endsWith(".java")) {
                    framework = FLOW;
                } else if (viewFile.endsWith("tsx")) {
                    framework = HILLA;
                }
                try {
                    views.get(framework).add(new TemplateView(label, viewFile, Files.readString(viewFilePath)));
                } catch (IOException e) {
                    throw new CopilotException("Could not read file content of " + viewFile, e);
                }

            });
        }
    }

    /**
     * Creates a view template string that will have view configuration based on
     * given parameters and invoke the template handler with it. It returns the
     * empty template when template name is blank or equals to <code>EMPTY</code> or
     * null.
     *
     * @param flowTemplateRequest
     *            Required parameters to create flow template
     * @return a CompletableFuture with the template string
     */
    public CompletableFuture<String> getFlowTemplate(FlowTemplateRequest flowTemplateRequest) {
        ensureTemplatesLoaded();
        AccessRequirement accessRequirement = flowTemplateRequest.accessRequirement();
        Class<?> layoutClass = flowTemplateRequest.layoutClass();
        String route = flowTemplateRequest.route();
        String className = flowTemplateRequest.className();
        String title = flowTemplateRequest.title();
        String packageName = flowTemplateRequest.packageName();
        boolean addToMenu = flowTemplateRequest.addToMenu();

        Class<? extends Annotation> accessClass = switch (accessRequirement.getType()) {
        case PERMIT_ALL -> PermitAll.class;
        case ANONYMOUS_ALLOWED -> AnonymousAllowed.class;
        case DENY_ALL -> DenyAll.class;
        case ROLES_ALLOWED -> RolesAllowed.class;
        default -> throw new IllegalArgumentException("Unknown access requirement: " + accessRequirement.getType());
        };

        CompletableFuture<String> future = new CompletableFuture<>();
        if (flowTemplateRequest.templateName().equals(RouteCreator.VIEW_FROM_IMAGE_TEMPLATE_NAME)) {
            String imageData = getBase64DataFromDataUrl(flowTemplateRequest.templateData());
            String mimeType = getMimeTypeFromDataUrl(flowTemplateRequest.templateData());
            UIFromImage.convertImageToCode(imageData, mimeType, true, flowCode -> {
                future.complete(setDynamicFlowParts(flowCode, packageName, className, accessClass, layoutClass, title,
                        accessRequirement, route, addToMenu));
            }, future::completeExceptionally);
        } else {
            String template = views.get(FLOW).stream().filter(f -> f.label().equals(flowTemplateRequest.templateName()))
                    .map(k -> k.content).findFirst().orElseThrow(() -> new CopilotException("Unknown file template"));

            future.complete(setDynamicFlowParts(template, packageName, className, accessClass, layoutClass, title,
                    accessRequirement, route, addToMenu));

        }
        return future;
    }

    private String getMimeTypeFromDataUrl(String dataUrl) {
        // data:image/webp;base64,UklGRgyTAQBXRUJQVlA4IACTAQBwkQWdASoABAAEPm00lUgkIqSlppZJ0LANiWVtocPf72nGZ
        String[] parts = dataUrl.split(",", 2);
        if (parts.length != 2 || !parts[0].startsWith("data:") || !parts[0].contains(";base64")) {
            throw new CopilotException("Invalid data URL: " + Util.truncate(dataUrl, 30));
        }
        return parts[0].substring("data:".length(), parts[0].indexOf(";base64"));
    }

    private String getBase64DataFromDataUrl(String dataUrl) {
        // data:image/webp;base64,UklGRgyTAQBXRUJQVlA4IACTAQBwkQWdASoABAAEPm00lUgkIqSlppZJ0LANiWVtocPf72nGZ
        String[] parts = dataUrl.split(",", 2);
        if (parts.length != 2 || !parts[0].startsWith("data:") || !parts[0].contains(";base64")) {
            throw new CopilotException("Invalid data URL: " + Util.truncate(dataUrl, 30));
        }
        return parts[1];
    }

    private static String setDynamicFlowParts(String template, String packageName, String className,
            Class<? extends Annotation> accessClass, Class<?> layoutClass, String title,
            AccessRequirement accessRequirement, String route, boolean addToMenu) {

        template = replaceTitle(template, title);

        CompilationUnit compilationUnit = new JavaSource(null, template, true).getCompilationUnit();

        compilationUnit.setPackageDeclaration(packageName);

        Optional<ClassOrInterfaceDeclaration> classByName = compilationUnit.getClassByName(className);
        Optional<ClassOrInterfaceDeclaration> firstClass = compilationUnit.findFirst(ClassOrInterfaceDeclaration.class);
        ClassOrInterfaceDeclaration classDeclaration = classByName.orElse(
                firstClass.orElseThrow(() -> new CopilotException("Could not find any class name in template")));

        classDeclaration.setName(className);
        classDeclaration.getConstructors().forEach(constructor -> constructor.setName(className));

        classDeclaration.getAnnotations().removeIf(annotationExpr -> annotationExpr.getNameAsString().equals("Route"));
        // imports
        compilationUnit.addImport(accessClass);
        if (layoutClass != null) {
            compilationUnit.addImport(layoutClass);
        }
        // page title
        compilationUnit.addImport("com.vaadin.flow.router.PageTitle");
        classDeclaration
                .addAnnotation(new SingleMemberAnnotationExpr(new Name("PageTitle"), new StringLiteralExpr(title)));
        // access control
        compilationUnit.addImport(accessClass);
        if (RolesAllowed.class.equals(accessClass)) {
            ArrayInitializerExpr arrayInitializerExpr = new ArrayInitializerExpr();
            Arrays.stream(accessRequirement.getRoles()).map(StringLiteralExpr::new)
                    .forEach(expr -> arrayInitializerExpr.getValues().add(expr));
            classDeclaration
                    .addAnnotation(new SingleMemberAnnotationExpr(new Name("RolesAllowed"), arrayInitializerExpr));
        } else {
            classDeclaration.addAnnotation(new MarkerAnnotationExpr(new Name(accessClass.getSimpleName())));
        }
        // route
        compilationUnit.addImport("com.vaadin.flow.router.Route");
        NormalAnnotationExpr routeAnnotationExpr = new NormalAnnotationExpr();
        routeAnnotationExpr.setName("Route");
        routeAnnotationExpr.getPairs().add(new MemberValuePair("value", new StringLiteralExpr(route)));
        if (layoutClass != null) {
            compilationUnit.addImport(layoutClass);
            routeAnnotationExpr.getPairs()
                    .add(new MemberValuePair("layout", new NameExpr(layoutClass.getSimpleName() + ".class")));
        }
        classDeclaration.addAnnotation(routeAnnotationExpr);

        if (addToMenu) {
            compilationUnit.addImport(MENU_ANNOTATION_IMPORT);
            NormalAnnotationExpr addToMenuExpr = new NormalAnnotationExpr();
            addToMenuExpr.setName("Menu");
            addToMenuExpr.getPairs().add(new MemberValuePair("title", new StringLiteralExpr(title)));
            classDeclaration.addAnnotation(addToMenuExpr);
        }

        return compilationUnit.toString();
    }

    private static String replaceTitle(String template, String title) {
        template = template.replace("TITLE_FOR_QUOTED_STRING", title.replace("\"", "\\\""));
        template = template.replace("TITLE_FOR_SINGLE_QUOTED_STRING", title.replace("'", "\\'"));
        template = template.replace("TITLE", title);
        return template;
    }

    /**
     * Creates a view template string that will have view configuration based on
     * given parameters and invoke the template handler with it. It returns the
     * empty template when template name is blank or equals to <code>EMPTY</code> or
     * null.
     *
     * @param hillaTemplateRequest
     *            Required parameters to create hilla template
     */
    public CompletableFuture<String> getHillaTemplate(HillaTemplateRequest hillaTemplateRequest) {
        ensureTemplatesLoaded();
        String routeName = hillaTemplateRequest.routeName();
        AccessRequirement accessRequirement = hillaTemplateRequest.accessRequirement();
        String title = hillaTemplateRequest.title();
        String functionName = getValidReactComponentFunctionName(
                SharedUtil.capitalize(SharedUtil.dashSeparatedToCamelCase(routeName)));
        String accessControl = switch (accessRequirement.getType()) {
        case PERMIT_ALL -> "loginRequired: true,";
        case ANONYMOUS_ALLOWED -> "";
        case DENY_ALL -> "rolesAllowed: [],";
        case ROLES_ALLOWED -> "rolesAllowed: [" + Arrays.stream(accessRequirement.getRoles())
                .map(role -> "\"" + role + "\"").collect(Collectors.joining(", ")) + "],";
        default -> throw new CopilotException("Unknown access requirement: " + accessRequirement.getType());
        };
        String menu = hillaTemplateRequest.addToMenu() ? "" : "menu: { exclude: true }";
        String accessControlAndMenu = menu + accessControl;
        CompletableFuture<String> future = new CompletableFuture<String>();
        if (hillaTemplateRequest.templateName().equals(RouteCreator.VIEW_FROM_IMAGE_TEMPLATE_NAME)) {
            String imageBase64Data = getBase64DataFromDataUrl(hillaTemplateRequest.templateData());
            String imageMimeType = getMimeTypeFromDataUrl(hillaTemplateRequest.templateData());
            UIFromImage.convertImageToCode(imageBase64Data, imageMimeType, false, future::complete,
                    future::completeExceptionally);
        } else {
            String template = views.get(HILLA).stream()
                    .filter(f -> f.label().equals(hillaTemplateRequest.templateName())).map(k -> k.content).findFirst()
                    .orElseThrow(() -> new CopilotException("Unknown file template"));
            template = replaceTitle(template, title);
            template = template.replace("FUNCTION_NAME", functionName).replace("ROUTE_NAME", routeName)
                    .replace("ACCESS_CONTROL_AND_MENU", accessControlAndMenu).replaceAll(PRETTIER_IGNORE_REGEX, "");
            future.complete(template);
        }
        return future;
    }

    private String getValidReactComponentFunctionName(String functionName) {
        if (functionName.isEmpty()) {
            return "Index";
        }
        if (functionName.equals("@index")) {
            return "Index";
        }
        if (functionName.matches("^[^A-Z].*")) {
            // React component names must start with a capital letter
            functionName = "A" + functionName;
        }

        return functionName.replaceAll("[^a-zA-Z0-9]", "");
    }

    /**
     * It returns the template list for the given framework. Label is the base file
     * name, value is the file name.
     *
     * @param framework
     *            Either "hilla" or "flow"
     * @return list of summary templates, which model has label and value properties
     */
    public List<String> getTemplateNames(String framework) {
        ensureTemplatesLoaded();
        return views.get(framework).stream().map(TemplateView::label).toList();
    }

    private synchronized void ensureTemplatesLoaded() {
        if (!loaded) {
            loadTemplates();
        }
        setLoadedTrue();
    }

    private static synchronized void setLoadedTrue() {
        NewRouteTemplateHandler.loaded = true;
    }

    /**
     * Contains information to generate new Flow view
     *
     * @param packageName
     *            Package name where file will be located. e.g.
     *            <code>com.vaadin.project.views</code>
     * @param route
     *            Route url of the view
     * @param title
     *            Title of the page which will be used for <code>@PageTitle</code>
     * @param className
     *            Java class name of the file. e.g. HelloWorldView for
     *            HelloWorldView.java
     * @param accessRequirement
     *            Access requirement parameters that are entered from UI
     * @param layoutClass
     *            Layout class that view is extended from. Might be null.
     * @param templateName
     *            Template file name. e.g. HelloWorldView.java
     * @param templateData
     *            data for the selected template, or {@code null}
     * @param addToMenu
     *            whether to add the view to an automatically generated menu
     */
    public record FlowTemplateRequest(String packageName, String route, String title, String className,
            AccessRequirement accessRequirement, Class<?> layoutClass, String templateName, String templateData,
            boolean addToMenu) {
    }

    /**
     * Contains information to generate new Flow view
     *
     * @param routeName
     *            Route of the file
     * @param title
     *            Title of the view. It will be used in viewConfig
     * @param accessRequirement
     *            Access requirement parameters that are entered from UI
     * @param templateName
     *            Template file name. e.g. HelloWorldView.tsx
     * @param templateData
     *            data for the selected template, or {@code null}
     * @param addToMenu
     *            whether to add the view to an automatically generated menu
     */
    public record HillaTemplateRequest(String routeName, String title, AccessRequirement accessRequirement,
            String templateName, String templateData, boolean addToMenu) {
    }

    /**
     * Contains information about Templates where information for listing templates
     * in UI and creation of the new template are present.
     *
     * @param label
     * @param fileAbsolutePath
     *            File absolute path in the class path.
     * @param content
     *            Content of the file, which is the source
     */
    private record TemplateView(String label, String fileAbsolutePath, String content) {
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(NewRouteTemplateHandler.class);
    }

}
