package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.theme.ApplicationDetectedThemeHolder;
import com.vaadin.copilot.theme.ApplicationTheme;

import tools.jackson.databind.JsonNode;

/**
 * Handles application theme-related commands received from the developer tools
 * interface.
 * <p>
 * This handler listens for the {@code "set-app-theme"} command and updates the
 * {@link ApplicationDetectedThemeHolder} with the new theme value provided in
 * the command data.
 * </p>
 *
 * <p>
 * Example usage:
 *
 * <pre>
 * {
 *   "command": "set-app-theme",
 *   "data": { "theme": "aura" }
 * }
 * </pre>
 * </p>
 *
 * <p>
 * The handler expects the JSON data to contain a {@code "theme"} field whose
 * value corresponds to one of the supported {@link ApplicationTheme} constants.
 * </p>
 *
 * @see ApplicationDetectedThemeHolder
 * @see ApplicationTheme
 */
public class ApplicationThemeHandler extends CopilotCommand {
    private static final String THEME_PROP_KEY = "theme";

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("set-app-theme")) {
            if (data.has(THEME_PROP_KEY)) {
                if (data.get(THEME_PROP_KEY).isNull()) {
                    ApplicationDetectedThemeHolder.getInstance().setTheme(ApplicationTheme.NONE);
                } else {
                    String text = data.get(THEME_PROP_KEY).asString();
                    ApplicationTheme applicationTheme = ApplicationTheme.fromString(text);
                    ApplicationDetectedThemeHolder.getInstance().setTheme(applicationTheme);
                }

            }
            return true;
        }
        return false;
    }
}
