package com.vaadin.copilot.plugins.info;

import java.io.File;
import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.jar.JarFile;
import java.util.stream.Collectors;

import com.vaadin.flow.server.frontend.FrontendVersion;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public record JdkInfo(boolean jrebel, boolean extendedClassDefCapable, boolean runningWithExtendClassDef,
        boolean hotswapAgentFound, String hotswapAgentLocation, boolean runningWitHotswap,
        boolean runningInJavaDebugMode, String hotswapVersion, boolean hotswapVersionOk) {

    /**
     * Available Hotswap solutions.
     */
    public enum HotswapSolution {
        JREBEL, HOTSWAP_AGENT, NONE
    }

    public static JdkInfo get() {
        RuntimeMXBean runtimeMXBean = ManagementFactory.getRuntimeMXBean();
        List<String> inputArguments = runtimeMXBean.getInputArguments();

        boolean jrebel = false;
        try {
            Class.forName("org.zeroturnaround.javarebel.ReloaderFactory");
            jrebel = true;
        } catch (Exception e) {
            getLogger().trace("JRebel not found", e);
        }
        boolean extendedClassDefCapable = runtimeMXBean.getVmVendor().contains("JetBrains");
        boolean runningWithExtendClassDef = inputArguments.contains("-XX:+AllowEnhancedClassRedefinition");
        boolean runningInJavaDebugMode = inputArguments.toString().toLowerCase(Locale.ENGLISH).contains("jdwp");

        List<String> hotswapJavaAgent = inputArguments.stream().filter(arg -> arg.startsWith("-javaagent:"))
                .filter(arg -> arg.endsWith("hotswap-agent.jar")).collect(Collectors.toList());

        File javaHome = new File(System.getProperty("java.home"));
        File hotswapAgentLocation = null;
        if (inputArguments.contains("-XX:HotswapAgent=fatjar")) {
            hotswapAgentLocation = new File(new File(new File(javaHome, "lib"), "hotswap"), "hotswap-agent.jar");
        } else if (!hotswapJavaAgent.isEmpty()) {
            hotswapAgentLocation = new File(hotswapJavaAgent.get(0).substring("-javaagent:".length()));
        }

        boolean runningWithHotswap = false;
        boolean hotswapAgentFound = false;
        String hotswapVersionString = null;
        String hotswapAgentJarPath = null;
        if (hotswapAgentLocation != null) {
            runningWithHotswap = true;
            hotswapAgentFound = hotswapAgentLocation.exists();
            hotswapVersionString = getHotswapAgentVersionString(hotswapAgentLocation).orElse(null);
            hotswapAgentJarPath = hotswapAgentLocation.getAbsolutePath();
        }

        boolean hotswapVersionOk = checkHotswapAgentVersion(hotswapVersionString);
        return new JdkInfo(jrebel, extendedClassDefCapable, runningWithExtendClassDef, hotswapAgentFound,
                hotswapAgentJarPath, runningWithHotswap, runningInJavaDebugMode, hotswapVersionString,
                hotswapVersionOk);
    }

    private static Optional<String> getHotswapAgentVersionString(File hotswapAgentLocation) {
        if (hotswapAgentLocation.exists()) {
            try (JarFile jarFile = new JarFile(hotswapAgentLocation)) {
                return Optional
                        .ofNullable(jarFile.getManifest().getMainAttributes().getValue("Implementation-Version"));
            } catch (IOException e) {
                getLogger().error("No META-INF/MANIFEST.MF found in {}", hotswapAgentLocation, e);
            }
        }

        return Optional.empty();
    }

    private static boolean checkHotswapAgentVersion(String hotswapVersionString) {
        return getHotswapAgentVersion(hotswapVersionString)
                .map(frontendVersion -> frontendVersion.isEqualOrNewer(new FrontendVersion(2, 0, 0))).orElse(false);
    }

    private static Optional<FrontendVersion> getHotswapAgentVersion(String versionString) {
        if (versionString == null) {
            return Optional.empty();
        }
        String withoutSnapshot = versionString.replace("-SNAPSHOT", "");
        return Optional.of(new FrontendVersion(withoutSnapshot));
    }

    /**
     * Returns the hotswap solution that is currently in use.
     * 
     * @return the hotswap solution that is currently in use
     */
    public HotswapSolution runningWith() {
        if (jrebel) {
            return HotswapSolution.JREBEL;
        }
        if (runningWithExtendClassDef && runningWitHotswap) {
            return HotswapSolution.HOTSWAP_AGENT;
        }
        return HotswapSolution.NONE;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JdkInfo.class);
    }
}
