package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.Map;

import com.vaadin.copilot.Copilot;

import com.github.javaparser.ParserConfiguration;
import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.printer.lexicalpreservation.LexicalPreservingPrinter;
import com.github.javaparser.resolution.declarations.ResolvedReferenceTypeDeclaration;
import com.github.javaparser.resolution.model.SymbolReference;
import com.github.javaparser.symbolsolver.JavaSymbolSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.ClassLoaderTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.CombinedTypeSolver;
import com.github.javaparser.symbolsolver.resolution.typesolvers.ReflectionTypeSolver;

/**
 * Represents a Java source file that can be parsed and rewritten using a custom
 * observer and parser configuration. Provides functionality to parse the given
 * Java source code.
 */
public class JavaSource {
    private final String source;
    private final boolean javaParserLexicalPreservation;
    private final JavaRewriterObserver observer = new JavaRewriterObserver();
    private final ParserConfiguration parserConfiguration = new ParserConfiguration();
    private final File file;
    private CompilationUnit compilationUnit;
    private boolean written = false;

    /**
     * Constructs a {@code JavaSource} with the given Java source code. The source
     * is parsed without attempting to resolve JDK classes by their simple names.
     *
     * @param file
     *            the file containing the Java source code
     * @param source
     *            the raw Java source code as a String
     */
    public JavaSource(File file, String source) {
        this(file, source, false);
    }

    /**
     * Constructs a {@code JavaSource} with the given Java source code. Allows
     * specifying whether JDK classes should be resolved by their simple names.
     *
     * @param file
     *            the file containing the Java source code
     * @param source
     *            the raw Java source code as a String
     * @param solveJdkClassesBySimpleName
     *            {@code true} to resolve JDK classes by simple name, {@code false}
     *            otherwise
     */
    public JavaSource(File file, String source, boolean solveJdkClassesBySimpleName) {
        this(file, source, solveJdkClassesBySimpleName, false);
    }

    /**
     * Constructs a {@code JavaSource} with the given Java source code. Allows
     * specifying whether JDK classes should be resolved by their simple names.
     *
     * @param file
     *            the file containing the Java source code
     * @param source
     *            the raw Java source code as a String
     * @param solveJdkClassesBySimpleName
     *            {@code true} to resolve JDK classes by simple name, {@code false}
     *            otherwise
     * @param javaParserLexicalPreservation
     *            true to use the JavaParser feature to preserve indentation, false
     *            (default) to use the faster but sometimes broken version in
     *            Copilot
     */
    public JavaSource(File file, String source, boolean solveJdkClassesBySimpleName,
            boolean javaParserLexicalPreservation) {
        this.file = file;
        this.source = source;
        this.javaParserLexicalPreservation = javaParserLexicalPreservation;
        parseSource(source, solveJdkClassesBySimpleName);
    }

    private void parseSource(String source, boolean solveJdkClassesBySimpleName) {
        parserConfiguration.setLanguageLevel(Copilot.LANGUAGE_LEVEL);
        CombinedTypeSolver combinedTypeSolver = new CombinedTypeSolver();
        combinedTypeSolver.add(new ReflectionTypeSolver(false));
        if (solveJdkClassesBySimpleName) {
            combinedTypeSolver.add(new ClassLoaderTypeSolver(ClassLoaderTypeSolver.class.getClassLoader()) {
                @Override
                public SymbolReference<ResolvedReferenceTypeDeclaration> tryToSolveType(String name) {
                    if (name.contains(".")) {
                        return SymbolReference.unsolved();
                    }

                    // java.lang is already handled by the default solver
                    for (String pkg : Arrays.asList(LocalDateTime.class.getPackageName(), Map.class.getPackageName())) {
                        SymbolReference<ResolvedReferenceTypeDeclaration> ref = super.tryToSolveType(pkg + "." + name);
                        if (ref.isSolved()) {
                            return ref;
                        }
                    }
                    return SymbolReference.unsolved();
                }
            });
        }
        JavaSymbolSolver symbolSolver = new JavaSymbolSolver(combinedTypeSolver);
        parserConfiguration.setSymbolResolver(symbolSolver);

        StaticJavaParser.setConfiguration(parserConfiguration);
        if (javaParserLexicalPreservation) {
            this.compilationUnit = LexicalPreservingPrinter.setup(StaticJavaParser.parse(source));
        } else {
            this.compilationUnit = StaticJavaParser.parse(source);
            this.compilationUnit.registerForSubtree(this.observer);
        }
    }

    /**
     * Parsed compilation unit of the given {@link #source} from constructor.
     *
     * @return Compilation Unit
     */
    public CompilationUnit getCompilationUnit() {
        return compilationUnit;
    }

    /**
     * Checks if there have been any changes based on added, modified, or removed
     * nodes. This method returns true if there are any added or modified nodes, or
     * if there are any removed ranges.
     *
     * @return true if there are changes (i.e., added/modified nodes or removed
     *         ranges), false if there are no changes.
     */
    public boolean isChanged() {
        return !observer.getAddedOrModifiedNodes().isEmpty() || !observer.getRemovedRanges().isEmpty();
    }

    /**
     * Updates the flag with the given written value.
     *
     * @param written
     *            {@code true} if it is saved, {@code false} otherwise.
     */
    public void setWritten(boolean written) {
        this.written = written;
    }

    /**
     * Checks if the file is written into disk.
     *
     * @return true if it is written, false otherwise.
     */
    public boolean isWritten() {
        return written;
    }

    /**
     * Applies changes to the given source and returns the result file content
     *
     * @return Result file content after changes are applied
     */
    public String getResult() {
        String result;
        if (!javaParserLexicalPreservation) {
            result = JavaRewriterMerger.apply(observer.getAddedOrModifiedNodes(), observer.getRemovedRanges(), source);
        } else {
            result = compilationUnit.toString();
        }
        return result.replaceAll(JavaRewriter.DELETE_THIS_MARKER, "");
    }

    /**
     * Finds the first modified row in the given source by comparing result and
     * initial source
     *
     * @return row number, or -1 if no change is found.
     */
    public int getFirstModifiedRow() {
        int row = 1;
        for (int i = 0; i < source.length(); i++) {
            if (source.charAt(i) != getResult().charAt(i)) {
                return row;
            }
            if (source.charAt(i) == '\n') {
                row++;
            }
        }
        return -1;
    }

    /**
     * Returns the source code.
     *
     * @return the source code
     */
    public String getSource() {
        return source;
    }

    /**
     * Returns the file associated with this Java source.
     *
     * @return the file containing the Java source code
     */
    public File getFile() {
        return file;
    }
}
