package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.plugins.info.CopilotInfo;
import com.vaadin.copilot.plugins.info.JdkInfo;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.internal.hilla.EndpointRequestUtil;
import com.vaadin.flow.router.Layout;
import com.vaadin.flow.router.RouteData;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import elemental.json.Json;
import elemental.json.JsonObject;
import org.jetbrains.annotations.NotNull;

public class ProjectInfoHandler extends CopilotCommand {

    @Override
    public void handleConnect(DevToolsInterface devToolsInterface) {
        ObjectNode data = JacksonUtils.createObjectNode();
        data.put("supportsHilla", EndpointRequestUtil.isHillaAvailable());
        boolean springAvailable = SpringBridge.isSpringAvailable(getVaadinContext());
        data.put("springApplication", springAvailable);

        if (springAvailable) {
            data.put("springSecurityEnabled", SpringBridge.isSpringSecurityEnabled(getVaadinContext()));
            data.put("urlPrefix", SpringBridge.getUrlPrefix(getVaadinContext()));
            data.put("springJpaDataEnabled", SpringBridge.isSpringDataJpaAvailable(getVaadinContext()));
        }
        CopilotInfo info = new CopilotInfo(ProjectFileManager.get().getApplicationConfiguration());
        data.putPOJO("serverVersions", info.getVersions());
        data.putPOJO("jdkInfo", JdkInfo.get());

        devToolsInterface.send("copilot-early-project-state", data);
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {

        if (command.equals("get-project-info")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());

            if (SpringBridge.isSpringAvailable(getVaadinContext())) {
                List<? extends Class<?>> jpaEntityClasses = SpringBridge.getJpaEntityClasses(getVaadinContext());
                responseData.set("jpaEntityClasses",
                        JacksonUtils.listToJson(jpaEntityClasses.stream().map(Class::getName).toList()));
                Optional<SpringBridge.H2Info> h2Info = SpringBridge.getH2Info(getVaadinContext());
                if (h2Info.isPresent()) {
                    ObjectNode h2InfoJson = JacksonUtils.createObjectNode();
                    h2InfoJson.put("jdbcUrl", h2Info.get().jdbcUrl());
                    h2InfoJson.put("path", h2Info.get().path());
                    responseData.set("h2Info", h2InfoJson);
                }
            }
            devToolsInterface.send(command + "-response", responseData);
            return true;
        } else if (command.equals("add-spring-security")) {
            JsonObject responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, data.get(KEY_REQ_ID).asText());

            try {
                ObjectMapper objectMapper = CopilotJacksonUtils.getObjectMapper();
                Path referenceFile = Util.findCurrentViewFile(getVaadinSession(), data.get("currentView")).orElseThrow()
                        .toPath();
                AccessRequirement accessRequrirement = objectMapper.treeToValue(data.get("viewAccess"),
                        AccessRequirement.class);
                Framework framework = objectMapper.treeToValue(data.get("loginViewType"), Framework.class);
                List<RouteData> serverRoutes = RouteHandler.getServerRoutes(getVaadinSession());
                Set<Class<?>> layouts = RouteHandler.getServerAutoLayouts(getVaadinSession());
                ProjectModifier.addSpringSecurity(getVaadinContext(), referenceFile, accessRequrirement, framework,
                        parseRouteData(serverRoutes), parseLayout(layouts));
            } catch (IOException e) {
                throw new CopilotException("Unable to make project modifications", e);
            }

            devToolsInterface.send(command + "-response", responseData);
            return true;
        }
        return false;
    }

    private List<ProjectModifier.RouteAndPath> parseLayout(Set<Class<?>> layoutClasses) {
        return layoutClasses.stream().map(layoutClass -> {
            File layoutFile = ProjectFileManager.get().getFileForClass(layoutClass);
            return new ProjectModifier.RouteAndPath(layoutFile, layoutClass.getAnnotation(Layout.class).value());
        }).toList();

    }

    @NotNull
    private static List<ProjectModifier.RouteAndPath> parseRouteData(Collection<RouteData> routeDataList) {
        return routeDataList.stream().map(routeData -> {
            File routeFile = ProjectFileManager.get().getFileForClass(routeData.getNavigationTarget());
            return new ProjectModifier.RouteAndPath(routeFile, routeData.getTemplate());
        }).toList();
    }
}
