package com.vaadin.copilot;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vaadin.flow.component.HasComponents;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;

import io.github.classgraph.ClassGraph;
import io.github.classgraph.ClassInfo;
import io.github.classgraph.ClassInfoList;
import io.github.classgraph.ScanResult;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Finds components in the project and relevant dependencies that can be listed
 * as options in Wrap With function. It uses ClassGraph to traverse classes in
 * the project.
 */
public class WrapWithComponentListFinder {
    private static final List<String> excludedClassQualifiedNames = List.of("com.vaadin.flow.component.UI");
    private final Map<String, ClassNameInfo> map = new HashMap<>();
    private final ClassGraph classGraph = new ClassGraph();
    private final VaadinServletContext vaadinServletContext;
    private final List<RouteData> serverRoutes;

    /**
     * Constructs {@link WrapWithComponentListFinder}
     *
     * @param vaadinServletContext
     *            VaadinServletContext to access SpringBridge
     * @param vaadinSession
     *            VaadinSession to access server routes to exclude from the
     *            available list
     */
    public WrapWithComponentListFinder(VaadinServletContext vaadinServletContext, VaadinSession vaadinSession) {
        this.vaadinServletContext = vaadinServletContext;
        this.serverRoutes = RouteHandler.getServerRoutes(vaadinSession);
    }

    /**
     * Finds the available components that has {@link HasComponents} interface in
     * Flow and in the project and returns the sorted list
     *
     * @return the sorted list
     */
    public List<ClassNameInfo> findAndGet() {
        searchComponents();
        return map.values().stream().sorted((a, b) -> StringUtils.compare(a.baseName(), b.baseName())).toList();
    }

    private String getBasePackage() {
        if (!SpringBridge.isSpringAvailable(vaadinServletContext)) {
            return null;
        }
        Class<?> applicationClass = SpringBridge.getApplicationClass(vaadinServletContext);
        return applicationClass.getPackage().getName();
    }

    private void searchComponents() {
        String basePackage = this.getBasePackage();
        String userProjectPackage = basePackage != null ? basePackage : "*";
        ScanResult scanResult = classGraph.enableClassInfo().acceptPackages("com.vaadin.flow", userProjectPackage)
                .scan();
        ClassInfoList classesImplementing = scanResult.getClassesImplementing(HasComponents.class);
        classesImplementing.forEach(this::putClassInfo);
    }

    private void putClassInfo(ClassInfo classInfo) {
        if (classInfo.getName().contains("$")) {
            return;
        }
        if (isRouteClass(classInfo)) {
            return;
        }
        if (classInfo.isInterface()) {
            return;
        }
        if (classInfo.isAbstract()) {
            return;
        }
        if (excludedClassQualifiedNames.stream().anyMatch(excludedClass -> classInfo.getName().equals(excludedClass))) {
            return;
        }
        boolean htmlComponent = classInfo.getPackageName().equals("com.vaadin.flow.component.html");
        boolean createdInUserProject = !classInfo.getPackageName().startsWith("com.vaadin.flow");
        map.put(classInfo.getName(),
                new ClassNameInfo(classInfo.getName(), classInfo.getSimpleName(), createdInUserProject, htmlComponent));
    }

    private boolean isRouteClass(ClassInfo classInfo) {
        return serverRoutes.stream()
                .anyMatch(serverRoute -> serverRoute.getNavigationTarget().getName().equals(classInfo.getName()));
    }

    /***
     * Info about the class name
     * 
     * @param qualifiedClassName
     *            qualified class name e.g. com.flow.component.Html.Div
     * @param baseName
     *            the simple class name e.g. VerticalLayout
     * @param createdInUserProject
     *            represents whether components is created in Flow or inside the
     *            Project
     * @param htmlElement
     *            represents if the given element is a basic HTML element
     */
    public record ClassNameInfo(String qualifiedClassName, String baseName, boolean createdInUserProject,
            boolean htmlElement) {
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(WrapWithComponentListFinder.class);
    }
}
