package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentInfoFinder;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaFileSourceProvider;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaSource;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LitTemplateHandler extends CopilotCommand {
    public static final String LIT_TEMPLATE_FILENAME = "litTemplateFilename";

    private final ComponentSourceFinder sourceFinder;
    private final ProjectFileManager projectFileManager;

    public LitTemplateHandler() {
        sourceFinder = new ComponentSourceFinder(getVaadinSession());
        projectFileManager = ProjectFileManager.get();
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        ObjectNode responseData = JacksonUtils.createObjectNode();
        responseData.put("reqId", data.get("reqId").asText());
        String errorMsg = "Unknown error handling Lit template command";
        try {
            if (command.equals("get-lit-template-name")) {
                errorMsg = "Unable to get Lit template name";
                handleGetLitTemplateName(data, responseData);
                devToolsInterface.send(command + "-resp", responseData);
                return true;
            } else if (command.equals("import-lit-template")) {
                errorMsg = "Unable to convert Lit template to Java";
                handleImportLitTemplate(data);
                devToolsInterface.send(command + "-resp", responseData);
                return true;
            }
        } catch (Exception e) {
            ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                    String.format("%s. Exception message: %s", errorMsg, e.getMessage()), e);
        }
        return false;
    }

    private void handleGetLitTemplateName(JsonNode data, ObjectNode responseData) throws IOException {
        ComponentTypeAndSourceLocation litTemplate = sourceFinder.findTypeAndSourceLocation(data.get("litTemplate"));
        var jsModule = JavaReflectionUtil.findClassWithJsModule(litTemplate.component().getClass())
                .getDeclaredAnnotation(JsModule.class);
        var litTemplateFile = new File(projectFileManager.getFrontendFolder(),
                jsModule.value().replace("/", File.separator));
        responseData.put(LIT_TEMPLATE_FILENAME, litTemplateFile.getCanonicalPath());
    }

    private void handleImportLitTemplate(JsonNode data) throws IOException {
        JsonNode warnings = data.get("warnings");
        if (warnings.get("hasTemplateExpressions").asBoolean(false)) {
            getLogger().warn("Lit template conversion: Template expressions will be lost during conversion.");
        }
        ArrayNode eventBindings = warnings.withArray("eventBindings");
        if (!eventBindings.isEmpty()) {
            List<String> eventBindingNames = new ArrayList<>();
            eventBindings.forEach(name -> eventBindingNames.add(name.asText()));
            getLogger().warn("Lit template conversion: Event bindings will be lost during conversion: {}",
                    eventBindingNames);
        }
        ArrayNode stateProperties = warnings.withArray("statePropertyNames");
        if (!stateProperties.isEmpty()) {
            List<String> statePropertyNames = new ArrayList<>();
            stateProperties.forEach(name -> statePropertyNames.add(name.asText()));
            getLogger().warn("Lit template conversion: State properties will be lost during conversion: {}",
                    statePropertyNames);
        }

        // Get styles from data
        ArrayNode stylesNode = data.withArray("styles");
        List<String> styles = new ArrayList<>();
        stylesNode.forEach(style -> styles.add(style.asText()));

        String litTemplateFileName = data.get(LIT_TEMPLATE_FILENAME).asText();
        JavaFileSourceProvider javaFileSourceProvider = new JavaFileSourceProvider(true);
        ComponentTypeAndSourceLocation litTemplate = sourceFinder.findTypeAndSourceLocation(data.get("litTemplate"));
        ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(javaFileSourceProvider, litTemplate);
        ComponentInfo litTemplateInfo = componentInfoFinder.find();
        var customComponentInfo = litTemplateInfo.customComponentInfo()
                .orElseThrow(() -> new IllegalArgumentException("The given lit template was not found in the project"));

        Class<?> classWithJsModule = JavaReflectionUtil.findClassWithJsModule(customComponentInfo.componentClass());

        // Transform styles for importing from css file
        String cssClassName = LitTemplateCssUtil.toCssClassName(litTemplateFileName);
        List<String> transformedStyles = styles.stream()
                .map(style -> LitTemplateCssUtil.transformHostToCssClass(style, cssClassName)).toList();

        JavaSource litTemplateSource;
        try {
            File litTemplateClassFile = projectFileManager.getFileForClass(classWithJsModule);
            litTemplateSource = javaFileSourceProvider.getJavaSource(litTemplateClassFile);
        } catch (Exception e) {
            throw new IOException("Unable to read lit template class source. "
                    + "The class might be in an external dependency, making conversion not possible.", e);
        }
        JavaRewriter javaRewriter = new JavaRewriter();
        boolean extendsLitTemplate = javaRewriter.importLitTemplate(litTemplateSource,
                JavaComponent.componentsFromJson(data.withArray("componentDefinitions")),
                transformedStyles.isEmpty() ? null : cssClassName);
        String result = litTemplateSource.getResult();
        // Apply extends change via string replacement to avoid AST modification and
        // preserve formatting.
        if (extendsLitTemplate) {
            result = result.replace("extends LitTemplate", "extends Div");
        }
        ProjectFileManager.get().writeFile(litTemplateSource.getFile(), "Convert LitTemplate to Java", result);

        // Write CSS to a dedicated file
        if (!transformedStyles.isEmpty()) {
            String cssFileName = cssClassName + "-converted.css";
            File cssFile = new File(projectFileManager.getFrontendFolder(), cssFileName);
            String cssContent = String.join("\n\n", transformedStyles);
            ProjectFileManager.get().writeFile(cssFile, "Convert LitTemplate styles", cssContent);
        }

        var litTemplateFile = new File(litTemplateFileName);
        var litTemplateBakFile = new File(litTemplateFileName + ".bak");
        ProjectFileManager.get().moveFile(litTemplateFile, litTemplateBakFile);
    }

    Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
