package com.vaadin.copilot;

import java.io.File;
import java.nio.charset.StandardCharsets;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.helger.css.ECSSVersion;
import com.helger.css.decl.CSSMediaRule;
import com.helger.css.decl.CSSSelector;
import com.helger.css.decl.CSSSelectorSimpleMember;
import com.helger.css.decl.CSSStyleRule;
import com.helger.css.decl.CSSSupportsRule;
import com.helger.css.decl.CascadingStyleSheet;
import com.helger.css.reader.CSSReader;
import com.helger.css.writer.CSSWriter;
import org.apache.commons.lang3.StringUtils;

public class LitTemplateCssUtil {

    static String toCssClassName(String fileName) {
        fileName = fileName.substring(fileName.lastIndexOf(File.separator) + 1);
        return fileName.split("\\.")[0];
    }

    static String transformHostToCssClass(String css, String cssClassName) {
        if (StringUtils.isBlank(css)) {
            return css;
        }

        String classSelector = "." + cssClassName;

        // Handle :host selectors
        String transformedCss = null;
        try {
            transformedCss = transformHostSelectors(css, classSelector);
        } catch (Exception e) {
            // Fallback to original css
            return css;
        }

        try {
            // Scope remaining selectors
            CascadingStyleSheet styleSheet = CSSReader.readFromString(transformedCss, ECSSVersion.LATEST);

            for (CSSStyleRule rule : styleSheet.getAllStyleRules()) {
                scopeSelectors(rule, classSelector);
            }
            for (CSSMediaRule mediaRule : styleSheet.getAllMediaRules()) {
                for (CSSStyleRule rule : mediaRule.getAllStyleRules()) {
                    scopeSelectors(rule, classSelector);
                }
            }
            for (CSSSupportsRule supportsRule : styleSheet.getAllSupportsRules()) {
                for (CSSStyleRule rule : supportsRule.getAllStyleRules()) {
                    scopeSelectors(rule, classSelector);
                }
            }

            return new CSSWriter().setWriteHeaderText(false).getCSSAsString(styleSheet);
        } catch (Exception e) {
            // Fallback to :host transformed css
            return transformedCss;
        }
    }

    private static String transformHostSelectors(String css, String classSelector) {
        // Replace :host followed by rule in parentheses with class selector
        Pattern hostWithParens = Pattern.compile(":host\\(([^)]+)\\)");
        Matcher matcher = hostWithParens.matcher(css);
        StringBuilder sb = new StringBuilder();
        while (matcher.find()) {
            String innerSelector = matcher.group(1);
            matcher.appendReplacement(sb, Matcher.quoteReplacement(classSelector + innerSelector));
        }
        matcher.appendTail(sb);

        // Replace :host without parentheses with class selector
        return sb.toString().replace(":host", classSelector);
    }

    private static void scopeSelectors(CSSStyleRule rule, String className) {
        for (CSSSelector selector : rule.getAllSelectors()) {
            if (selector.hasMembers()) {
                if (!selector.getMemberAtIndex(0).getAsCSSString().startsWith(className)) {
                    CSSSelectorSimpleMember scopeMember = new CSSSelectorSimpleMember(className + " ");
                    selector.addMember(0, scopeMember);
                }
            }
        }
    }
}
