package com.vaadin.copilot;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentInfoFinder;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaFileSourceProvider;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.flow.component.dependency.JsModule;
import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LitTemplateHandler extends CopilotCommand {
    private ComponentSourceFinder sourceFinder;

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        ProjectFileManager projectFileManager = ProjectFileManager.get();
        if (command.equals("get-lit-template-name")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put("reqId", data.get("reqId").asText());
            try {
                this.sourceFinder = new ComponentSourceFinder(getVaadinSession());
                ComponentTypeAndSourceLocation litTemplate = sourceFinder
                        .findTypeAndSourceLocation(data.get("litTemplate"));
                var jsModule = litTemplate.component().getClass().getAnnotation(JsModule.class);
                var litTemplateFile = new File(projectFileManager.getFrontendFolder(),
                        jsModule.value().replace("/", File.separator));
                responseData.put("litTemplateFilename", litTemplateFile.getCanonicalPath());
                devToolsInterface.send(command + "-resp", responseData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                        "Unable to get lit template name", e);
            }
        } else if (command.equals("import-lit-template")) {
            ObjectNode responseData = JacksonUtils.createObjectNode();
            responseData.put("reqId", data.get("reqId").asText());
            try {

                JsonNode warnings = data.get("warnings");
                if (warnings.get("hasTemplateExpressions").asBoolean(false)) {
                    getLogger().warn("Lit template conversion: Template expressions will be lost during conversion.");
                }
                ArrayNode eventBindings = warnings.withArray("eventBindings");
                if (!eventBindings.isEmpty()) {
                    List<String> eventBindingNames = new ArrayList<>();
                    eventBindings.forEach(name -> eventBindingNames.add(name.toString()));
                    getLogger().warn("Lit template conversion: Event bindings will be lost during conversion: {}",
                            eventBindingNames);
                }
                ArrayNode stateProperties = warnings.withArray("statePropertyNames");
                if (!stateProperties.isEmpty()) {
                    List<String> statePropertyNames = new ArrayList<>();
                    stateProperties.forEach(name -> statePropertyNames.add(name.toString()));
                    getLogger().warn("Lit template conversion: State properties will be lost during conversion: {}",
                            statePropertyNames);
                }

                var componentDefinitions = JavaComponent.componentsFromJson(data.withArray("componentDefinitions"));
                ComponentTypeAndSourceLocation litTemplate = sourceFinder
                        .findTypeAndSourceLocation(data.get("litTemplate"));
                JavaFileSourceProvider javaFileSourceProvider = new JavaFileSourceProvider(true);
                ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(javaFileSourceProvider, litTemplate);
                ComponentInfo litTemplateInfo = componentInfoFinder.find();
                var customComponentInfo = litTemplateInfo.customComponentInfo().orElseThrow(
                        () -> new IllegalArgumentException("The given lit template was not found in the project"));
                File litTemplateClass = projectFileManager.getFileForClass(customComponentInfo.componentClass());
                var litTemplateSource = javaFileSourceProvider.getJavaSource(litTemplateClass);
                JavaRewriter javaRewriter = new JavaRewriter();
                javaRewriter.importLitTemplate(litTemplateSource, componentDefinitions);
                String result = litTemplateSource.getResult();
                ProjectFileManager.get().writeFile(litTemplateSource.getFile(), "Convert LitTemplate to Java", result);

                var litTemplateFile = new File(data.get("litTemplateFilename").asText());
                ProjectFileManager.get().deleteFile(litTemplateFile);
                devToolsInterface.send(command + "-resp", responseData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                        "Unable to get lit template name", e);
            }
        }
        return false;
    }

    Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
