package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.vaadin.copilot.JavaReflectionUtil;
import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.customcomponent.CustomComponent;
import com.vaadin.copilot.customcomponent.CustomComponents;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.body.BodyDeclaration;
import com.github.javaparser.ast.body.CallableDeclaration;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.Parameter;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExplicitConstructorInvocationStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import com.github.javaparser.ast.type.Type;
import com.github.javaparser.resolution.declarations.ResolvedConstructorDeclaration;
import com.github.javaparser.resolution.declarations.ResolvedMethodDeclaration;
import com.github.javaparser.resolution.types.ResolvedReferenceType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Responsible for locating and aggregating information about a UI component's
 * creation and attachment within a Java project.
 * <p>
 * This class uses a {@link ProjectFileManager} and
 * {@link ComponentTypeAndSourceLocation} to determine where a component is
 * created and (optionally) where it is attached, then compiles that information
 * into a {@link ComponentInfo} object.
 */
public class ComponentInfoFinder {
    private final ProjectFileManager fileManager;
    private final ComponentTypeAndSourceLocation typeAndSourceLocation;
    private final JavaFileSourceProvider javaFileSourceProvider;

    /**
     * Constructs a {@code ComponentInfoFinder} using the default
     * {@link ProjectFileManager} instance and the given component source metadata.
     *
     * @param componentTypeAndSourceLocation
     *            metadata describing the component's type and source locations
     */
    public ComponentInfoFinder(JavaFileSourceProvider javaFileSourceProvider,
            ComponentTypeAndSourceLocation componentTypeAndSourceLocation) {
        this.fileManager = ProjectFileManager.get();
        this.typeAndSourceLocation = componentTypeAndSourceLocation;
        this.javaFileSourceProvider = javaFileSourceProvider;
    }

    /**
     * Finds and compiles information about a component's creation and (if
     * applicable) attachment in the project.
     * <p>
     * This method:
     * <ul>
     * <li>Loads the Java source file where the component is created</li>
     * <li>Extracts related creation details</li>
     * <li>If available, loads the Java source file where the component is attached
     * and extracts attachment details</li>
     * </ul>
     *
     * @return a {@link ComponentInfo} object containing the collected creation and
     *         attachment data
     * @throws IOException
     *             if any source file could not be read
     */
    public ComponentInfo find() throws IOException {
        File createLocationJavaFile = fileManager.getSourceFile(typeAndSourceLocation.getCreateLocationOrThrow());
        ComponentInfoBuilder builder = new ComponentInfoBuilder();

        findCreateLocationRelatedInfo(createLocationJavaFile, typeAndSourceLocation, builder);
        if (typeAndSourceLocation.attachLocationInProject().isPresent()) {
            File attachLocationJavaFile = fileManager.getSourceFile(typeAndSourceLocation.getAttachLocationOrThrow());
            findAttachLocationRelatedInfo(attachLocationJavaFile, typeAndSourceLocation, builder);
        }
        return builder.build();
    }

    private void findAttachLocationRelatedInfo(File file, ComponentTypeAndSourceLocation typeAndSourceLocation,
            ComponentInfoBuilder builder) throws IOException {
        JavaSource javaSource = javaFileSourceProvider.getJavaSource(file);

        boolean inSameFileWithCreateLocation = typeAndSourceLocation.getCreateLocationOrThrow().filename()
                .equals(typeAndSourceLocation.getAttachLocationOrThrow().filename());

        builder.createAndAttachLocationsAreInSameFile(inSameFileWithCreateLocation);

        if (inSameFileWithCreateLocation) {
            findAttachLocationInSameFile(javaSource, builder);
        } else {
            findAttachLocationFromAnotherFile(javaSource, builder);
        }
        Optional<ComponentAttachInfo> componentAttachInfoOptional = builder.getComponentAttachInfoOptional();
        if (componentAttachInfoOptional.isPresent()) {
            ComponentAttachInfo componentAttachInfo = componentAttachInfoOptional.get();
            AttachExpression attachCall = componentAttachInfo.getAttachCall();
            boolean compositeContainer = isChildOfCompositeContainer(attachCall.expression());
            if (compositeContainer && attachCall.expression().isMethodCallExpr()) {
                MethodCallExpr methodCallExpr = attachCall.getMethodCallExpression();
                if (methodCallExpr != null && methodCallExpr.getNameAsString().equals("getContent")
                        && methodCallExpr.getParentNode().isPresent()) {
                    componentAttachInfo
                            .setAttachCall(new AttachExpression((Expression) methodCallExpr.getParentNode().get()));
                }
            }
            builder.containerComposite(compositeContainer);
        }

    }

    /**
     * Finds attach location from a different class than the file that contains
     * Create location. Steps as follows:
     * <ol>
     * <li>All parameter usages of the variable is found from the class where
     * variable is initialized aka create location</li>
     * <li>Finds method call or constructor calls to find relevant calls from the
     * target class</li>
     * <li>Full class name is obtained by resolving methods or constructor
     * calls</li>
     * <li>Static class parser is used to analyze method declarations&constructors
     * in the target class</li>
     * <li>Target attach call is tracked with using argument index</li>
     * <li>When attach location is found within method or constructor declarations,
     * builder data is updated and process is completed</li>
     * </ol>
     * Firstly, we need to get all parameter usages of the variable. we look for
     * relevant methods or constructors of where parameter is used. Then,
     *
     * @param javaSource
     *            Java source of attach location file
     * @param builder
     *            Builder that holds metadata
     * @throws IOException
     *             is thrown when file read fails
     */
    private void findAttachLocationFromAnotherFile(JavaSource javaSource, ComponentInfoBuilder builder)
            throws IOException {
        ComponentInfo temporaryInfo = builder.build();
        List<Expression> parameters = JavaRewriterUtil.findParameterUsage(temporaryInfo);

        for (Expression parameter : parameters) {
            MethodCallExpr relevantMethod = JavaRewriterUtil.findAncestor(parameter, MethodCallExpr.class);
            if (relevantMethod != null) {
                // aClass.fooMethod(component);
                ResolvedMethodDeclaration resolve = relevantMethod.resolve();
                String qualifiedClassName = resolve.declaringType().getQualifiedName();
                int argumentIndex = JavaRewriterUtil.findArgumentIndex(relevantMethod, parameter);
                Expression argumentExpression = findMethodArgumentByStaticAnalysis(qualifiedClassName,
                        relevantMethod.getNameAsString(), argumentIndex);
                if (setComponentInfo(javaSource, builder, argumentExpression)) {
                    return;
                }
            }
            ExplicitConstructorInvocationStmt relevantExplicitConstructorInvocationStmt = JavaRewriterUtil
                    .findAncestor(parameter, ExplicitConstructorInvocationStmt.class);
            if (relevantExplicitConstructorInvocationStmt != null) {
                // super(new Grid<>());
                ResolvedConstructorDeclaration resolve = relevantExplicitConstructorInvocationStmt.resolve();
                String currentClassName = resolve.declaringType().getQualifiedName();
                int argumentIndex = JavaRewriterUtil.findArgumentIndex(relevantExplicitConstructorInvocationStmt,
                        parameter);
                Expression argumentExpression = findMethodArgumentByStaticAnalysis(currentClassName, "<init>",
                        argumentIndex);
                if (setComponentInfo(javaSource, builder, argumentExpression)) {
                    return;
                }
            }
        }
    }

    private boolean setComponentInfo(JavaSource javaSource, ComponentInfoBuilder builder,
            Expression argumentExpression) {
        // TODO support field assignments.
        if (argumentExpression != null) {
            ComponentAttachInfo attachInfo = new ComponentAttachInfo(javaSource);
            attachInfo.setAttachCall(new AttachExpression(
                    JavaRewriterUtil.findAncestorOrThrow(argumentExpression, MethodCallExpr.class)));
            attachInfo.setComponentAttachScope(JavaRewriterUtil.findBlock(argumentExpression));
            if (argumentExpression.isNameExpr()) {
                attachInfo.setLocalVariableName(argumentExpression.asNameExpr().getNameAsString());
            }
            builder.componentAttachInfo(attachInfo);
            return true;
        }
        return false;
    }

    private void findAttachLocationInSameFile(JavaSource javaSource, ComponentInfoBuilder builder) {
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        Optional<BlockStmt> componentAttachScope;
        AttachExpression attachCall;
        Optional<MethodCallExpr> attachMethodCall = Optional.empty();
        var createInfo = builder.getCreateInfoOrThrow();
        String varName;
        if (typeAndSourceLocation.attachLocationInProject().isPresent()) {
            if (createInfo.getFieldName() != null) {
                varName = createInfo.getFieldName();
            } else if (createInfo.getLocalVariableName() != null) {
                varName = createInfo.getLocalVariableName();
            } else {
                varName = null;
            }
            attachMethodCall = typeAndSourceLocation.attachLocationInProject()
                    .flatMap(k -> findAttachMethodCallExpressionOnLine(compilationUnit, k.lineNumber(), varName));
        } else {
            varName = null;
        }
        if (attachMethodCall.isEmpty() && !builder.isClassSource()) {
            Optional<ObjectCreationExpr> attachObjectCreationCall = typeAndSourceLocation.attachLocationInProject()
                    .flatMap(location -> JavaRewriterUtil.findNodeOfType(compilationUnit, location.lineNumber(),
                            ObjectCreationExpr.class));
            if (attachObjectCreationCall.isEmpty()) {
                // location and attach location are not in the same file.

                Optional<AttachExpression> possibleAttachExpressionFromParentInfo = findPossibleAttachExpressionFromParentInfo(
                        typeAndSourceLocation, createInfo.getObjectCreationExpr(), varName);

                if (possibleAttachExpressionFromParentInfo.isEmpty()) {
                    throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                            "Attach not found at the expected location");
                }
                attachCall = possibleAttachExpressionFromParentInfo.get();
                componentAttachScope = Optional
                        .of(JavaRewriterUtil.findBlock(possibleAttachExpressionFromParentInfo.get().getNode()));
            } else {
                componentAttachScope = attachObjectCreationCall.map(JavaRewriterUtil::findBlock);
                attachCall = new AttachExpression(attachObjectCreationCall.orElse(null));
            }
        } else {
            componentAttachScope = attachMethodCall.map(JavaRewriterUtil::findBlock);
            attachCall = new AttachExpression(attachMethodCall.orElse(null));
        }

        ComponentAttachInfo attachInfo = createAttachInfo(javaSource, attachCall, componentAttachScope.orElse(null),
                builder.getCreateInfoOrThrow());

        builder.createAndAttachLocationsAreInSameFile(true);
        builder.componentAttachInfo(attachInfo);
    }

    public static ComponentAttachInfo createAttachInfo(JavaSource javaSource, AttachExpression attachCall,
            BlockStmt componentAttachScope, ComponentCreateInfo createInfo) {

        ComponentAttachInfo attachInfo = new ComponentAttachInfo(javaSource);
        attachInfo.setAttachCall(attachCall);
        attachInfo.setComponentAttachScope(componentAttachScope);

        attachInfo.setFieldName(createInfo.getFieldName());
        attachInfo.setFieldDeclaration(createInfo.getFieldDeclaration());
        attachInfo.setFieldDeclarationAndAssignment(createInfo.getFieldDeclarationAndAssignment());
        attachInfo.setLocalVariableName(createInfo.getLocalVariableName());
        attachInfo.setLocalVariableDeclarator(createInfo.getLocalVariableDeclarator());

        return attachInfo;
    }

    private void findCreateLocationRelatedInfo(File file, ComponentTypeAndSourceLocation typeAndSourceLocation,
            ComponentInfoBuilder builder) throws IOException {
        JavaSource javaSource = javaFileSourceProvider.getJavaSource(file);
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        List<ObjectCreationExpr> objectCreationExprs = new ArrayList<>();
        for (Class clazz : typeAndSourceLocation.inheritanceChain()) {
            Optional<ComponentTracker.Location> maybeLocationInProject = typeAndSourceLocation
                    .createLocationInProject();
            if (maybeLocationInProject.isEmpty()) {
                continue;
            }
            ComponentTracker.Location createLocationInProject = maybeLocationInProject.get();
            objectCreationExprs = JavaRewriterUtil.findNodesOfType(compilationUnit,
                    createLocationInProject.lineNumber(), ObjectCreationExpr.class,
                    node -> node.getType().asClassOrInterfaceType().getName().asString().equals(clazz.getSimpleName()));
            if (!objectCreationExprs.isEmpty()) {
                break;
            }
        }

        if (objectCreationExprs.size() > 1) {
            throw new IllegalArgumentException(
                    "There are multiple components created on the given line and we are unable to determine which one to modify");
        }

        Optional<ObjectCreationExpr> objectCreationExpr = Optional
                .ofNullable(objectCreationExprs.isEmpty() ? null : objectCreationExprs.get(0));

        boolean classSource = false;
        boolean isAnonymousComponent = false;
        boolean isReturnValue = false;
        if (objectCreationExpr.isEmpty()) {
            classSource = JavaRewriterUtil.isLocationRefersSource(typeAndSourceLocation, compilationUnit);
            if (!classSource) {
                int lineNumber = typeAndSourceLocation.getCreateLocationOrThrow().lineNumber();
                String[] lines = javaSource.getSource().split("[\r\n]");
                String lineContents;
                if (lines.length > lineNumber) {
                    lineContents = "\"" + lines[lineNumber - 1].trim() + "\"";
                } else {
                    lineContents = "no line with number " + lineNumber + " as the file only has " + lines.length
                            + " lines";
                }
                throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                        "Expected to find an object creation expression such as \"new "
                                + typeAndSourceLocation.type().getSimpleName() + "()\" at "
                                + typeAndSourceLocation.javaFile().map(File::getAbsolutePath).orElse("?") + ":"
                                + lineNumber + " but found " + lineContents);
            }
        } else {
            ObjectCreationExpr n = objectCreationExpr.orElse(null);
            Node parent = n.getParentNode().orElse(null);
            if (n != null && parent != null && !(parent instanceof VariableDeclarator || parent instanceof AssignExpr
                    || parent instanceof FieldDeclaration)) {
                isAnonymousComponent = true;
            }
            if (n != null && parent != null && (parent instanceof ReturnStmt)) {
                isReturnValue = true;
            }
        }
        Optional<ConstructorDeclaration> targetConstructorDeclaration = Optional.empty();
        if (classSource) {
            targetConstructorDeclaration = findTargetConstructorDeclaration(compilationUnit, typeAndSourceLocation);
        }
        boolean expressionInLoop = JavaRewriterUtil.checkExpressionInLoop(objectCreationExpr.orElse(null))
                || JavaRewriterUtil.checkMethodHasExpressionCalledInLoop(objectCreationExpr.orElse(null));

        Optional<CustomComponent> customComponentInfo = CustomComponents
                .getCustomComponentInfo(typeAndSourceLocation.type());

        ComponentCreateInfo componentCreateInfo = createComponentCreateInfo(javaSource,
                objectCreationExpr.orElse(null));
        builder.type(typeAndSourceLocation.type()).customComponentInfo(customComponentInfo.orElse(null))
                .routeConstructor(targetConstructorDeclaration.orElse(null)).componentCreateInfo(componentCreateInfo)
                .isReturnValue(isReturnValue).classSource(classSource).isAnonymousComponent(isAnonymousComponent)
                .createdInLoop(expressionInLoop);

    }

    public static ComponentCreateInfo createComponentCreateInfo(JavaSource javaSource,
            ObjectCreationExpr objectCreationExpr) {
        Optional<ObjectCreationExpr> objectCreationExprOptional = Optional.ofNullable(objectCreationExpr);
        Optional<BlockStmt> componentCreateScope = objectCreationExprOptional.map(JavaRewriterUtil::findBlock);

        Optional<VariableDeclarator> localVariableDeclarator = objectCreationExprOptional
                .map(expr -> JavaRewriterUtil.findAncestor(expr, VariableDeclarator.class));
        Optional<AssignExpr> assignmentExpression = objectCreationExprOptional
                .map(expr -> JavaRewriterUtil.findAncestor(expr, AssignExpr.class));
        Optional<FieldDeclaration> fieldDeclarationAndAssignment = objectCreationExprOptional
                .map(expr -> JavaRewriterUtil.findAncestor(expr, FieldDeclaration.class));
        Optional<FieldDeclaration> fieldDeclaration = fieldDeclarationAndAssignment;
        if (localVariableDeclarator.isPresent() && fieldDeclarationAndAssignment.isPresent()) {
            // A variable declarator is found also for assignments for fields
            // but we want to differentiate between the two cases later on
            localVariableDeclarator = Optional.empty();
        }

        String localVariableName = null;
        String fieldName = null;
        if (localVariableDeclarator.isPresent()) {
            // TextField foo = new TextField();
            localVariableName = localVariableDeclarator.get().getNameAsString();
        } else if (fieldDeclarationAndAssignment.isPresent()) {
            // private TextField foo = new TextField();
            fieldName = fieldDeclarationAndAssignment.get().getVariable(0).getNameAsString();
        } else if (assignmentExpression.isPresent()) {
            // foo = new TextField();
            // Here foo can be a local variable or field
            String localVariableOrFieldName;
            Expression target = assignmentExpression.get().getTarget();
            if (target.isNameExpr()) {
                localVariableOrFieldName = target.asNameExpr().getNameAsString();
            } else if (target.isFieldAccessExpr()) {
                localVariableOrFieldName = target.asFieldAccessExpr().getNameAsString();
            } else {
                throw new IllegalArgumentException("Unhandled target type for assignment. Expression=" + target);
            }

            if (componentCreateScope.isPresent() && JavaRewriterUtil
                    .findLocalVariableDeclarator(localVariableOrFieldName, componentCreateScope.get()) != null) {
                localVariableName = localVariableOrFieldName;
            } else {
                fieldName = localVariableOrFieldName;
                fieldDeclaration = Optional
                        .ofNullable(JavaRewriterUtil.findFieldDeclaration(assignmentExpression.get(), fieldName));
            }
        }

        ComponentCreateInfo componentCreateInfo = new ComponentCreateInfo(javaSource);
        componentCreateInfo.setFieldDeclaration(fieldDeclaration.orElse(null));
        componentCreateInfo.setFieldDeclarationAndAssignment(fieldDeclarationAndAssignment.orElse(null));
        componentCreateInfo.setFieldName(fieldName);
        componentCreateInfo.setComponentCreateScope(componentCreateScope.orElse(null));
        componentCreateInfo.setLocalVariableDeclarator(localVariableDeclarator.orElse(null));
        componentCreateInfo.setLocalVariableName(localVariableName);
        componentCreateInfo.setObjectCreationExpr(objectCreationExprOptional.orElse(null));
        componentCreateInfo.setAssignmentExpression(assignmentExpression.orElse(null));
        return componentCreateInfo;

    }

    private Optional<ConstructorDeclaration> findTargetConstructorDeclaration(CompilationUnit compilationUnit,
            ComponentTypeAndSourceLocation typeAndSourceLocation) {
        ComponentTracker.Location createLocation = typeAndSourceLocation.getCreateLocationOrThrow();
        Optional<ConstructorDeclaration> targetConstructorDeclaration = JavaRewriterUtil.findNodeOfType(compilationUnit,
                createLocation.lineNumber(), ConstructorDeclaration.class);
        if (targetConstructorDeclaration.isPresent()) {
            return targetConstructorDeclaration;
        }
        if (createLocation.className().contains("$")) {
            Optional<ClassOrInterfaceDeclaration> maybeClassDeclaration = JavaRewriterUtil
                    .findNodeOfType(compilationUnit, createLocation.lineNumber(), ClassOrInterfaceDeclaration.class);
            // it is possible that createLocation would have the line number where class
            // declaration presents.
            if (maybeClassDeclaration.isPresent()) {
                targetConstructorDeclaration = Optional.of(new ConstructorDeclaration()
                        .setName(typeAndSourceLocation.type().getSimpleName()).setPublic(true));
                ClassOrInterfaceDeclaration classOrInterfaceDeclaration = maybeClassDeclaration.get();
                classOrInterfaceDeclaration.addMember(targetConstructorDeclaration.get());
                return targetConstructorDeclaration;
            }
        }
        // Potentially no constructor
        List<ConstructorDeclaration> constructors = compilationUnit.findAll(ConstructorDeclaration.class);
        if (constructors.isEmpty()) {
            // Need to create a constructor
            targetConstructorDeclaration = Optional.of(
                    new ConstructorDeclaration().setName(typeAndSourceLocation.type().getSimpleName()).setPublic(true));
            String className = typeAndSourceLocation.type().getSimpleName();
            ClassOrInterfaceDeclaration classDeclaration = compilationUnit.findAll(ClassOrInterfaceDeclaration.class)
                    .stream().filter(c -> c.getNameAsString().equals(className)).findFirst()
                    .orElseThrow(() -> new IllegalArgumentException("Class " + className + " not found"));
            classDeclaration.addMember(targetConstructorDeclaration.get());
        } else {
            throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                    "Route class has constructors but none at the expected location: " + createLocation);
        }
        return targetConstructorDeclaration;
    }

    private boolean isChildOfCompositeContainer(Expression attachCallExpr) {
        if (attachCallExpr == null) {
            return false;
        }
        return JavaRewriterUtil.isNodeInCompositeClass(attachCallExpr);
    }

    /**
     * To find dynamically generated component attach location.
     *
     * @param componentTypeAndSourceLocation
     *            component location
     * @param varName
     *            local or field variable name
     * @param objectCreationExpr
     *            creation expression is used for inline variables.
     * @return returns empty if parent is null or argument variables are not present
     */
    private Optional<AttachExpression> findPossibleAttachExpressionFromParentInfo(
            ComponentTypeAndSourceLocation componentTypeAndSourceLocation, ObjectCreationExpr objectCreationExpr,
            String varName) {
        if (componentTypeAndSourceLocation.parent() == null) {
            return Optional.empty();
        }
        List<Expression> possibleArgs = new ArrayList<>();
        if (varName != null) {
            possibleArgs.add(new NameExpr(varName));
        }
        if (objectCreationExpr != null) {
            possibleArgs.add(objectCreationExpr);
        }
        if (possibleArgs.isEmpty()) {
            return Optional.empty();
        }
        ComponentTypeAndSourceLocation parent = componentTypeAndSourceLocation.parent();

        ComponentInfo componentInfo;
        try {
            ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(this.javaFileSourceProvider, parent);
            componentInfo = componentInfoFinder.find();
        } catch (IOException e) {
            getLogger().warn("Unable to find component info for {}", componentTypeAndSourceLocation.type());
            return Optional.empty();
        }

        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(componentInfo);

        return methodCallStatements.stream().filter(
                f -> f.getNameAsString().equals("add") && f.getArguments().stream().anyMatch(possibleArgs::contains))
                .findFirst().map(AttachExpression::new);
    }

    /**
     * The method parses finds the file of given java class and parses it using
     * {@link CompilationUnit}. Then, method declarations in the class are found by
     * comparing method names with the given name. If there is no method found in
     * the class, superclasses are called recursively until method is found or there
     * is no super class to check. If there are methods,
     * {@link #findAttachArgumentInMethodDeclaration} is called for each method
     * until the relevant method is found.
     *
     * @param qualifiedJavaClassName
     *            full java class name. e.g. com.vaadin.copilot.JavaRewriter
     * @param methodName
     *            method name to check e.g. setContent. <code> &lt;init&gt;</code>
     *            for constructor
     * @param argumentIndex
     *            argument index of the method
     * @return Argument expression if found, null otherwise.
     * @throws IOException
     *             might be thrown if IO exception happens while reading javaSource
     *             of the class.
     */
    private Expression findMethodArgumentByStaticAnalysis(String qualifiedJavaClassName, String methodName,
            int argumentIndex) throws IOException {
        File fileForClass = ProjectFileManager.get().getFileForClass(qualifiedJavaClassName);
        if (!fileForClass.exists()) {
            return null;
        }
        Class<?> clazz = JavaReflectionUtil.getClass(qualifiedJavaClassName);
        ComponentTracker.Location attachLocationOrThrow = typeAndSourceLocation.getAttachLocationOrThrow();

        JavaSource javaSource = javaFileSourceProvider.getJavaSource(fileForClass);
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        ClassOrInterfaceDeclaration classDeclaration = ClassSourceStaticAnalyzer
                .findRelevantClassDeclaration(compilationUnit, clazz);
        List<CallableDeclaration> list = classDeclaration.getMembers().stream()
                .filter(BodyDeclaration::isCallableDeclaration).map(CallableDeclaration.class::cast)
                .filter(callableDeclaration -> {
                    if (callableDeclaration.getParameters().size() <= argumentIndex) {
                        return false;
                    }
                    if (methodName.equals("<init>") && callableDeclaration instanceof ConstructorDeclaration) {
                        return true;
                    }
                    return callableDeclaration instanceof MethodDeclaration methodDeclaration
                            && methodName.equals(methodDeclaration.getNameAsString());
                }).toList();
        if (list.isEmpty()) {
            if (clazz.getSuperclass() != null && !clazz.getSuperclass().equals(Object.class)) {
                // no relevant method found with the given method name. Method might be
                // inherited from a super class.
                return findMethodArgumentByStaticAnalysis(clazz.getSuperclass().getName(), methodName, argumentIndex);
            }
            return null;
        }

        for (CallableDeclaration<?> callableDeclaration : list) {
            Expression attachArgumentInMethodDeclaration = findAttachArgumentInMethodDeclaration(compilationUnit,
                    callableDeclaration, attachLocationOrThrow, argumentIndex);
            if (attachArgumentInMethodDeclaration != null) {
                return attachArgumentInMethodDeclaration;
            }
        }
        // Could not find relevant method calls. The attach location might be passed to
        // another class or another method.
        return null;
    }

    /**
     * This method looks for method declaration or constructors from a class that
     * has the location provided by Attach Location. The main purpose of this method
     * is to get the variable name from attach call. We have the line number from
     * {@link ComponentTracker.Location} but no argument, or column number.
     * Furthermore, methods can be overloaded, so method parameters are resolved and
     * checked to ensure that the parameter we found is the correct one.
     *
     * @param compilationUnit
     *            Compilation unit to search for method calls.
     * @param callableDeclaration
     *            Method declaration to check
     * @param attachLocation
     *            Attach location to find relevant statement in method or
     *            construction call
     * @param argumentIndex
     *            Argument to be called from other classes
     * @return Argument expression e.g. button from <code>add(button);</code>
     */
    private Expression findAttachArgumentInMethodDeclaration(CompilationUnit compilationUnit,
            CallableDeclaration<?> callableDeclaration, ComponentTracker.Location attachLocation, int argumentIndex) {
        Parameter parameter = callableDeclaration.getParameter(argumentIndex);
        Type type = parameter.getType();
        if (type.isReferenceType()) {
            ResolvedReferenceType resolved = type.asReferenceType().resolve().asReferenceType();
            Class<?> aClass = JavaReflectionUtil.getClass(resolved.getQualifiedName());
            if (!Component.class.isAssignableFrom(aClass)) {
                return null;
            }
        }
        String parameterName = parameter.getName().asString();
        BlockStmt blockStmt = null;
        if (callableDeclaration instanceof ConstructorDeclaration constructorDeclaration) {
            blockStmt = constructorDeclaration.getBody();
        } else if (callableDeclaration instanceof MethodDeclaration methodDeclaration) {
            blockStmt = methodDeclaration.getBody()
                    .orElseThrow(() -> new IllegalArgumentException("Unable to get method body"));
        }
        if (blockStmt == null) {
            return null;
        }

        Optional<MethodCallExpr> attachMethodCallExpressionOnLine = findAttachMethodCallExpressionOnLine(
                compilationUnit, attachLocation.lineNumber(), parameterName);
        if (attachMethodCallExpressionOnLine.isPresent()) {
            MethodCallExpr methodCallExpr = attachMethodCallExpressionOnLine.get();
            return methodCallExpr.getArguments().stream().filter(Expression::isNameExpr).map(Expression::asNameExpr)
                    .filter(arg -> arg.toString().equals(parameterName)).findFirst().orElse(null);
        }
        return null;
    }

    /**
     * Attempts to find a {@link MethodCallExpr} on the specified line of a
     * {@link CompilationUnit} that uses the given variable name as one of its
     * arguments. If {@code variableName} is {@code null}, or if no such method call
     * is found, it attempts to locate any {@code MethodCallExpr} at the line,
     * optionally falling back to a source location if available.
     *
     * @param compilationUnit
     *            the compilation unit to search in
     * @param lineNumber
     *            the line number to inspect
     * @param variableName
     *            the name of the variable to look for in the method arguments, or
     *            {@code null} to ignore
     * @return an {@link Optional} containing the matched {@link MethodCallExpr}, or
     *         empty if none found
     */
    private Optional<MethodCallExpr> findAttachMethodCallExpressionOnLine(CompilationUnit compilationUnit,
            int lineNumber, String variableName) {
        if (variableName == null) {
            return typeAndSourceLocation.attachLocationInProject().flatMap(location -> JavaRewriterUtil
                    .findNodeOfType(compilationUnit, location.lineNumber(), MethodCallExpr.class)).stream().findFirst();
        }
        List<MethodCallExpr> methodCallExprs = JavaRewriterUtil
                .findNodes(compilationUnit, lineNumber, MethodCallExpr.class::isInstance).stream()
                .map(MethodCallExpr.class::cast).toList();
        for (MethodCallExpr methodCallExpr : methodCallExprs) {
            if (methodCallExpr.getArguments().stream()
                    .anyMatch(arg -> JavaRewriterUtil.equalsByNameAsString(arg, new NameExpr(variableName)))) {
                return Optional.of(methodCallExpr);
            }
        }
        // falling back to pick the method at the line without checking the arguments.
        return findAttachMethodCallExpressionOnLine(compilationUnit, lineNumber, null);
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
