package com.vaadin.copilot.plugins.vaadinversionupdate;

import java.net.ConnectException;
import java.net.URI;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotCommand;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.ErrorHandler;
import com.vaadin.flow.internal.JacksonUtils;
import com.vaadin.flow.server.Platform;
import com.vaadin.flow.server.startup.ApplicationConfiguration;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handler for displaying and upgrading to new versions
 */
public class VaadinVersionUpdateHandler extends CopilotCommand {
    private static final String ERROR_MESSAGE = "errorMessage";
    private static final String ERROR_KEY = "error";
    private final VaadinVersionUpdate vaadinVersionUpdate;

    private final CopilotServerClient copilotServerClient = new CopilotServerClient();

    /**
     * Constructs a new instance of VersionCheckerHandler
     *
     * @param applicationConfiguration
     *            Application configuration to access project root folder
     */
    public VaadinVersionUpdateHandler(ApplicationConfiguration applicationConfiguration) {
        this.vaadinVersionUpdate = new VaadinVersionUpdate(applicationConfiguration.getProjectFolder().toPath());
    }

    @Override
    public boolean handleMessage(String command, JsonNode data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-new-vaadin-versions")) {
            var reqId = data.get(KEY_REQ_ID).asText();
            var responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, reqId);

            Optional<String> vaadinVersion = Platform.getVaadinVersion();
            if (vaadinVersion.isEmpty()) {
                responseData.put(ERROR_KEY, true);
                responseData.put(ERROR_MESSAGE, "Unable to find vaadin version");
                return true;
            }
            boolean includePreReleases = data.has("includePreReleases") && data.get("includePreReleases").asBoolean();
            VersionCheckRequest versionCheckRequest = new VersionCheckRequest(vaadinVersion.get(), includePreReleases);
            sendVersionRequestAndFillResponse(versionCheckRequest, responseData);
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        } else if (command.equals("update-vaadin-version")) {
            var reqId = data.get(KEY_REQ_ID).asText();
            var responseData = JacksonUtils.createObjectNode();
            responseData.put(KEY_REQ_ID, reqId);
            try {
                vaadinVersionUpdate.updateVaadinVersion(data.get("newVersion").asText(),
                        data.has("preRelease") && data.get("preRelease").asBoolean());
                devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData,
                        "Unable to update Vaadin Version", e);
            }
            return true;
        }
        return false;
    }

    private void sendVersionRequestAndFillResponse(VersionCheckRequest versionCheckRequest, ObjectNode responseData) {
        try {
            URI uri = copilotServerClient.getQueryURI("vaadin-version");
            String vaadinVersion = versionCheckRequest.version();
            HttpRequest request = copilotServerClient.buildRequest(uri, versionCheckRequest);
            HttpResponse<String> response = copilotServerClient.getHttpClient().send(request,
                    HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() == 200) {
                responseData.set("newVersions", JacksonUtils.writeValue(response.body()));
            } else {
                if (response.statusCode() == 404) {
                    responseData.put(ERROR_MESSAGE, "Requested version[" + vaadinVersion + "] not found");
                } else if (response.statusCode() == 500) {
                    responseData.put(ERROR_MESSAGE, "Internal Server Error. Please try again later.");
                } else if (response.statusCode() == 502) {
                    responseData.put(ERROR_MESSAGE, "The server is temporarily unavailable. Please try again later.");
                } else {
                    responseData.put(ERROR_MESSAGE, "Unexpected error occurred. Please try again later.");
                }
                responseData.put(ERROR_KEY, true);
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        } catch (ConnectException e) {
            getLogger().debug("Unable to connect to server", e);
            responseData.put(ERROR_KEY, true);
            responseData.put(ERROR_MESSAGE, "Unable to connect to server. Please try again later.");
        } catch (Exception e) {
            getLogger().error("Error while sending version check request", e);
            responseData.put(ERROR_KEY, true);
            responseData.put(ERROR_MESSAGE, "An unexpected error occurred while processing the request");
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(VaadinVersionUpdateHandler.class);
    }

    @Override
    public boolean canBeParallelCommand(String command) {
        return command.equals("get-new-vaadin-versions");
    }

    /**
     * Request data for copilot server to fetch new versions after given version.
     *
     * @param version
     *            Version of the project
     * @param includePreReleases
     *            flag to display pre releases
     */
    public record VersionCheckRequest(String version, boolean includePreReleases) {
    }
}
