package com.vaadin.copilot.javarewriter;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.function.BiFunction;

import com.vaadin.copilot.JavaReflectionUtil;
import com.vaadin.flow.component.HasStyle;
import com.vaadin.flow.component.Text;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.StaticJavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.LambdaExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.MethodReferenceExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.VariableDeclarationExpr;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;

/**
 * This class contains hacks related to Flow components and how they work.
 */
public final class FlowComponentQuirks {

    public static final String HAS_PREFIX_QUALIFIED_CLASS_NAME = "com.vaadin.flow.component.shared.HasPrefix";
    public static final String HAS_SUFFIX_QUALIFIED_CLASS_NAME = "com.vaadin.flow.component.shared.HasSuffix";

    public static final String CHART_TYPE = "com.vaadin.flow.component.charts.model.ChartType";
    private static final String AVATAR_GROUP = "com.vaadin.flow.component.avatar.AvatarGroup";
    private static final String AVATAR_GROUP_ITEM = AVATAR_GROUP + ".AvatarGroupItem";
    private static final String MESSAGE_LIST = "com.vaadin.flow.component.messages.MessageList";
    private static final String MESSAGE_LIST_ITEM = "com.vaadin.flow.component.messages.MessageListItem";
    private static final String CHARTS_DATA_SERIES = "com.vaadin.flow.component.charts.model.DataSeries";
    private static final String CHARTS_LIST_SERIES = "com.vaadin.flow.component.charts.model.ListSeries";
    private static final String CHARTS_DATA_SERIES_ITEM = "com.vaadin.flow.component.charts.model.DataSeriesItem";
    private static final String CARD = "com.vaadin.flow.component.card.Card";

    private static final String GRID_TREE_COLUMN = "GridTreeColumn";
    private static final String GRID_COLUMN = "GridColumn";
    private static final String GRID_SELECTION_COLUMN = "GridSelectionColumn";
    private static final String GRID_SORT_COLUMN = "GridSortColumn";
    private static final String ITEMS = "items";
    private static final String TAB_SHEET_TAB = "TabSheetTab";
    private static final String MENU_ITEM = "MenuItem";
    private static final String SUB_MENU = "SubMenu";

    private FlowComponentQuirks() {
        // Utility class
    }

    /**
     * Returns the Java class name for a given HTML tag.
     *
     * @param javaComponent
     *            the java Component
     * @return the Java class name
     */
    public static String getClassForComponent(JavaComponent javaComponent) {
        String tag = javaComponent.tag();
        if (tag.equals(JavaComponent.TEXT_NODE)) {
            return Text.class.getName();
        }
        if (javaComponent.children().stream().anyMatch(c -> c.tag().equalsIgnoreCase(GRID_TREE_COLUMN))) {
            tag = "TreeGrid";
        }
        if (tag.equalsIgnoreCase("label")) {
            tag = "NativeLabel";
        } else if (tag.equalsIgnoreCase("A")) {
            tag = "Anchor";
        } else if (tag.equalsIgnoreCase("P")) {
            tag = "Paragraph";
        } else if (tag.equalsIgnoreCase("OL")) {
            tag = "OrderedList";
        } else if (tag.equalsIgnoreCase("ul")) {
            tag = "UnorderedList";
        } else if (tag.equalsIgnoreCase("dl")) {
            tag = "DescriptionList";
        } else if (tag.equalsIgnoreCase("dt")) {
            tag = "DescriptionList.Term";
        } else if (tag.equalsIgnoreCase("dd")) {
            tag = "DescriptionList.Description";
        } else if (tag.equalsIgnoreCase("LI")) {
            tag = "ListItem";
        } else if (tag.equalsIgnoreCase("img")) {
            tag = "Image";
        } else if (tag.equalsIgnoreCase("DashboardLayout")) {
            tag = "Dashboard";
        }
        String capitalizedTag = SharedUtil.capitalize(tag);
        String lowercaseTag = tag.toLowerCase(Locale.ENGLISH);
        String lowercaseFirstPart = removeInitialDash(SharedUtil.camelCaseToDashSeparated(tag)).split("-", 2)[0]
                .toLowerCase(Locale.ENGLISH);

        String[] probes = new String[] {
                // H1 and other html elements
                "com.vaadin.flow.component.html." + capitalizedTag,
                // Button and other Vaadin components
                "com.vaadin.flow.component." + lowercaseTag + "." + capitalizedTag,
                // AvatarGroup is avatar.AvatarGroup
                "com.vaadin.flow.component." + lowercaseFirstPart + "." + capitalizedTag,
                "com.vaadin.flow.component.orderedlayout." + capitalizedTag,
                "com.vaadin.flow.component.messages." + capitalizedTag,
                "com.vaadin.flow.component.sidenav." + capitalizedTag,
                "com.vaadin.flow.component.textfield." + capitalizedTag,
                "com.vaadin.flow.component.combobox." + capitalizedTag,
                "com.vaadin.flow.component.applayout." + capitalizedTag,
                "com.vaadin.flow.component.charts." + capitalizedTag,
                "com.vaadin.flow.component.tabs." + capitalizedTag, "com.vaadin.flow.component.grid." + capitalizedTag,
                "com.vaadin.flow.component.treegrid." + capitalizedTag,
                "com.vaadin.flow.component.dashboard." + capitalizedTag, };
        for (String probe : probes) {
            if (JavaReflectionUtil.exists(probe)) {
                return probe;
            }
        }
        if (tag.equalsIgnoreCase("BoardRow")) {
            return "com.vaadin.flow.component.board.Row";
        }
        if (tag.equalsIgnoreCase("RadioGroup")) {
            return "com.vaadin.flow.component.radiobutton.RadioButtonGroup";
        }
        if (tag.equalsIgnoreCase("ChartSeries") && javaComponent.props().containsKey("values")) {
            List values = (List) javaComponent.props().get("values");
            if (!values.isEmpty() && (values.get(0) instanceof Map || values.get(0) instanceof List)) {
                return CHARTS_DATA_SERIES;
            }
            return CHARTS_LIST_SERIES;
        }
        throw new IllegalArgumentException("Unable to find Java class for <" + tag + ">");
    }

    public static List<String> getClassHierarchy(JavaComponent javaComponent) {
        return JavaReflectionUtil.getClassHierarchy(
                javaComponent.className() != null ? javaComponent.className() : getClassForComponent(javaComponent));
    }

    private static String removeInitialDash(String s) {
        if (s.startsWith("-")) {
            return s.substring(1);
        }
        return s;
    }

    /**
     * Rewrites the value of a property for a specific component type.
     *
     * @param componentType
     *            the component type
     * @param prop
     *            the property name
     * @param value
     *            the property value
     * @return the rewritten value or the original value if no rewriting is needed
     */
    public static Object componentSpecificValueMapping(Class<?> componentType, String prop, Object value) {
        String itemType = getListSetterType(componentType, prop);
        if (itemType != null) {

            // Handling the AvatarGroup items, it has a Hashmapt{type:ARRAY,
            // object:ArrayList} where the ArrayList
            // contains the items, which are AvatarGroupItems
            if (value instanceof Map<?, ?> && ((Map<?, ?>) value).containsKey("object")) {
                // transform the value to list
                value = ((Map<?, ?>) value).get("object");
            }

            List itemList = (List) value;
            Object firstItem = itemList.isEmpty() ? null : itemList.get(0);
            if (firstItem instanceof Map) {
                return mapToClass(itemList, itemType, null);
            } else if (firstItem instanceof List) {
                return ((List<List<?>>) itemList).stream().map(item -> {
                    Map<String, Object> props = new HashMap<>();
                    props.put("x", item.get(0));
                    props.put("y", item.get(1));
                    return new JavaComponent(null, itemType, props, new ArrayList<>());
                }).toList();
            }
        }

        return value;
    }

    private static String getListSetterType(Class<?> componentType, String prop) {
        // These could be detected using reflection
        if (componentType.getName().equals(AVATAR_GROUP) && prop.equals(ITEMS)) {
            return AVATAR_GROUP_ITEM;
        } else if (componentType.getName().equals(MESSAGE_LIST) && prop.equals(ITEMS)) {
            return MESSAGE_LIST_ITEM;
        } else if (componentType.getName().equals(CHARTS_DATA_SERIES) && prop.equals("values")) {
            return CHARTS_DATA_SERIES_ITEM;
        }
        return null;
    }

    private static List<JavaComponent> mapToClass(List<Map<String, Object>> value, String itemClass,
            BiFunction<String, Object, Object> customMapper) {
        return value.stream().map(item -> {
            Map<String, Object> props = new HashMap<>();
            for (Map.Entry<String, Object> entry : item.entrySet()) {
                Object propValue = entry.getValue();
                String property = entry.getKey();
                if (property.equals("z")) {
                    // Skip charts Z for now as it requires changing the used
                    // list class
                    continue;
                }
                if (customMapper != null) {
                    propValue = customMapper.apply(property, propValue);
                }
                props.put(property, propValue);
            }

            return new JavaComponent(null, itemClass, props, new ArrayList<>());
        }).toList();
    }

    /**
     * Converts a React property name to a Java setter method name.
     *
     * <p>
     * Used to map React properties to Java setters when the react property API does
     * not match the Java API.
     *
     * @param property
     *            the property name
     * @param type
     *            the component type
     * @return the Java setter method name or {@code null} if no mapping is found
     */
    public static String convertReactPropertyToJavaSetter(String property, Class<?> type) {
        if (property.equals("theme")) {
            if (JavaRewriterUtil.hasMethod(type, "setThemeName")) {
                return "setThemeName";
            }
            return "getElement().getThemeList().add";
        }

        String typeName = type.getName();
        if (property.equals("checked") && typeName.equals("com.vaadin.flow.component.checkbox.Checkbox")) {
            return "setValue";
        }
        if (property.equals("summary") && (typeName.equals("com.vaadin.flow.component.accordion.AccordionPanel")
                || typeName.equals("com.vaadin.flow.component.details.Details"))) {
            return "setSummaryText";
        }
        if (property.equals("img") && typeName.equals("com.vaadin.flow.component.avatar.Avatar")) {
            return "setImage";
        }
        if (property.equals("img") && typeName.equals("com.vaadin.flow.component.avatar.AvatarGroup$AvatarGroupItem")) {
            return "setImage";
        }
        if (property.equals("abbr") && (typeName.equals("com.vaadin.flow.component.avatar.Avatar")
                || typeName.equals("com.vaadin.flow.component.avatar.AvatarGroup$AvatarGroupItem"))) {
            return "setAbbreviation";
        }
        if (property.equals("userAbbr") && (typeName.equals("com.vaadin.flow.component.messages.MessageListItem"))) {
            return "setUserAbbreviation";
        }
        if (property.equals("nodrop") && typeName.equals("com.vaadin.flow.component.upload.Upload")) {
            return "setDropAllowed";
        }
        if (property.equals("dataProvider")) {
            return "setItems";
        }
        if (property.equals("items") && typeName.equals("com.vaadin.flow.component.menubar.MenuBar")) {
            return "addItem";
        }
        if (property.equals("values") && typeName.equals(CHARTS_LIST_SERIES)) {
            return "setData";
        }
        if (property.equals("values") && typeName.equals(CHARTS_DATA_SERIES)) {
            return "setData";
        }
        if (property.equals("title") && (typeName.equals(CHARTS_LIST_SERIES) || typeName.equals(CHARTS_DATA_SERIES))) {
            return "setName";
        }
        if (property.equals("type") && typeName.equals(CHARTS_LIST_SERIES)) {
            return "setPlotOptions";
        }
        if (property.equals("footer") && typeName.equals(CARD)) {
            return "addToFooter";
        }
        return null;
    }

    public static String getInnerTextProperty(Class<?> componentType) {
        String typeName = componentType.getName();
        if (typeName.equals("com.vaadin.flow.component.tabs.Tab")
                || typeName.equals("com.vaadin.flow.component.sidenav.SideNavItem")) {
            return "label";
        }
        return "text";
    }

    public static boolean isInvertedBoolean(String property, Class<?> type) {
        return property.equals("nodrop") && type.getName().equals("com.vaadin.flow.component.upload.Upload");
    }

    /**
     * Returns true if child is not a component but an item that is used in
     * DataProvider.
     *
     * @param parent
     *            Parent component
     * @param child
     *            Child definition
     * @return true if it is, false otherwise.
     */
    public static boolean childrenGeneratesData(JavaComponent parent, JavaComponent child) {
        if ("ListBox".equals(parent.tag()) && "Item".equals(child.tag())) {
            return true;
        }
        if ("RadioGroup".equals(parent.tag()) && "RadioButton".equals(child.tag())) {
            return true;
        }
        return "CheckboxGroup".equals(parent.tag()) && "Checkbox".equals(child.tag());
    }

    /**
     * Checks the given component has items property and supported.
     *
     * @param component
     *            component which has tag and props.
     * @return true if component tag is listed in supported list and has items
     *         property, false otherwise.
     */
    public static boolean hasSetItemsProps(JavaComponent component) {
        if (!component.props().containsKey(ITEMS) && !component.props().containsKey("dataProvider")) {
            return false;
        }
        return hasSetItemsMethod(component);
    }

    public static boolean hasSetItemsMethod(JavaComponent component) {
        return component.tag() != null
                && Set.of("Select", "ComboBox", "MultiSelectComboBox", "Grid", "TreeGrid").contains(component.tag());
    }

    public static boolean skipProps(JavaComponent component, String propKey) {
        if ("ListBox".equals(component.tag()) && propKey.equals("selected")) {
            // We should not skip this, but it needs to be implemented in some
            // other way through setValue
            return true;
        }
        if ("Message".equals(component.tag()) && propKey.equals("className")) {
            // MessageListItem has no class name setter
            return true;
        }
        if (propKey.equals("slot")) {
            // Parent must handle this;
            return true;
        }
        if (propKey.equals("stacking") && "Chart".equals(component.tag())) {
            // Stacking is not handled currently, but it should be something like
            // PlotOptionsBar plotOptions = new PlotOptionsBar();
            // plotOptions.setStacking(Stacking.NORMAL);
            // chart.getConfiguration().setPlotOptions(plotOptions);
            return true;
        }

        return hasSetItemsProps(component) && (propKey.equals("items") || propKey.equals("dataProvider"));
    }

    /**
     * Returns the list of children of the given JavaComponent that should be added
     * as method calls
     *
     * @param javaComponent
     *            The JavaComponent
     * @return The list of children to be added as method calls
     */
    public static List<JavaComponent> getMethodCallChildren(JavaComponent javaComponent) {
        List<JavaComponent> methodCallChildren = new ArrayList<>();
        if (javaComponent.tag() != null
                && (javaComponent.tag().equalsIgnoreCase("Grid") || javaComponent.tag().equalsIgnoreCase("TreeGrid"))) {
            methodCallChildren.addAll(javaComponent.children().stream().map(child -> {
                if (child.tag() != null && (isGridColumnDefinition(child) || isGridColumnAttribute(child))) {
                    return child;
                }
                return null;
            }).toList());
        }
        return methodCallChildren;
    }

    /**
     * Returns the name of the setter method to be called on the owner object for
     * the given JavaComponent
     *
     * @param child
     *            The JavaComponent
     * @return The name of the setter method
     */
    public static String getSetterNameFromComponent(JavaComponent child) {
        if (child == null || child.tag() == null) {
            return null;
        }
        if (child.tag().equalsIgnoreCase(GRID_SORT_COLUMN)) {
            return "setSortable";
        } else if (child.tag().equals(GRID_SELECTION_COLUMN)) {
            return "setSelectionMode";
        }
        return null;
    }

    /**
     * Returns the MethodCallExpr to be added to the owner object for the given
     * JavaComponent
     *
     * @param child
     *            The JavaComponent
     * @param owner
     *            The owner object
     * @param dataEntityRecordName
     *            The name of the data entity record
     * @return The MethodCallExpr
     */
    public static List<MethodCallExpr> getMethodCallExprFromComponent(CompilationUnit compilationUnit,
            JavaComponent child, Expression owner, String dataEntityRecordName) {

        List<MethodCallExpr> expressions = new ArrayList<>();
        if (child == null || child.tag() == null) {
            return expressions;
        }
        if (isGridColumnDefinition(child)) {
            // Create the method reference expression: ClassName::methodName
            MethodReferenceExpr methodReference = new MethodReferenceExpr();
            methodReference.setScope(new NameExpr(dataEntityRecordName));
            methodReference.setIdentifier(child.props().get("path").toString());

            MethodCallExpr expression;
            if (child.tag().equalsIgnoreCase(GRID_TREE_COLUMN)) {
                expression = new MethodCallExpr(owner, "addHierarchyColumn").addArgument(methodReference)
                        .asMethodCallExpr();
            } else {
                expression = new MethodCallExpr(owner, "addColumn").addArgument(methodReference).asMethodCallExpr();
            }
            MethodCallExpr expressionHeader = new MethodCallExpr(expression, "setHeader")
                    .addArgument(new StringLiteralExpr(child.props().get("path").toString())).asMethodCallExpr();
            expressions.add(expressionHeader);
        } else if (isGridColumnAttribute(child) && child.tag().equalsIgnoreCase(GRID_SELECTION_COLUMN)) {
            FieldAccessExpr selectionModeExpr = new FieldAccessExpr(
                    new FieldAccessExpr(new NameExpr("Grid"), "SelectionMode"), "MULTI");
            expressions.add(
                    new MethodCallExpr(owner, "setSelectionMode").addArgument(selectionModeExpr).asMethodCallExpr());
            // Create dummy listener
            String lambdaString = """
                    selection -> {
                        System.out.printf("Number of selected people: %s%n",
                        selection.getAllSelectedItems().size());
                    }""";
            LambdaExpr lambdaExpr = StaticJavaParser.parseExpression(lambdaString).asLambdaExpr();
            expressions
                    .add(new MethodCallExpr(owner, "addSelectionListener").addArgument(lambdaExpr).asMethodCallExpr());
        }
        return expressions;
    }

    /**
     * Returns true when the java component indicates a Grid column definition
     * according to Hilla React template. Grid column components are GridColumn,
     * GridSortColumn and GridTreeColumn. If this component are present for columns
     * defined for a Grid.
     *
     * @param component
     *            The JavaComponent
     * @return true if the component is a Grid column component, false otherwise
     */
    public static boolean isGridColumnDefinition(JavaComponent component) {
        return component.tag() != null
                && (component.tag().equalsIgnoreCase(GRID_COLUMN) || component.tag().equalsIgnoreCase(GRID_SORT_COLUMN)
                        || component.tag().equalsIgnoreCase(GRID_TREE_COLUMN));
    }

    /**
     * Returns true when the java component indicates a Tree Grid column definition
     * according to Hilla React template. Tree Grid column attribute components is
     * GridTreeColumn.
     *
     * @param component
     *            The JavaComponent
     * @return true if the component is a TreeGrid column attribute, false otherwise
     */
    public static boolean isGridTreeColumnDefinition(JavaComponent component) {
        return component.tag() != null && (component.tag().equalsIgnoreCase(GRID_TREE_COLUMN));
    }

    /**
     * Returns true when the java component indicates a Grid attribute according to
     * Hilla React template. Grid attribute components so far is
     * GridSelectionColumn.
     *
     * @param component
     *            The JavaComponent
     * @return true if the component is a Grid attribute, false otherwise
     */
    public static boolean isGridColumnAttribute(JavaComponent component) {
        return component.tag() != null && (component.tag().equalsIgnoreCase(GRID_SELECTION_COLUMN));
    }

    public static boolean isTabSheetDefinition(JavaComponent component) {
        return component.tag() != null && (component.tag().equalsIgnoreCase(TAB_SHEET_TAB));
    }

    /**
     * Specific method to handle MenuBar items property. Maybe in the future it can
     * be reused for other components.
     */
    public static void menuBarInsertItemsPropsToAddItem(JavaComponent javaComponent, InsertionPoint insertionPoint,
            Expression owner, Expression parent, String setterName, Object value, boolean submenu) {

        if (value instanceof LinkedHashMap<?, ?> map) {
            if (!map.containsKey("text")) {
                // Only support text
                return;
            }
            String menuItemVarName = generateVariableName(map, MENU_ITEM, insertionPoint);
            String subMenuVarName = generateVariableName(map, SUB_MENU, insertionPoint);

            for (Map.Entry<?, ?> entry : map.entrySet()) {
                String key = entry.getKey().toString();
                Object val = entry.getValue();

                if (key.equalsIgnoreCase("text")) {
                    createMenuItem(insertionPoint, owner, setterName, menuItemVarName, val);
                } else if (key.equalsIgnoreCase("children") && val instanceof List<?> children) {
                    createSubMenu(insertionPoint, menuItemVarName, subMenuVarName);
                    addChildren(javaComponent, insertionPoint, subMenuVarName, owner, setterName, children);
                }
            }
        } else {
            throw new IllegalArgumentException("Invalid or not supported items structure for MenuBar with items class "
                    + value.getClass().getName());
        }
    }

    /**
     * Generates variable names with a suffix. TODO: add the suffix and prefix to
     * the actual JavaRewriterUtil method
     */
    private static String generateVariableName(LinkedHashMap<?, ?> map, String suffix, InsertionPoint insertionPoint) {
        return JavaRewriterUtil.findFreeVariableName(
                JavaRewriterUtil.getJavaIdentifier(map.get("text").toString(), 10) + suffix, insertionPoint.getBlock());
    }

    /**
     * Creates a MenuItem and adds it to the owner object.
     *
     * @param insertionPoint
     *            The insertion point
     * @param owner
     *            the menu that will add the item
     * @param setterName
     *            the setter method name
     * @param menuItemVarName
     *            the variable name for the MenuItem
     * @param val
     *            the value for the MenuItem
     */
    private static void createMenuItem(InsertionPoint insertionPoint, Expression owner, String setterName,
            String menuItemVarName, Object val) {
        ClassOrInterfaceType menuItemType = StaticJavaParser.parseClassOrInterfaceType(MENU_ITEM);
        VariableDeclarator variableDeclarator = new VariableDeclarator(menuItemType, menuItemVarName);
        MethodCallExpr addCall = new MethodCallExpr(owner, setterName,
                new NodeList<>(new StringLiteralExpr(val.toString())));
        variableDeclarator.setInitializer(addCall);
        VariableDeclarationExpr variableDeclarationExpr = new VariableDeclarationExpr(variableDeclarator);
        insertionPoint.add(new ExpressionStmt(variableDeclarationExpr));
    }

    /**
     * Creates a SubMenu and adds it to the correspondent menu item.
     *
     * @param insertionPoint
     *            The insertion point
     * @param menuItemVarName
     *            the variable name for the caller MenuItem
     * @param subMenuVarName
     *            the variable name for the new SubMenu
     */
    private static void createSubMenu(InsertionPoint insertionPoint, String menuItemVarName, String subMenuVarName) {
        ClassOrInterfaceType subMenuItemType = StaticJavaParser.parseClassOrInterfaceType(SUB_MENU);
        VariableDeclarator subMenuVariableDeclarator = new VariableDeclarator(subMenuItemType, subMenuVarName);
        MethodCallExpr getSubMenu = new MethodCallExpr(new NameExpr(menuItemVarName), "getSubMenu", new NodeList<>());
        subMenuVariableDeclarator.setInitializer(getSubMenu);
        VariableDeclarationExpr subMenuVariableDeclarationExpr = new VariableDeclarationExpr(subMenuVariableDeclarator);
        insertionPoint.add(new ExpressionStmt(subMenuVariableDeclarationExpr));
    }

    private static void addChildren(JavaComponent javaComponent, InsertionPoint insertionPoint, String subMenuVarName,
            Expression owner, String setterName, List<?> children) {
        for (Object child : children) {
            menuBarInsertItemsPropsToAddItem(javaComponent, insertionPoint, new NameExpr(subMenuVarName), owner,
                    setterName, child, true);
        }
    }

    /**
     * Returns the expression to be used to set a property in a JavaComponent. This
     * method modifies the code adding everything required to set the property like
     * intermediate variables or method calls.
     *
     * @param javaComponent
     *            The JavaComponent
     * @param parentSetterName
     *            The name of the parent setter method
     * @param setterName
     *            The name of the setter method
     * @param value
     *            The value of the property
     * @param owner
     *            The owner object where the property will be set
     * @param insertionPoint
     *            The insertion point where the expression will be added to add more
     *            required expressions different to the actual setter if needed
     * @return
     */
    public static Expression getPropertySetExpression(JavaComponent javaComponent, String parentSetterName,
            String setterName, Object value, Expression owner, InsertionPoint insertionPoint) {
        if (javaComponent.tag() != null && javaComponent.tag().equalsIgnoreCase("Chart")) {
            if (parentSetterName != null
                    && (parentSetterName.equalsIgnoreCase("xAxis") || parentSetterName.equalsIgnoreCase("yAxis"))) {
                if (setterName.equalsIgnoreCase("categories")) {
                    MethodCallExpr getConfigurationCall = new MethodCallExpr(owner, "getConfiguration",
                            new NodeList<>());
                    MethodCallExpr getAxisCall = new MethodCallExpr(getConfigurationCall, "get" + parentSetterName,
                            new NodeList<>());
                    MethodCallExpr setCategoriesCall = new MethodCallExpr(getAxisCall, "setCategories",
                            new NodeList<>(JavaRewriterUtil.toExpressionList(value)));
                    return setCategoriesCall;
                } else if (setterName.equalsIgnoreCase("title")) {
                    Map<String, String> title = (Map<String, String>) value;
                    MethodCallExpr getConfigurationCall = new MethodCallExpr(owner, "getConfiguration",
                            new NodeList<>());
                    MethodCallExpr getAxisCall = new MethodCallExpr(getConfigurationCall, "get" + parentSetterName,
                            new NodeList<>());
                    MethodCallExpr setCategoriesCall = new MethodCallExpr(getAxisCall, "setTitle",
                            new NodeList<>(JavaRewriterUtil.toExpressionList(title.get("text"))));
                    return setCategoriesCall;
                }
            } else {
                if (!JavaRewriterUtil.hasMethod(HasStyle.class, setterName)) {
                    // Many chart properties are set in the configuration
                    MethodCallExpr getConfigurationCall = new MethodCallExpr(owner, "getConfiguration",
                            new NodeList<>());
                    return getConfigurationCall;
                }
            }
        }
        if (javaComponent.tag() != null && javaComponent.tag().equalsIgnoreCase("ChartSeries")
                && setterName.equals("setPlotOptions")) {
            MethodCallExpr getSeriesSetPlotOptionsCall = new MethodCallExpr(owner, "setPlotOptions",
                    new NodeList<>(new NameExpr(createPlotOptions(value.toString(), insertionPoint))));
            return getSeriesSetPlotOptionsCall;
        } else {
            return owner;
        }
    }

    private static String createPlotOptions(String type, InsertionPoint insertionPoint) {
        ExpressionStmt plotOptions = null;
        String className = getChartClassName(type);
        String variableName = JavaRewriterUtil.findFreeVariableName(SharedUtil.firstToLower(className),
                insertionPoint.getBlock());
        VariableDeclarator variableDeclarator = new VariableDeclarator(
                StaticJavaParser.parseClassOrInterfaceType(className), variableName);
        ObjectCreationExpr objectCreationExpr = new ObjectCreationExpr();
        objectCreationExpr.setType(className);
        variableDeclarator.setInitializer(objectCreationExpr);
        VariableDeclarationExpr variableDeclarationExpr = new VariableDeclarationExpr(variableDeclarator);
        plotOptions = new ExpressionStmt(variableDeclarationExpr);
        JavaRewriterUtil.addImport(insertionPoint.getCompilationUnit(), StaticJavaParser
                .parseClassOrInterfaceType("com.vaadin.flow.component.charts.model." + className).getNameWithScope());
        insertionPoint.add(plotOptions);
        return variableName;
    }

    private static String getChartClassName(String type) {
        if (type.equalsIgnoreCase("bar")) {
            return "PlotOptionsBar";
        } else if (type.equalsIgnoreCase("line")) {
            return "PlotOptionsLine";
        } else if (type.equalsIgnoreCase("pie")) {
            return "PlotOptionsPie";
        } else if (type.equalsIgnoreCase("column")) {
            return "PlotOptionsColumn";
        } else {
            throw new IllegalArgumentException("Invalid chart type: " + type);
        }
    }
}
