package com.vaadin.copilot.customcomponent;

import static com.vaadin.copilot.javarewriter.FlowComponentQuirks.HAS_PREFIX_QUALIFIED_CLASS_NAME;
import static com.vaadin.copilot.javarewriter.FlowComponentQuirks.HAS_SUFFIX_QUALIFIED_CLASS_NAME;

import java.lang.reflect.Parameter;
import java.util.Arrays;
import java.util.List;

import com.vaadin.copilot.JavaReflectionUtil;

/**
 * A wrapper class for custom component. <br/>
 * Custom components may be defined in multiple ways. Some can be found in
 * project source, some can be imported from an external Maven module e.g.
 * Addon. This wrapper represents all custom components and type of custom
 * component can be distinguished by {@link CustomComponent#getType()} method.
 */
public interface CustomComponent {

    /**
     * Class instance of the component
     *
     * @return Class instance
     */
    Class<?> componentClass();

    /**
     * Type of custom component.
     *
     * @return type
     */
    Type getType();

    /**
     * Checks if the component extends LitTemplate.
     *
     * @return true if it is a LitTemplate, false otherwise
     */
    boolean litTemplate();

    /**
     * List of method names where children can be added
     *
     * @return method name list
     */
    default List<CustomComponentAddMethodInfo> getChildAddableMethods() {
        return Arrays.stream(componentClass().getMethods()).filter(method -> {
            if (method.getParameterCount() != 1) {
                return false;
            }
            Parameter parameter = method.getParameters()[0];
            boolean paramFlowComponentType = JavaReflectionUtil.isParamFlowComponentType(parameter);
            if (!paramFlowComponentType) {
                return false;
            }

            Class<?> declaringClass = method.getDeclaringClass();
            if (declaringClass.equals(Object.class)) {
                return false;
            }
            if (declaringClass.getPackageName().startsWith("com.vaadin.flow.component")) {
                if (method.getName().equals("setPrefixComponent")
                        && declaringClass.getName().equals(HAS_PREFIX_QUALIFIED_CLASS_NAME)) {
                    return true;
                }
                if (method.getName().equals("setSuffixComponent")
                        && declaringClass.getName().equals(HAS_SUFFIX_QUALIFIED_CLASS_NAME)) {
                    return true;
                }
                return method.getName().equals("add");
            }
            return true;
        }).map(method -> {
            Parameter parameter = method.getParameters()[0];
            String paramJavaClassName;
            if (parameter.getType().isArray()) {
                paramJavaClassName = parameter.getType().componentType().getName();
            } else {
                paramJavaClassName = parameter.getType().getName();
            }
            return new CustomComponentAddMethodInfo(method.getDeclaringClass().getSimpleName(), method.getName(),
                    paramJavaClassName);
        }).toList();
    }

    /**
     * Type of the custom component.
     */
    enum Type {
        IN_PROJECT, EXTERNAL,
    }
}
