package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Locale;

import com.vaadin.base.devserver.stats.ProjectHelpers;
import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.databind.JsonNode;

import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MachineConfiguration {

    private final JsonNode json;

    public MachineConfiguration(String json) {
        this.json = JacksonUtils.readTree(json);
    }

    public AIUsageAllowed isAIUsageAllowed() {
        return AIUsageAllowed.valueOf(getOrDefault("aiUsageAllowed", "ask").toUpperCase(Locale.ENGLISH));
    }

    public boolean isSendErrorReportsAllowed() {
        return getOrDefault("sendErrorReportsAllowed", true);
    }

    public CopilotServerClient.AIProvider getAiProvider() {
        try {
            String providerString = getOrDefault("aiProvider", "ANY");
            return CopilotServerClient.AIProvider.valueOf(providerString.toUpperCase(Locale.ENGLISH));
        } catch (Exception e) {
            getLogger().warn("Error reading ai provider configuration", e);
        }
        return CopilotServerClient.AIProvider.ANY;
    }

    private String getOrDefault(String key, String defaultValue) {
        if (json.has(key)) {
            return json.get(key).asText();
        }
        return defaultValue;
    }

    private boolean getOrDefault(String key, boolean defaultValue) {
        if (json.has(key)) {
            return json.get(key).asBoolean();
        }
        return defaultValue;
    }

    public JsonNode getJson() {
        return json;
    }

    public static MachineConfiguration get() throws CopilotException {
        File confFile = getFile();
        if (confFile.exists()) {
            try {
                String json = FileUtils.readFileToString(confFile, StandardCharsets.UTF_8);
                getLogger().debug("Reading configuration: {}", json);
                return new MachineConfiguration(json);
            } catch (IOException e) {
                throw new CopilotException("Unable to read machine configuration", e);
            }
        }
        return new MachineConfiguration("{}");
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(MachineConfiguration.class);
    }

    public static File getFile() {
        return new File(ProjectHelpers.resolveVaadinHomeDirectory(), "copilot-configuration.json");
    }
}
