package com.vaadin.copilot.javarewriter;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.vaadin.copilot.ProjectFileManager;
import com.vaadin.copilot.customcomponent.CustomComponent;
import com.vaadin.copilot.customcomponent.CustomComponents;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.flow.component.internal.ComponentTracker;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Node;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Responsible for locating and aggregating information about a UI component's
 * creation and attachment within a Java project.
 * <p>
 * This class uses a {@link ProjectFileManager} and
 * {@link ComponentTypeAndSourceLocation} to determine where a component is
 * created and (optionally) where it is attached, then compiles that information
 * into a {@link ComponentInfo} object.
 */
public class ComponentInfoFinder {
    private final ProjectFileManager fileManager;
    private final ComponentTypeAndSourceLocation typeAndSourceLocation;
    private final JavaFileSourceProvider javaFileSourceProvider;

    /**
     * Constructs a {@code ComponentInfoFinder} using the default
     * {@link ProjectFileManager} instance and the given component source metadata.
     *
     * @param componentTypeAndSourceLocation
     *            metadata describing the component's type and source locations
     */
    public ComponentInfoFinder(JavaFileSourceProvider javaFileSourceProvider,
            ComponentTypeAndSourceLocation componentTypeAndSourceLocation) {
        this.fileManager = ProjectFileManager.get();
        this.typeAndSourceLocation = componentTypeAndSourceLocation;
        this.javaFileSourceProvider = javaFileSourceProvider;
    }

    /**
     * Finds and compiles information about a component's creation and (if
     * applicable) attachment in the project.
     * <p>
     * This method:
     * <ul>
     * <li>Loads the Java source file where the component is created</li>
     * <li>Extracts related creation details</li>
     * <li>If available, loads the Java source file where the component is attached
     * and extracts attachment details</li>
     * </ul>
     *
     * @return a {@link ComponentInfo} object containing the collected creation and
     *         attachment data
     * @throws IOException
     *             if any source file could not be read
     */
    public ComponentInfo find() throws IOException {
        File createLocationJavaFile = fileManager.getSourceFile(typeAndSourceLocation.getCreateLocationOrThrow());
        ComponentInfoBuilder builder = new ComponentInfoBuilder();

        findCreateLocationRelatedInfo(createLocationJavaFile, typeAndSourceLocation, builder);
        if (typeAndSourceLocation.attachLocationInProject().isPresent()) {
            File attachLocationJavaFile = fileManager.getSourceFile(typeAndSourceLocation.getAttachLocationOrThrow());
            findAttachLocationRelatedInfo(attachLocationJavaFile, typeAndSourceLocation, builder);
        }
        return builder.build();
    }

    private void findAttachLocationRelatedInfo(File file, ComponentTypeAndSourceLocation typeAndSourceLocation,
            ComponentInfoBuilder builder) throws IOException {
        JavaSource javaSource = javaFileSourceProvider.getJavaSource(file);
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        Optional<BlockStmt> componentAttachScope;
        AttachExpression attachCall;
        boolean compositeContainer = false;

        boolean inSameFileWithCreateLocation = typeAndSourceLocation.getCreateLocationOrThrow().filename()
                .equals(typeAndSourceLocation.getAttachLocationOrThrow().filename());

        Optional<MethodCallExpr> attachMethodCall = typeAndSourceLocation.attachLocationInProject()
                .flatMap(location -> JavaRewriterUtil.findNodeOfType(compilationUnit, location.lineNumber(),
                        MethodCallExpr.class));
        if (attachMethodCall.isEmpty() && !builder.isClassSource()) {
            Optional<ObjectCreationExpr> attachObjectCreationCall = typeAndSourceLocation.attachLocationInProject()
                    .flatMap(location -> JavaRewriterUtil.findNodeOfType(compilationUnit, location.lineNumber(),
                            ObjectCreationExpr.class));
            if (attachObjectCreationCall.isEmpty() && inSameFileWithCreateLocation) {
                // TODO somehow we need to obtain localVariableName and field name when create
                // location and attach location are not in the same file.
                ComponentCreateInfo componentCreateInfo = builder.getCreateInfoOrThrow();
                String varName = componentCreateInfo.getLocalVariableName() != null
                        ? componentCreateInfo.getLocalVariableName()
                        : componentCreateInfo.getFieldName();

                Optional<AttachExpression> possibleAttachExpressionFromParentInfo = findPossibleAttachExpressionFromParentInfo(
                        typeAndSourceLocation, componentCreateInfo.getObjectCreationExpr(), varName);

                if (possibleAttachExpressionFromParentInfo.isEmpty()) {
                    throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                            "Attach not found at the expected location");
                }
                attachCall = possibleAttachExpressionFromParentInfo.get();
                componentAttachScope = Optional
                        .of(JavaRewriterUtil.findBlock(possibleAttachExpressionFromParentInfo.get().getNode()));
            } else {
                componentAttachScope = attachObjectCreationCall.map(JavaRewriterUtil::findBlock);
                attachCall = new AttachExpression(attachObjectCreationCall.orElse(null));
            }
        } else {
            componentAttachScope = attachMethodCall.map(JavaRewriterUtil::findBlock);
            attachCall = new AttachExpression(attachMethodCall.orElse(null));
        }

        compositeContainer = isChildOfCompositeContainer(attachCall.expression());

        if (compositeContainer && attachCall.expression().isMethodCallExpr()) {
            MethodCallExpr methodCallExpr = attachCall.getMethodCallExpression();
            if (methodCallExpr != null && methodCallExpr.getNameAsString().equals("getContent")
                    && methodCallExpr.getParentNode().isPresent()) {
                attachCall = new AttachExpression((Expression) methodCallExpr.getParentNode().get());
            }
        }
        ComponentAttachInfo attachInfo = new ComponentAttachInfo(javaSource, file);
        attachInfo.setAttachCall(attachCall);
        attachInfo.setComponentAttachScope(componentAttachScope.orElse(null));
        builder.createAndAttachLocationsAreInSameFile(inSameFileWithCreateLocation);
        builder.componentAttachInfo(attachInfo);
        builder.containerComposite(compositeContainer);

        if (inSameFileWithCreateLocation) {
            ComponentCreateInfo createInfo = builder.getCreateInfoOrThrow();
            attachInfo.setFieldName(createInfo.getFieldName());
            attachInfo.setFieldDeclaration(createInfo.getFieldDeclaration());
            attachInfo.setFieldDeclarationAndAssignment(createInfo.getFieldDeclarationAndAssignment());
            attachInfo.setLocalVariableName(createInfo.getLocalVariableName());
            attachInfo.setLocalVariableDeclarator(createInfo.getLocalVariableDeclarator());

        }
    }

    private void findCreateLocationRelatedInfo(File file, ComponentTypeAndSourceLocation typeAndSourceLocation,
            ComponentInfoBuilder builder) throws IOException {
        JavaSource javaSource = javaFileSourceProvider.getJavaSource(file);
        CompilationUnit compilationUnit = javaSource.getCompilationUnit();
        List<ObjectCreationExpr> objectCreationExprs = new ArrayList<>();
        for (Class clazz : typeAndSourceLocation.inheritanceChain()) {
            Optional<ComponentTracker.Location> maybeLocationInProject = typeAndSourceLocation
                    .createLocationInProject();
            if (maybeLocationInProject.isEmpty()) {
                continue;
            }
            ComponentTracker.Location createLocationInProject = maybeLocationInProject.get();
            objectCreationExprs = JavaRewriterUtil.findNodesOfType(compilationUnit,
                    createLocationInProject.lineNumber(), ObjectCreationExpr.class,
                    node -> node.getType().asClassOrInterfaceType().getName().asString().equals(clazz.getSimpleName()));
            if (!objectCreationExprs.isEmpty()) {
                break;
            }
        }

        if (objectCreationExprs.size() > 1) {
            throw new IllegalArgumentException(
                    "There are multiple components created on the given line and we are unable to determine which one to modify");
        }

        Optional<ObjectCreationExpr> objectCreationExpr = Optional
                .ofNullable(objectCreationExprs.isEmpty() ? null : objectCreationExprs.get(0));

        boolean classSource = false;
        boolean isAnonymousComponent = false;
        boolean isReturnValue = false;
        if (objectCreationExpr.isEmpty()) {
            classSource = JavaRewriterUtil.isLocationRefersSource(typeAndSourceLocation, compilationUnit);
            if (!classSource) {
                int lineNumber = typeAndSourceLocation.getCreateLocationOrThrow().lineNumber();
                String[] lines = javaSource.getSource().split("[\r\n]");
                String lineContents;
                if (lines.length > lineNumber) {
                    lineContents = "\"" + lines[lineNumber - 1].trim() + "\"";
                } else {
                    lineContents = "no line with number " + lineNumber + " as the file only has " + lines.length
                            + " lines";
                }
                throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                        "Expected to find an object creation expression such as \"new "
                                + typeAndSourceLocation.type().getSimpleName() + "()\" at "
                                + typeAndSourceLocation.javaFile().map(File::getAbsolutePath).orElse("?") + ":"
                                + lineNumber + " but found " + lineContents);
            }
        } else {
            ObjectCreationExpr n = objectCreationExpr.orElse(null);
            Node parent = n.getParentNode().orElse(null);
            if (n != null && parent != null && !(parent instanceof VariableDeclarator || parent instanceof AssignExpr
                    || parent instanceof FieldDeclaration)) {
                isAnonymousComponent = true;
            }
            if (n != null && parent != null && (parent instanceof ReturnStmt)) {
                isReturnValue = true;
            }
        }
        Optional<BlockStmt> componentCreateScope = objectCreationExpr.map(JavaRewriterUtil::findBlock);

        Optional<VariableDeclarator> localVariableDeclarator = objectCreationExpr
                .map(expr -> JavaRewriterUtil.findAncestor(expr, VariableDeclarator.class));
        Optional<AssignExpr> assignmentExpression = objectCreationExpr
                .map(expr -> JavaRewriterUtil.findAncestor(expr, AssignExpr.class));
        Optional<FieldDeclaration> fieldDeclarationAndAssignment = objectCreationExpr
                .map(expr -> JavaRewriterUtil.findAncestor(expr, FieldDeclaration.class));
        Optional<FieldDeclaration> fieldDeclaration = fieldDeclarationAndAssignment;
        if (localVariableDeclarator.isPresent() && fieldDeclarationAndAssignment.isPresent()) {
            // A variable declarator is found also for assignments for fields
            // but we want to differentiate between the two cases later on
            localVariableDeclarator = Optional.empty();
        }

        String localVariableName = null;
        String fieldName = null;
        if (localVariableDeclarator.isPresent()) {
            // TextField foo = new TextField();
            localVariableName = localVariableDeclarator.get().getNameAsString();
        } else if (fieldDeclarationAndAssignment.isPresent()) {
            // private TextField foo = new TextField();
            fieldName = fieldDeclarationAndAssignment.get().getVariable(0).getNameAsString();
        } else if (assignmentExpression.isPresent()) {
            // foo = new TextField();
            // Here foo can be a local variable or field
            String localVariableOrFieldName;
            Expression target = assignmentExpression.get().getTarget();
            if (target.isNameExpr()) {
                localVariableOrFieldName = target.asNameExpr().getNameAsString();
            } else if (target.isFieldAccessExpr()) {
                localVariableOrFieldName = target.asFieldAccessExpr().getNameAsString();
            } else {
                throw new IllegalArgumentException("Unhandled target type for assignment. Expression=" + target);
            }

            if (componentCreateScope.isPresent() && JavaRewriterUtil
                    .findLocalVariableDeclarator(localVariableOrFieldName, componentCreateScope.get()) != null) {
                localVariableName = localVariableOrFieldName;
            } else {
                fieldName = localVariableOrFieldName;
                fieldDeclaration = Optional
                        .ofNullable(JavaRewriterUtil.findFieldDeclaration(assignmentExpression.get(), fieldName));
            }
        }
        Optional<ConstructorDeclaration> targetConstructorDeclaration = Optional.empty();
        if (classSource) {
            targetConstructorDeclaration = findTargetConstructorDeclaration(compilationUnit, typeAndSourceLocation);
        }
        Optional<CustomComponent> customComponentInfo = CustomComponents
                .getCustomComponentInfo(typeAndSourceLocation.type());

        boolean expressionInLoop = JavaRewriterUtil.checkExpressionInLoop(objectCreationExpr.orElse(null))
                || JavaRewriterUtil.checkMethodHasExpressionCalledInLoop(objectCreationExpr.orElse(null));

        ComponentCreateInfo componentCreateInfo = new ComponentCreateInfo(javaSource, file);
        componentCreateInfo.setFieldDeclaration(fieldDeclaration.orElse(null));
        componentCreateInfo.setFieldDeclarationAndAssignment(fieldDeclarationAndAssignment.orElse(null));
        componentCreateInfo.setFieldName(fieldName);
        componentCreateInfo.setComponentCreateScope(componentCreateScope.orElse(null));
        componentCreateInfo.setLocalVariableDeclarator(localVariableDeclarator.orElse(null));
        componentCreateInfo.setLocalVariableName(localVariableName);
        componentCreateInfo.setObjectCreationExpr(objectCreationExpr.orElse(null));
        componentCreateInfo.setAssignmentExpression(assignmentExpression.orElse(null));

        builder.type(typeAndSourceLocation.type()).customComponentInfo(customComponentInfo.orElse(null))
                .routeConstructor(targetConstructorDeclaration.orElse(null)).componentCreateInfo(componentCreateInfo)
                .isReturnValue(isReturnValue).classSource(classSource).isAnonymousComponent(isAnonymousComponent)
                .createdInLoop(expressionInLoop);

    }

    private Optional<ConstructorDeclaration> findTargetConstructorDeclaration(CompilationUnit compilationUnit,
            ComponentTypeAndSourceLocation typeAndSourceLocation) {
        ComponentTracker.Location createLocation = typeAndSourceLocation.getCreateLocationOrThrow();
        Optional<ConstructorDeclaration> targetConstructorDeclaration = JavaRewriterUtil.findNodeOfType(compilationUnit,
                createLocation.lineNumber(), ConstructorDeclaration.class);
        if (targetConstructorDeclaration.isPresent()) {
            return targetConstructorDeclaration;
        }
        if (createLocation.className().contains("$")) {
            Optional<ClassOrInterfaceDeclaration> maybeClassDeclaration = JavaRewriterUtil
                    .findNodeOfType(compilationUnit, createLocation.lineNumber(), ClassOrInterfaceDeclaration.class);
            // it is possible that createLocation would have the line number where class
            // declaration presents.
            if (maybeClassDeclaration.isPresent()) {
                targetConstructorDeclaration = Optional.of(new ConstructorDeclaration()
                        .setName(typeAndSourceLocation.type().getSimpleName()).setPublic(true));
                ClassOrInterfaceDeclaration classOrInterfaceDeclaration = maybeClassDeclaration.get();
                classOrInterfaceDeclaration.addMember(targetConstructorDeclaration.get());
                return targetConstructorDeclaration;
            }
        }
        // Potentially no constructor
        List<ConstructorDeclaration> constructors = compilationUnit.findAll(ConstructorDeclaration.class);
        if (constructors.isEmpty()) {
            // Need to create a constructor
            targetConstructorDeclaration = Optional.of(
                    new ConstructorDeclaration().setName(typeAndSourceLocation.type().getSimpleName()).setPublic(true));
            String className = typeAndSourceLocation.type().getSimpleName();
            ClassOrInterfaceDeclaration classDeclaration = compilationUnit.findAll(ClassOrInterfaceDeclaration.class)
                    .stream().filter(c -> c.getNameAsString().equals(className)).findFirst()
                    .orElseThrow(() -> new IllegalArgumentException("Class " + className + " not found"));
            classDeclaration.addMember(targetConstructorDeclaration.get());
        } else {
            throw new ComponentInfoNotFoundException(typeAndSourceLocation,
                    "Route class has constructors but none at the expected location: " + createLocation);
        }
        return targetConstructorDeclaration;
    }

    private boolean isChildOfCompositeContainer(Expression attachCallExpr) {
        if (attachCallExpr == null) {
            return false;
        }
        return JavaRewriterUtil.isNodeInCompositeClass(attachCallExpr);
    }

    /**
     * To find dynamically generated component attach location.
     *
     * @param componentTypeAndSourceLocation
     *            component location
     * @param varName
     *            local or field variable name
     * @param objectCreationExpr
     *            creation expression is used for inline variables.
     * @return returns empty if parent is null or argument variables are not present
     */
    private Optional<AttachExpression> findPossibleAttachExpressionFromParentInfo(
            ComponentTypeAndSourceLocation componentTypeAndSourceLocation, ObjectCreationExpr objectCreationExpr,
            String varName) {
        if (componentTypeAndSourceLocation.parent() == null) {
            return Optional.empty();
        }
        List<Expression> possibleArgs = new ArrayList<>();
        if (varName != null) {
            possibleArgs.add(new NameExpr(varName));
        }
        if (objectCreationExpr != null) {
            possibleArgs.add(objectCreationExpr);
        }
        if (possibleArgs.isEmpty()) {
            return Optional.empty();
        }
        ComponentTypeAndSourceLocation parent = componentTypeAndSourceLocation.parent();

        ComponentInfo componentInfo;
        try {
            ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(this.javaFileSourceProvider, parent);
            componentInfo = componentInfoFinder.find();
        } catch (IOException e) {
            getLogger().warn("Unable to find component info for {}", componentTypeAndSourceLocation.type());
            return Optional.empty();
        }

        List<MethodCallExpr> methodCallStatements = JavaRewriterUtil.findMethodCallStatements(componentInfo);

        return methodCallStatements.stream().filter(
                f -> f.getNameAsString().equals("add") && f.getArguments().stream().anyMatch(possibleArgs::contains))
                .findFirst().map(AttachExpression::new);
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
