package com.vaadin.copilot.analytics;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Collections;
import java.util.HashMap;
import java.util.HexFormat;
import java.util.Map;

import com.vaadin.base.devserver.stats.ProjectHelpers;
import com.vaadin.copilot.Copilot;
import com.vaadin.copilot.CopilotException;
import com.vaadin.copilot.CopilotServerClient;
import com.vaadin.copilot.CopilotVersion;
import com.vaadin.copilot.userinfo.UserInfo;
import com.vaadin.copilot.userinfo.UserInfoServerClient;
import com.vaadin.pro.licensechecker.LocalProKey;
import com.vaadin.pro.licensechecker.MachineId;
import com.vaadin.pro.licensechecker.ProKey;

/**
 * Client for tracking features of Copilot
 */
public final class AnalyticsClient extends CopilotServerClient {

    private static AnalyticsClient instance;

    private AnalyticsUserInfo analyticsUserInfo;

    private record AnalyticsUserInfo(String userId, String machineId, String proKeySha256, boolean vaadiner,
            String email) {
    }

    private static boolean enabled;

    private Map<String, String> userContext;

    /**
     * Constructor loading ProKey
     */
    private AnalyticsClient() {
    }

    /**
     * Get the singleton instance.
     *
     * @return the singleton instance of the client, never null
     */
    public static synchronized AnalyticsClient getInstance() {
        if (instance == null) {
            instance = new AnalyticsClient();
        }
        return instance;
    }

    /**
     * Set whether tracking is enabled.
     * <p>
     * Should only ever be called from
     * {@link com.vaadin.copilot.CopilotIndexHtmlLoader}.
     *
     * @param enabled
     *            true if tracking should be enabled, false otherwise
     */
    public static void setEnabled(boolean enabled) {
        AnalyticsClient.enabled = enabled;
    }

    /**
     * Check if tracking is enabled.
     *
     * @return true if tracking is enabled, false otherwise
     */
    public static boolean isEnabled() {
        return enabled;
    }

    public void setUserContext(Map<String, String> userContext) {
        this.userContext = Collections.unmodifiableMap(userContext);
    }

    /**
     * Track event to copilot-server
     *
     * @param event
     *            Event name
     * @param properties
     *            Map of event properties
     */
    public void track(String event, Map<String, String> properties) {
        if (!enabled) {
            return;
        }

        String prefixedEvent = event.startsWith(Copilot.PREFIX) ? event : Copilot.PREFIX + event;
        properties = properties == null ? new HashMap<>() : new HashMap<>(properties);
        properties.put("Vaadiner", String.valueOf(getAnalyticsUserInfo().vaadiner()));
        properties.put("Version", CopilotVersion.getVersion());
        if (getAnalyticsUserInfo().email() != null) {
            properties.put("Email", getAnalyticsUserInfo().email());
        }
        properties.put("Pro", getAnalyticsUserInfo().proKeySha256());
        sendRequest(prefixedEvent, properties, userContext);
    }

    private AnalyticsUserInfo getAnalyticsUserInfo() {
        if (analyticsUserInfo == null) {
            String machineId = MachineId.get();
            String userId = ProjectHelpers.getUserKey();
            ProKey proKey = LocalProKey.get();
            UserInfo userInfo = UserInfoServerClient.getUserInfoWithLocalProKey();
            if (userInfo != null) {
                analyticsUserInfo = new AnalyticsUserInfo(userId, machineId, toSha256Hex(proKey.getProKey()),
                        userInfo.vaadiner(), userInfo.email());
            } else {
                analyticsUserInfo = new AnalyticsUserInfo(userId, machineId, "null", false, null);
            }
        }
        return analyticsUserInfo;
    }

    protected void sendRequest(String event, Map<String, String> properties, Map<String, String> context) {
        AnalyticsRequest trackingRequest = new AnalyticsRequest(getAnalyticsUserInfo().userId(),
                getAnalyticsUserInfo().machineId(), event, properties, context);
        String json = writeAsJsonString(trackingRequest);
        sendReactive("analytics", json);
    }

    private String toSha256Hex(String text) {
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] proKeyDigest = digest.digest(text.getBytes(StandardCharsets.UTF_8));
            return HexFormat.of().formatHex(proKeyDigest);
        } catch (NoSuchAlgorithmException e) {
            throw new CopilotException(e);
        }
    }

}
