package com.vaadin.copilot;

import java.lang.reflect.Method;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.internal.hilla.EndpointRequestUtil;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonBoolean;
import elemental.json.JsonNumber;
import elemental.json.JsonObject;
import elemental.json.JsonValue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides endpoint information to the client.
 */
public class UiServiceHandler extends CopilotCommand {

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-browser-callables")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));
            List<SpringBridge.ServiceMethodInfo> browserCallables = EndpointRequestUtil.isHillaAvailable()
                    ? SpringBridge.getEndpoints(getVaadinContext())
                    : new ArrayList<>();

            JsonArray browserCallableJson = browserCallables.stream().sorted(UiServiceHandler::sortByClassAndMethodName)
                    .map(serviceMethod -> serviceMethodToJson(serviceMethod, true)).collect(JsonUtils.asArray());
            returnData.put("browserCallables", browserCallableJson);

            JsonArray flowUiServicesJson;
            if (SpringBridge.isSpringAvailable(getVaadinContext())) {
                List<SpringBridge.ServiceMethodInfo> flowUIServices = SpringBridge
                        .getFlowUIServices(getVaadinContext());
                flowUiServicesJson = flowUIServices.stream().sorted(UiServiceHandler::sortByClassAndMethodName)
                        .map(serviceMethod -> serviceMethodToJson(serviceMethod, false)).collect(JsonUtils.asArray());
            } else {
                flowUiServicesJson = Json.createArray();
            }

            returnData.put("flowServices", flowUiServicesJson);

            devToolsInterface.send("resp" + command, returnData);
            return true;
        } else if (command.equals("create-service-entity")) {
            JsonObject responseData = Json.createObject();
            responseData.put("reqId", data.getString("reqId"));
            try {
                Path referenceFile = Util.findCurrentViewFile(getVaadinSession(), data.getObject("currentView"))
                        .orElseThrow().toPath();
                JavaSourcePathDetector.ModuleInfo moduleInfo = getProjectFileManager()
                        .findModule(referenceFile.toFile()).orElseThrow();
                String mainPackage = Util
                        .getPackageName(SpringBridge.getApplicationClass(getVaadinContext()).getName());

                String entityPackage = Util.decideEntityPackage(moduleInfo, referenceFile, mainPackage);
                String repositoryPackage = Util.decideRepositoryPackage(moduleInfo, referenceFile, mainPackage);
                String servicePackage = Util.decideServicePackage(moduleInfo, referenceFile, mainPackage);
                String beanName = data.getString("beanName");
                String qualifiedBeanName = entityPackage + "." + beanName;
                UIServiceCreator.FieldInfo[] properties = JsonUtils.stream(data.getArray("properties"))
                        .map(JsonObject.class::cast).map(obj -> UIServiceCreator.FieldInfo.fromJson(obj))
                        .toArray(UIServiceCreator.FieldInfo[]::new);
                UIServiceCreator.BeanInfo beanInfo = new UIServiceCreator.BeanInfo(qualifiedBeanName, properties);
                UIServiceCreator.DataStorage dataStorage = "jpa".equals(data.getString("dataStorage"))
                        ? UIServiceCreator.DataStorage.JPA
                        : UIServiceCreator.DataStorage.IN_MEMORY;
                List<Map<String, Object>> exampleData = null;
                if (data.hasKey("exampleData")) {
                    exampleData = JsonUtils.stream(data.getArray("exampleData")).map(JsonObject.class::cast)
                            .map(jsonObject -> {
                                Map<String, Object> exampleValues = new HashMap<>();
                                Arrays.stream(jsonObject.keys()).forEach(key -> {
                                    JsonValue value = jsonObject.get(key);
                                    if (value instanceof JsonNumber) {
                                        exampleValues.put(key, value.asNumber());
                                    } else if (value instanceof JsonBoolean) {
                                        exampleValues.put(key, value.asBoolean());
                                    } else {
                                        exampleValues.put(key, value.asString());
                                    }
                                });
                                return exampleValues;
                            }).toList();

                }
                boolean browserCallable = data.getBoolean("browserCallable");
                UIServiceCreator.ServiceAndBeanInfo serviceAndBeanInfo = new UIServiceCreator.ServiceAndBeanInfo(
                        beanInfo, dataStorage, browserCallable, exampleData, servicePackage, repositoryPackage);
                boolean restartNeeded = new UIServiceCreator().createServiceAndBean(serviceAndBeanInfo, moduleInfo);
                responseData.put("restartNeeded", restartNeeded);
                JsonObject serviceJson = Json.createObject();
                serviceJson.put("className", serviceAndBeanInfo.getServiceName());
                serviceJson.put("methodName", "list");
                JavaReflectionUtil.TypeInfo listReturnType = new JavaReflectionUtil.TypeInfo("java.util.List",
                        List.of(new JavaReflectionUtil.TypeInfo(qualifiedBeanName, List.of())));
                serviceJson.put("returnType", typeToJson(listReturnType));
                JavaReflectionUtil.ParameterTypeInfo listParameter = new JavaReflectionUtil.ParameterTypeInfo(
                        "pageable",
                        new JavaReflectionUtil.TypeInfo("org.springframework.data.domain.Pageable", List.of()));
                serviceJson.put("parameters", parametersToJson(List.of(listParameter)));
                responseData.put("service", serviceJson);
                devToolsInterface.send(command + "-resp", responseData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, responseData, "Unable to create service", e);
            }
            return true;
        }
        return false;
    }

    private JsonObject serviceMethodToJson(SpringBridge.ServiceMethodInfo serviceMethodInfo,
            boolean availableInTypescript) {
        JsonObject json = Json.createObject();
        // canonical name can be null e.g. for anonymous classes

        json.put("className", JavaReflectionUtil.getClassName(serviceMethodInfo.serviceClass()));
        json.put("filename",
                getProjectFileManager().getFileForClass(serviceMethodInfo.serviceClass()).getAbsolutePath());
        json.put("lineNumber", 1);
        json.put("methodName", serviceMethodInfo.serviceMethod().getName());
        try {
            List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes = JavaReflectionUtil
                    .getParameterTypes(serviceMethodInfo.serviceMethod(), serviceMethodInfo.serviceClass());

            json.put("parameters", parametersToJson(parameterTypes));
        } catch (Exception e) {
            getLogger().error("Unable to determine parameter types for " + serviceMethodInfo, e);
            json.put("parameters", Json.createArray());
        }
        try {
            JavaReflectionUtil.TypeInfo returnTypeInfo = JavaReflectionUtil
                    .getReturnType(serviceMethodInfo.serviceMethod(), serviceMethodInfo.serviceClass());
            json.put("returnType", typeToJson(returnTypeInfo));
        } catch (Exception e) {
            getLogger().error("Unable to determine return type for " + serviceMethodInfo, e);
            json.put("returnType", Json.createObject());
        }
        json.put("availableInTypescript", availableInTypescript);
        json.put("canBindToFlowGrid", ConnectToService.canBindFlowGridToService(serviceMethodInfo).name());
        json.put("canBindToHillaGrid", ConnectToService.canBindHillaGridToService(serviceMethodInfo).name());
        json.put("canBindToFlowComboBox", ConnectToService.canBindFlowComboBoxToService(serviceMethodInfo).name());
        json.put("canBindToHillaComboBox", ConnectToService.canBindHillaComboBoxToService(serviceMethodInfo).name());
        try {
            AccessRequirement req = AccessRequirementUtil.getAccessRequirement(serviceMethodInfo.serviceMethod(),
                    serviceMethodInfo.serviceClass());
            json.put("accessRequirement", JsonUtils.beanToJson(req));
        } catch (Exception e) {
            getLogger().error("Unable to determine access requirement", e);
        }
        return json;
    }

    private JsonArray parametersToJson(List<JavaReflectionUtil.ParameterTypeInfo> parameterTypes) {
        return parameterTypes.stream().map(param -> {
            JsonObject paramInfo = Json.createObject();
            paramInfo.put("name", param.name());
            paramInfo.put("type", typeToJson(param.type()));
            return paramInfo;
        }).collect(JsonUtils.asArray());
    }

    private JsonObject typeToJson(JavaReflectionUtil.TypeInfo typeInfo) {
        JsonObject json = Json.createObject();
        json.put("typeName", typeInfo.typeName());
        json.put("typeParameters",
                typeInfo.typeParameters().stream().map(this::typeToJson).collect(JsonUtils.asArray()));
        return json;
    }

    public static int sortByClassAndMethodName(SpringBridge.ServiceMethodInfo e1, SpringBridge.ServiceMethodInfo e2) {
        return sortByClassAndMethodName(e1.serviceClass(), e1.serviceMethod(), e2.serviceClass(), e2.serviceMethod());
    }

    private static int sortByClassAndMethodName(Class<?> class1, Method method1, Class<?> class2, Method method2) {
        if (!class1.equals(class2)) {
            return JavaReflectionUtil.getClassName(class1).compareTo(JavaReflectionUtil.getClassName(class2));
        } else {
            return method1.getName().compareTo(method2.getName());
        }

    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(UiServiceHandler.class);
    }
}
