package com.vaadin.copilot;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.HasElement;
import com.vaadin.flow.internal.JsonUtils;
import com.vaadin.flow.router.RouteData;
import com.vaadin.flow.router.RouterLayout;
import com.vaadin.flow.server.RouteRegistry;
import com.vaadin.flow.server.VaadinSession;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import elemental.json.JsonValue;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Provides server side route information to the client.
 */
public class RouteHandler extends CopilotCommand {
    private final RouteCreator routeCreator;
    private final NewRouteTemplateHandler templateFinder;

    public RouteHandler() {
        this.routeCreator = new RouteCreator(getVaadinSession());
        this.templateFinder = new NewRouteTemplateHandler();
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-routes")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));
            boolean securityEnabled = isViewSecurityEnabled();
            returnData.put("securityEnabled", securityEnabled);
            try {
                JsonArray serverRoutes = RouteHandler.getServerRoutes(getVaadinSession()).stream().map(routeData -> {
                    JsonObject route = Json.createObject();
                    Class<? extends Component> target = routeData.getNavigationTarget();
                    route.put("path", addInitialSlash(routeData.getTemplate()));
                    route.put("navigationTarget", target.getName());
                    if (securityEnabled) {
                        try {
                            AccessRequirement req = AccessRequirementUtil.getAccessRequirement(target, null);

                            route.put("accessRequirement", JsonUtils.beanToJson(req));
                        } catch (Exception e) {
                            getLogger().error("Unable to determine access requirement", e);
                        }
                    }
                    route.put("filename",
                            getProjectFileManager().getFileForClass(routeData.getNavigationTarget()).getAbsolutePath());
                    return route;
                }).collect(JsonUtils.asArray());
                returnData.put("routes", serverRoutes);
                devToolsInterface.send("server-routes", returnData);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Error getting routes", e);
            }
            return true;

        } else if (command.equals("add-route")) {
            JsonObject returnData = Json.createObject();
            returnData.put("reqId", data.getString("reqId"));

            String title = data.getString("title");
            String route = data.getString("route");
            if (route.startsWith("/")) {
                route = route.substring(1);
            }
            String template = data.getString("template");

            String framework = data.getString("framework");
            AccessRequirement.Type accessControl = AccessRequirement.Type.valueOf(data.getString("accesscontrol"));
            AccessRequirement accessRequirement;
            if (accessControl == AccessRequirement.Type.ROLES_ALLOWED) {
                accessRequirement = new AccessRequirement(accessControl,
                        JsonUtils.stream(data.getArray("roles")).map(JsonValue::asString).toArray(String[]::new));
            } else {
                accessRequirement = new AccessRequirement(accessControl);
            }
            try {
                if ("hilla".equals(framework)) {
                    routeCreator.createHillaView(title, route, accessRequirement, template);
                } else if ("flow".equals(framework)) {
                    Class<? extends HasElement> layoutClass = null;
                    if (data.hasKey("uiId")) {
                        Optional<RouteData> routeData = FlowUtil.getRouteData(getVaadinSession(),
                                (int) data.getNumber("uiId"));
                        if (routeData.isPresent()) {
                            layoutClass = routeData.get().getParentLayout();
                        }

                    }
                    routeCreator.createFlowView(title, route, accessRequirement, layoutClass, template);
                } else {
                    throw new IllegalArgumentException("Unknown framework: " + framework);
                }
                devToolsInterface.send(command + "-response", returnData);
            } catch (RouteDuplicatedException e) {
                returnData.put("errorDuplicatedRoute", true);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Route already exists", e);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, returnData, "Error adding route", e);
            }

            return true;
        } else if (command.equals("get-new-route-templates")) {
            JsonObject response = Json.createObject();
            try {
                response.put("reqId", data.getString("reqId"));
                String framework = data.getString("framework");
                List<String> templateNames = templateFinder.getTemplateNames(framework);
                JsonArray array = JsonUtils.listToJson(templateNames);
                response.put("templates", array);

                devToolsInterface.send(command + "-response", response);
            } catch (Exception e) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, response, "Error loading template views", e);

            }

            return true;
        }
        return false;
    }

    private boolean isViewSecurityEnabled() {
        return SpringBridge.isSpringAvailable(getVaadinContext())
                && SpringBridge.isViewSecurityEnabled(getVaadinContext());
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(RouteHandler.class);
    }

    private String addInitialSlash(String template) {
        if (!template.startsWith("/")) {
            return "/" + template;
        }
        return template;
    }

    public static List<RouteData> getServerRoutes(VaadinSession vaadinSession) {
        try {
            vaadinSession.lock();
            return vaadinSession.getService().getRouter().getRegistry().getRegisteredRoutes();
        } catch (Exception e) {
            getLogger().error("Error getting server routes", e);
        } finally {
            vaadinSession.unlock();
        }
        return null;
    }

    /**
     * Returns the unique layout classes that are used in given route list
     *
     * @param vaadinSession
     *            Vaadin session to access router
     * @param routeDataList
     *            List of routes
     * @return the unique classes of Layouts
     */
    public static Set<Class<? extends RouterLayout>> getLayoutClasses(VaadinSession vaadinSession,
            List<RouteData> routeDataList) {
        Set<Class<? extends RouterLayout>> layoutClasses = new HashSet<>();
        try {
            vaadinSession.lock();
            RouteRegistry registry = vaadinSession.getService().getRouter().getRegistry();
            for (RouteData routeData : routeDataList) {
                Class<? extends RouterLayout> template = registry.getLayout(routeData.getTemplate());
                if (template != null) {
                    layoutClasses.add(template);
                }
                if (routeData.getParentLayouts() != null) {
                    layoutClasses.addAll(routeData.getParentLayouts());
                }

            }
        } catch (Exception e) {
            getLogger().error("Error getting server routes", e);
        } finally {
            vaadinSession.unlock();
        }
        return layoutClasses;
    }
}
