package com.vaadin.copilot;

import static com.vaadin.copilot.ConnectToService.STRING_TYPE;
import static com.vaadin.copilot.javarewriter.JavaFileSourceProvider.UNDO_LABEL;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Stream;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.customcomponent.CustomComponentHelper;
import com.vaadin.copilot.customcomponent.CustomComponents;
import com.vaadin.copilot.ide.IdeUtils;
import com.vaadin.copilot.javarewriter.ComponentAttachInfo;
import com.vaadin.copilot.javarewriter.ComponentCreateInfo;
import com.vaadin.copilot.javarewriter.ComponentInfo;
import com.vaadin.copilot.javarewriter.ComponentInfoFinder;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.JavaBatchRewriter;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.copilot.javarewriter.JavaDataProviderHandler;
import com.vaadin.copilot.javarewriter.JavaFileSourceProvider;
import com.vaadin.copilot.javarewriter.JavaRewriter;
import com.vaadin.copilot.javarewriter.JavaRewriterCopyPasteHandler;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.copilot.javarewriter.JavaSource;
import com.vaadin.copilot.javarewriter.SourceSyncChecker;
import com.vaadin.copilot.javarewriter.exception.ComponentInfoNotFoundException;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JsonUtils;

import elemental.json.Json;
import elemental.json.JsonArray;
import elemental.json.JsonObject;
import elemental.json.JsonType;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.javaparser.ast.expr.MethodCallExpr;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles commands to rewrite Java source code.
 */
public class JavaRewriteHandler extends CopilotCommand {

    private final ObjectMapper objectMapper = new ObjectMapper();
    private static final String CAN_BE_EDITED = "canBeEdited";
    private static final String IS_TRANSLATION = "isTranslation";
    private static final String COMPONENT_ID = "componentId";
    private final SourceSyncChecker sourceSyncChecker;
    private ComponentSourceFinder sourceFinder;

    private static final String COMPONENT_PROPERTY = "component";
    private static final String PROPERTY_TO_CHECK_PROPERTY = "propertyToCheck";

    private static final String[] supportedEditableProperties = new String[] { "label", "helperText", "text", "title" };

    private interface Handler {
        void handle(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
                throws IOException;
    }

    private static class RewriteHandler {
        private final String what;
        private final Handler handler;

        public RewriteHandler(String what, Handler handler) {
            this.what = what;
            this.handler = handler;
        }

        public void handle(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
                throws IOException {
            this.handler.handle(javaFileSourceProvider, data, respData);
        }

        public String getWhat() {
            return what;
        }
    }

    private final Map<String, RewriteHandler> handlers = new HashMap<>();

    /**
     * Creates the one and only handler.
     *
     * @param sourceSyncChecker
     *            the source sync checker for detecting out of sync scenarios
     */
    public JavaRewriteHandler(SourceSyncChecker sourceSyncChecker) {
        this.sourceSyncChecker = sourceSyncChecker;
        this.sourceFinder = new ComponentSourceFinder(getVaadinSession());

        handlers.put("set-component-property",
                new RewriteHandler("set component property", this::handleSetComponentProperty));
        handlers.put("add-call", new RewriteHandler("add call", this::handleAddCall));
        handlers.put("add-template", new RewriteHandler("add call", this::handleAddTemplate));
        handlers.put("delete-components", new RewriteHandler("delete components", this::handleDeleteComponents));
        handlers.put("duplicate-components",
                new RewriteHandler("duplicate components", this::handleDuplicateComponents));
        handlers.put("drag-and-drop", new RewriteHandler("drop component", this::handleDragAndDrop));
        handlers.put("set-alignment", new RewriteHandler("set alignment", this::handleAlignment));
        handlers.put("set-gap", new RewriteHandler("set gap", this::handleGap));
        handlers.put("wrap-with", new RewriteHandler("wrap with", this::handleWrapWith));
        handlers.put("set-styles", new RewriteHandler("set styles", this::handleSetStyles));
        handlers.put("set-padding", new RewriteHandler("set padding", this::handlePadding));
        handlers.put("copy", new RewriteHandler("copy", this::handleCopy));
        handlers.put("can-be-edited", new RewriteHandler("can be edited", this::handleCanBeEdited));
        handlers.put("set-sizing", new RewriteHandler("set sizing", this::handleSetSizing));
        handlers.put("connect-to-service", new RewriteHandler("connect to service", this::handleConnectToService));
        handlers.put("extract-component", new RewriteHandler("extract component", this::handleExtractComponent));
    }

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        RewriteHandler handler = handlers.get(command);
        if (handler == null) {
            return false;
        }
        String reqId = data.getString(KEY_REQ_ID);
        JsonObject respData = Json.createObject();
        respData.put(KEY_REQ_ID, reqId);
        try {
            handler.handle(new JavaFileSourceProvider(), data, respData);
            devToolsInterface.send(command + "-response", respData);
        } catch (ComponentInfoNotFoundException e) {
            if (sourceSyncChecker.maybeOutOfSync(e)) {
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData,
                        e.getComponentTypeAndSourceLocation().javaFile().map(File::getName).orElse("?")
                                + " may be out of sync. Please recompile and deploy the file",
                        e);
            } else {
                getLogger().debug("Failed to {} for input {}", handler.getWhat(), data.toJson(), e);
                ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Failed to " + handler.getWhat(),
                        e);
            }
        } catch (Exception e) {
            getLogger().debug("Failed to {} for input {}", handler.getWhat(), data.toJson(), e);
            ErrorHandler.sendErrorResponse(devToolsInterface, command, respData, "Failed to " + handler.getWhat(), e);
        }

        return true;
    }

    private void handleAddCall(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        String func = data.getString("func");
        String parameter = data.getString("parameter");
        Integer lineToShowInIde = data.hasKey("lineToShowInIde") ? (int) data.getNumber("lineToShowInIde") : null;
        var component = data.getObject(COMPONENT_PROPERTY);

        ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);

        ComponentInfoFinder finder = new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation);
        ComponentInfo info = finder.find();

        if (!JavaRewriterUtil.hasSingleParameterMethod(info.type(), func)) {
            throw new IllegalArgumentException(
                    "Component " + info.type().getName() + " does not support the given method " + func);
        }
        var rewriter = new JavaRewriter();
        rewriter.addCall(info, func, new JavaRewriter.Code(parameter));
        javaFileSourceProvider.saveComponentInfoSourceFiles(info);

        if (lineToShowInIde != null) {
            showComponentSourceFileInIde(info, lineToShowInIde);
        }
    }

    private void showComponentSourceFileInIde(ComponentInfo info, int lineToShowInIde) {
        if (info.componentCreateInfoOptional().isPresent()) {
            ComponentCreateInfo componentCreateInfo = info.componentCreateInfoOptional().get();
            if (componentCreateInfo.getJavaSource().isChanged()) {
                showFileInIde(componentCreateInfo.getJavaSource(), componentCreateInfo.getFile(), lineToShowInIde);
                return;
            }
        }
        if (info.componentAttachInfoOptional().isPresent()) {
            ComponentAttachInfo componentAttachInfo = info.componentAttachInfoOptional().get();
            if (componentAttachInfo.getJavaSource().isChanged()) {
                showFileInIde(componentAttachInfo.getJavaSource(), componentAttachInfo.getFile(), lineToShowInIde);
            }
        }
    }

    private void showFileInIde(JavaSource source, File javaFile, int lineToShowInIde) {
        int firstModifiedRow = source.getFirstModifiedRow();
        int lineNumber = firstModifiedRow + lineToShowInIde;
        IdeUtils.openFile(javaFile, lineNumber);
        CompletableFuture.runAsync(() -> {
            try {
                // Workaround for
                // https://youtrack.jetbrains.com/issue/IDEA-342750
                Thread.sleep(1000);
                IdeUtils.openFile(javaFile, lineNumber);
            } catch (InterruptedException e) {
                getLogger().error("Failed to show file in IDE", e);
                Thread.currentThread().interrupt();
            }
        });
    }

    private void handleDeleteComponents(JavaFileSourceProvider javaFileSourceProvider, JsonObject data,
            JsonObject respData) {
        JsonArray componentsJson = data.getArray("components");
        String activeJavaClassName = safeGet(data, "activeJavaClassName");
        boolean drilledDownComponent;
        if (activeJavaClassName != null) {
            // for components that are not a descendant of a non custom components e.g.
            // components from main layout.
            drilledDownComponent = CustomComponents.isCustomComponent(activeJavaClassName);
        } else {
            drilledDownComponent = false;
        }

        List<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            List<ComponentTypeAndSourceLocation> found = findTypeAndSourceLocationIncludingChildren(
                    componentsJson.getObject(i));
            components.addAll(found);
        }
        List<ComponentInfo> list = components.stream().filter(typeAndSourceLocation -> Stream
                .of(typeAndSourceLocation.createLocationInProject(), typeAndSourceLocation.attachLocationInProject())
                .flatMap(Optional::stream).map(ComponentTracker.Location::className).anyMatch(locationClass -> {
                    if (drilledDownComponent) {
                        return locationClass.equals(activeJavaClassName);
                    }
                    // the component might be in a layout related class
                    boolean customComponent = CustomComponents.isCustomComponent(locationClass);
                    return !customComponent;
                })).map(typeAndSourceLocation -> {
                    try {
                        return new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation).find();
                    } catch (IOException e) {
                        throw new IllegalArgumentException("Unable to find component info");
                    }
                }).toList();
        JavaRewriter javaRewriter = new JavaRewriter();
        list.forEach(javaRewriter::delete);
        list.forEach(javaFileSourceProvider::saveComponentInfoSourceFiles);
    }

    private List<ComponentTypeAndSourceLocation> findTypeAndSourceLocationIncludingChildren(JsonObject object) {
        ArrayList<ComponentTypeAndSourceLocation> all = new ArrayList<>();
        ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(object, true);

        addRecursively(all, root);
        return all;
    }

    /**
     * Finds the type and source location of the given component and its first level
     * children. The target of this method is to identify a selected component in
     * the UI that is why is not necessary to find all the children but just the
     * first level ones to make equals method to work.
     *
     * @param object
     *            the component
     * @return the type and source location of the component and its first level
     *         children
     */
    private ComponentTypeAndSourceLocation findTypeAndSourceLocationIncludingFirstLevelChildren(JsonObject object) {
        ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(object, true);
        return root;
    }

    private void addRecursively(ArrayList<ComponentTypeAndSourceLocation> all, ComponentTypeAndSourceLocation root) {
        all.add(root);
        File rootFile = root.javaFile().orElse(null);
        for (ComponentTypeAndSourceLocation child : root.children()) {
            File childFile = child.javaFile().orElse(null);
            if (childFile != null && childFile.exists()) {
                // Only recurse if the child file is the same as the root file as custom
                // components and other components can be in different files then when they were
                // referred.
                if (rootFile != null && rootFile.equals(childFile)) {
                    addRecursively(all, child);
                } else {
                    getLogger().debug("Skipping recursion for {} because it is in a different file from the root {}",
                            childFile, rootFile);
                }
            } else {
                getLogger().debug(
                        "Excluding file {} because it does not exist. Assuming this is a component created internally by the parent component and not from the project source",
                        childFile);
            }
        }
    }

    private void handleCopy(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject response)
            throws IOException {
        int componentId = (int) data.getNumber(COMPONENT_ID);

        int uiId = (int) data.getNumber("uiId");
        ComponentTypeAndSourceLocation copiedComponent = sourceFinder.findTypeAndSourceLocation(uiId, componentId,
                true);

        JavaRewriterCopyPasteHandler handler = new JavaRewriterCopyPasteHandler();
        JavaComponent copiedJavaComponent = handler.getCopiedJavaComponent(javaFileSourceProvider, copiedComponent);
        String s = objectMapper.writeValueAsString(copiedJavaComponent);
        response.put(COMPONENT_PROPERTY, s);
    }

    private void handleDuplicateComponents(JavaFileSourceProvider javaFileSourceProvider, JsonObject data,
            JsonObject respData) {
        JsonArray componentsJson = data.getArray("components");
        ArrayList<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        List<ComponentTypeAndSourceLocation> selectedComponents = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            ComponentTypeAndSourceLocation root = sourceFinder.findTypeAndSourceLocation(componentsJson.getObject(i),
                    true);
            selectedComponents.add(root);
            addRecursively(components, root);
        }

        JavaBatchRewriter batchRewriter = new JavaBatchRewriter(javaFileSourceProvider, components);
        selectedComponents.forEach(batchRewriter::duplicate);
        batchRewriter.writeResult();
    }

    private void handleWrapWith(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        JsonArray componentsJson = data.getArray("components");
        JavaComponent wrapperComponent = JavaComponent.componentFromJson(data.getObject("wrapperComponent"));
        List<ComponentTypeAndSourceLocation> components = new ArrayList<>();
        for (int i = 0; i < componentsJson.length(); i++) {
            components.add(sourceFinder.findTypeAndSourceLocation(componentsJson.getObject(i)));
        }
        File javaFile = getProjectFileManager().getSourceFile(components.get(0).getCreateLocationOrThrow());
        KotlinUtil.throwIfKotlin(javaFile);

        JavaRewriter rewriter = new JavaRewriter();

        List<ComponentInfo> componentInfos = new ArrayList<>();
        for (ComponentTypeAndSourceLocation typeAndSourceLocation : components) {
            ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation).find();
            componentInfos.add(componentInfo);
        }

        rewriter.mergeAndReplace(componentInfos, wrapperComponent);
        for (ComponentInfo componentInfo : componentInfos) {
            javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
        }
    }

    private void handleSetComponentProperty(JavaFileSourceProvider javaFileSourceProvider, JsonObject data,
            JsonObject respData) throws IOException {
        String property = data.getString("property");
        String value = data.getString("value");
        var component = data.getObject("component");

        ComponentTypeAndSourceLocation typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        if (JavaDataProviderHandler.isDataProviderItemChange(typeAndSourceLocation)) {
            JavaDataProviderHandler dataProviderHandler = new JavaDataProviderHandler(javaFileSourceProvider,
                    typeAndSourceLocation);
            JavaDataProviderHandler.JavaDataProviderHandlerResult javaDataProviderHandlerResult = dataProviderHandler
                    .handleSetComponentProperty(javaFileSourceProvider, property, value);
            getProjectFileManager().writeFile(javaDataProviderHandlerResult.file(), UNDO_LABEL,
                    javaDataProviderHandlerResult.result());
            return;
        }
        ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation).find();
        var rewriter = new JavaRewriter();
        String setter = JavaRewriterUtil.getSetterName(property, componentInfo.type(), true);

        JavaRewriter.ReplaceResult result = rewriter.replaceFunctionCall(componentInfo, setter, value);
        if (result.variableRenamedTo() != null) {
            respData.put("variableRenamedTo", result.variableRenamedTo());
        }
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleAddTemplate(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        final String whereJsonKey = "where";
        List<JavaComponent> template = JavaComponent.componentsFromJson(data.getArray("template"));
        JavaRewriter.Where where = null;
        if (data.hasKey(whereJsonKey) && data.get(whereJsonKey).getType() != JsonType.NULL) {
            where = JavaRewriter.Where.valueOf(data.getString(whereJsonKey).toUpperCase(Locale.ENGLISH));
        }
        ComponentTypeAndSourceLocation refSource = getRealOrAnalyzedComponentTypeAndSourceLocation(data, "refNode");

        var componentInfo = new ComponentInfoFinder(javaFileSourceProvider, refSource).find();
        var rewriter = new JavaRewriter();

        JavaRewriter.AddTemplateOptions options = new JavaRewriter.AddTemplateOptions(
                data.getBoolean("javaFieldsForLeafComponents"), safeGet(data, "methodName"));

        rewriter.addComponentUsingTemplate(componentInfo, where, template, options);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleDragAndDrop(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        JavaRewriter.Where where = JavaRewriter.Where.valueOf(data.getString("where").toUpperCase(Locale.ENGLISH));

        ComponentTypeAndSourceLocation dragged = getRealOrAnalyzedComponentTypeAndSourceLocation(data, "dragged");
        ComponentTypeAndSourceLocation container = getRealOrAnalyzedComponentTypeAndSourceLocation(data, "container");

        ComponentTypeAndSourceLocation insertBefore = where == JavaRewriter.Where.BEFORE
                ? sourceFinder.findTypeAndSourceLocation(data.getObject("insertBefore"))
                : null;

        var componentInfo = new ComponentInfoFinder(javaFileSourceProvider, container).find();
        var rewriter = new JavaRewriter();

        if (!dragged.javaFile().equals(container.javaFile())) {
            throw new IllegalArgumentException("Cannot move a component in one file (" + dragged.javaFile()
                    + ") to another file (" + container.javaFile() + ")");
        }

        ComponentInfo draggedRef = new ComponentInfoFinder(javaFileSourceProvider, dragged).find();
        ComponentInfo containerRef = new ComponentInfoFinder(javaFileSourceProvider, container).find();
        ComponentInfo insertBeforeRef = insertBefore == null ? null
                : new ComponentInfoFinder(javaFileSourceProvider, insertBefore).find();

        rewriter.moveComponent(draggedRef, containerRef, insertBeforeRef, where);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    /**
     * Gets the component type and source location by considering if given node is
     * the drilled down component.
     *
     * @param data
     *            request data
     * @param nodeKey
     *            key to get component nodeId and uiId info
     * @return source from sourceFinder if not drilled down.
     * @throws IOException
     *             is thrown when file operation fails.
     */
    private ComponentTypeAndSourceLocation getRealOrAnalyzedComponentTypeAndSourceLocation(JsonObject data,
            String nodeKey) throws IOException {
        if (CustomComponentHelper.isDrilledDownComponent(getVaadinSession(), data, data.getObject(nodeKey))) {
            JsonObject component = data.getObject(nodeKey);
            return sourceFinder
                    .analyzeSourceFileAndGetComponentTypeAndSourceLocation((int) component.getNumber("nodeId"),
                            (int) component.getNumber("uiId"))
                    .orElseThrow(() -> new RuntimeException("Could not find ref source file"));
        } else {
            return sourceFinder.findTypeAndSourceLocation(data.getObject(nodeKey));
        }
    }

    private void handleAlignment(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));

        String alignItemsClassName = safeGet(data, "alignItemsClassName");
        String justifyContentClassName = safeGet(data, "justifyContentClassName");
        ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation)
                .find();
        var rewriter = new JavaRewriter();
        rewriter.setAlignment(componentInfo, alignItemsClassName, justifyContentClassName);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleSetStyles(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        JsonArray added = data.getArray("added");
        JsonArray removed = data.getArray("removed");
        Set<String> toRemove = new HashSet<>();
        for (int i = 0; i < removed.length(); i++) {
            toRemove.add(removed.getObject(i).getString("key"));
        }
        ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation)
                .find();
        var rewriter = new JavaRewriter();

        for (int i = 0; i < added.length(); i++) {
            JsonObject rule = added.getObject(i);
            String key = rule.getString("key");
            String value = rule.getString("value");
            rewriter.setStyle(componentInfo, key, value);
            toRemove.remove(key);
        }

        for (String key : toRemove) {
            rewriter.setStyle(componentInfo, key, null);
        }
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleConnectToService(JavaFileSourceProvider javaFileSourceProvider, JsonObject data,
            JsonObject respData) throws IOException {
        if (!SpringBridge.isSpringAvailable(getVaadinContext())) {
            throw new IllegalStateException("Connecting to a service is only supported for Spring applications");
        }

        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        ComponentInfoFinder componentInfoFinder = new ComponentInfoFinder(javaFileSourceProvider,
                componentTypeAndSourceLocation);
        ComponentInfo componentInfo = componentInfoFinder.find();

        var rewriter = new JavaRewriter();
        JsonObject service = data.getObject("service");

        String serviceClassName = service.getString("className");
        String serviceMethodName = service.getString("methodName");
        JavaReflectionUtil.TypeInfo methodReturnType = objectMapper.readValue(service.getObject("returnType").toJson(),
                JavaReflectionUtil.TypeInfo.class);
        List<JavaReflectionUtil.ParameterTypeInfo> parameters = JsonUtils.stream(service.getArray("parameters"))
                .map(parameter -> {
                    try {
                        return objectMapper.readValue(parameter.toJson(), JavaReflectionUtil.ParameterTypeInfo.class);
                    } catch (JsonProcessingException e) {
                        throw new RuntimeException(e);
                    }
                }).toList();

        if (methodReturnType.typeParameters().size() != 1) {
            throw new IllegalArgumentException("Method return type " + methodReturnType + " is not a List of items");
        }
        JavaReflectionUtil.TypeInfo itemTypeInfo = methodReturnType.typeParameters().get(0);
        // Grid<Something> -> Grid<Product>

        String itemType = itemTypeInfo.typeName();
        List<UIServiceCreator.FieldInfo> propertiesInSourceOrder = JavaRewriterUtil
                .getPropertiesInSourceOrder(itemType);

        // This is very common to have and you don't want to show it
        propertiesInSourceOrder.removeIf(fieldInfo -> fieldInfo.name().equals("id"));

        if (isGrid(componentInfo)) {
            rewriter.setGridDataSource(componentInfo, serviceClassName, serviceMethodName, parameters, itemType,
                    propertiesInSourceOrder);
        } else if (isComboBox(componentInfo)) {
            UIServiceCreator.FieldInfo firstStringProperty = propertiesInSourceOrder.stream()
                    .filter(f -> f.javaType().equals(STRING_TYPE)).findFirst().orElse(null);
            rewriter.setComboBoxDataSource(componentInfo, serviceClassName, serviceMethodName, parameters, itemType,
                    firstStringProperty);
        } else {
            throw new IllegalArgumentException(
                    "Connecting to a service is not supported for " + componentInfo.type().getName());
        }
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private boolean isGrid(ComponentInfo componentInfo) {
        return isComponent(componentInfo, "com.vaadin.flow.component.grid.Grid");
    }

    private boolean isComboBox(ComponentInfo componentInfo) {
        return isComponent(componentInfo, "com.vaadin.flow.component.combobox.ComboBox");
    }

    private boolean isComponent(ComponentInfo componentInfo, String componentClass) {
        try {
            return Class.forName(componentClass).isAssignableFrom(componentInfo.type());
        } catch (Exception e) {
            return false;
        }

    }

    private void handleExtractComponent(JavaFileSourceProvider javaFileSourceProvider, JsonObject data,
            JsonObject respData) throws IOException {
        ComponentTypeAndSourceLocation rootComponent = sourceFinder
                .findTypeAndSourceLocation(data.getObject("component"), true);
        ComponentInfo rootComponentInfo = new ComponentInfoFinder(javaFileSourceProvider, rootComponent).find();
        JavaRewriter rewriter = new JavaRewriter();

        IdentityHashMap<ComponentInfo, List<ComponentInfo>> childrenMap = new IdentityHashMap<>();
        populateChildren(rootComponent, rootComponentInfo, childrenMap, javaFileSourceProvider);
        rewriter.extractComponent(rootComponentInfo, childrenMap, "MyComponent");
        javaFileSourceProvider.saveComponentInfoSourceFiles(rootComponentInfo);

    }

    private void populateChildren(ComponentTypeAndSourceLocation component, ComponentInfo componentInfo,
            IdentityHashMap<ComponentInfo, List<ComponentInfo>> childrenMap,
            JavaFileSourceProvider javaFileSourceProvider) throws IOException {
        List<ComponentTypeAndSourceLocation> children = component.children();
        List<ComponentInfo> childComponentInfo = new ArrayList<>();
        for (ComponentTypeAndSourceLocation child : children) {
            if (child.createLocationInProject().isEmpty()) {
                continue;
            }
            ComponentInfo childInfo = new ComponentInfoFinder(javaFileSourceProvider, child).find();
            populateChildren(child, childInfo, childrenMap, javaFileSourceProvider);
            childComponentInfo.add(childInfo);
        }
        childrenMap.put(componentInfo, childComponentInfo);
    }

    private void handleSetSizing(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));

        var componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation).find();

        JsonObject changesJson = data.getObject("changes");
        TypeReference<HashMap<String, String>> typeRef = new TypeReference<>() {
        };
        HashMap<String, String> changes = objectMapper.readValue(changesJson.toJson(), typeRef);
        JavaRewriter rewriter = new JavaRewriter();
        rewriter.setSizing(componentInfo, changes);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handleGap(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        String lumoClassAll = safeGet(data, "all");
        String lumoClassRow = safeGet(data, "row");
        String lumoClassColumn = safeGet(data, "column");
        ComponentInfo componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation)
                .find();
        JavaRewriter rewriter = new JavaRewriter();
        rewriter.setGap(componentInfo, lumoClassAll, lumoClassColumn, lumoClassRow);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    private void handlePadding(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject respData)
            throws IOException {
        ComponentTypeAndSourceLocation componentTypeAndSourceLocation = sourceFinder
                .findTypeAndSourceLocation(data.getObject(COMPONENT_ID));
        String allPaddingLumoClass = safeGet(data, "all");
        String topPaddingLumoClass = safeGet(data, "top");
        String rightPaddingLumoClass = safeGet(data, "right");
        String bottomPaddingLumoClass = safeGet(data, "bottom");
        String leftPaddingLumoClass = safeGet(data, "left");
        var componentInfo = new ComponentInfoFinder(javaFileSourceProvider, componentTypeAndSourceLocation).find();
        JavaRewriter rewriter = new JavaRewriter();
        rewriter.setPadding(componentInfo, allPaddingLumoClass, topPaddingLumoClass, rightPaddingLumoClass,
                bottomPaddingLumoClass, leftPaddingLumoClass);
        javaFileSourceProvider.saveComponentInfoSourceFiles(componentInfo);
    }

    // TODO remove this and use a proper way.
    private String safeGet(JsonObject data, String key) {
        return data.hasKey(key) && !data.get(key).getType().equals(JsonType.NULL) ? data.getString(key) : null;
    }

    private void handleCanBeEdited(JavaFileSourceProvider javaFileSourceProvider, JsonObject data, JsonObject response)
            throws IOException {
        var component = data.getObject(COMPONENT_PROPERTY);
        String propertyToCheck = data.getString(PROPERTY_TO_CHECK_PROPERTY);

        if (Arrays.stream(supportedEditableProperties)
                .noneMatch(property -> property.equalsIgnoreCase(propertyToCheck))) {
            response.put(CAN_BE_EDITED, false);
            response.put(IS_TRANSLATION, false);
            return;
        }
        var typeAndSourceLocation = sourceFinder.findTypeAndSourceLocation(component);
        ComponentInfoFinder finder = new ComponentInfoFinder(javaFileSourceProvider, typeAndSourceLocation);
        ComponentInfo info = finder.find();
        var rewriter = new JavaRewriter();

        try {
            var result = rewriter.getPropertyValue(info, propertyToCheck);
            response.put(CAN_BE_EDITED, result instanceof String);
            if (result instanceof MethodCallExpr methodCallExpr) {
                response.put(IS_TRANSLATION, methodCallExpr.getNameAsString().equals("translate"));
            } else {
                response.put(IS_TRANSLATION, false);
            }
        } catch (Exception e) {
            getLogger().error("Failed to check if property can be edited", e);
            response.put(CAN_BE_EDITED, false);
            response.put(IS_TRANSLATION, false);
        }
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
