package com.vaadin.copilot;

import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import com.vaadin.flow.component.Component;

import com.googlecode.gentyref.GenericTypeReflector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JavaReflectionUtil {

    public record TypeInfo(String typeName, List<TypeInfo> typeParameters) {
    }

    public record ParameterTypeInfo(String name, TypeInfo type) {
    }

    public static List<ParameterTypeInfo> getParameterTypes(Method method, Class<?> cls) {
        Type[] parameterTypes;
        try {
            parameterTypes = GenericTypeReflector.getExactParameterTypes(method, cls);
        } catch (Exception e) {
            parameterTypes = method.getGenericParameterTypes();
        }
        List<ParameterTypeInfo> parameterTypeInfo = new ArrayList<>();
        for (int i = 0; i < parameterTypes.length; i++) {
            String name = method.getParameters()[i].getName();
            TypeInfo typeInfo = getTypeInfo(parameterTypes[i]);
            parameterTypeInfo.add(new ParameterTypeInfo(name, typeInfo));
        }
        return parameterTypeInfo;
    }

    private static TypeInfo getTypeInfo(Type type) {
        if (type instanceof ParameterizedType parameterizedReturnType) {
            Type rawType = parameterizedReturnType.getRawType();
            String typeName;
            if (rawType instanceof Class cls) {
                typeName = getClassName(cls);
            } else {
                typeName = rawType.getTypeName();
            }
            List<TypeInfo> typeParameters = Arrays.stream(parameterizedReturnType.getActualTypeArguments())
                    .map(JavaReflectionUtil::getTypeInfo).toList();
            return new TypeInfo(typeName, typeParameters);
        } else if (type instanceof Class<?> classType) {
            return new TypeInfo(getClassName(classType), new ArrayList<>());
        } else if (type instanceof TypeVariable<?> typeVariable) {
            return new TypeInfo(typeVariable.getName(), new ArrayList<>());
        }
        return new TypeInfo("???", new ArrayList<>());
    }

    public static TypeInfo getReturnType(Method method, Class<?> cls) {
        Type returnType;
        try {
            returnType = GenericTypeReflector.getExactReturnType(method, cls);
        } catch (Exception e) {
            returnType = method.getGenericReturnType();
        }
        return getTypeInfo(returnType);
    }

    public static String getClassName(Class<?> aClass) {
        return Optional.ofNullable(aClass.getCanonicalName()).orElse(aClass.getName());
    }

    /**
     * Checks whether the given method parameter is (or extends) a {@link Component}
     * type, either directly or through inheritance. If the parameter is an array,
     * the component type of the array is used.
     *
     * <p>
     * This method walks up the class hierarchy to determine if the parameter type
     * (or its component type in case of an array) is a subclass of
     * {@code Component}.
     *
     * @param parameter
     *            the method parameter to inspect
     * @return {@code true} if the parameter type or its component type is a
     *         subclass of {@code Component}, {@code false} otherwise
     */
    public static boolean isParamFlowComponentType(Parameter parameter) {
        Class<?> type;
        if (parameter.getType().isArray()) {
            type = parameter.getType().componentType();
        } else {
            type = parameter.getType();
        }
        while (type != null) {
            if (type.equals(Component.class)) {
                return true;
            }
            type = type.getSuperclass();
        }
        return false;
    }

    /**
     * Calls the method of given component instance to obtain result value
     *
     * @param object
     *            Component that method will be executed
     * @param getterMethod
     *            Method that returns a value
     * @return {@link Optional#empty} if no value is returned or an exception is
     *         thrown, returns the value otherwise
     */
    public static Optional<Object> getMethodInvocationResult(Object object, Method getterMethod) {
        if (getterMethod == null) {
            return Optional.empty();
        }
        try {
            Object result = getterMethod.invoke(object);
            if (result == null) {
                return Optional.empty();
            }
            return Optional.of(result);
        } catch (Exception e) {
            getLogger().trace("Could not get property value of {} from {} method", object.getClass().getName(),
                    getterMethod.getName(), e);
            return Optional.empty();
        }
    }

    /**
     * Finds the class for the given source type.
     *
     * @param name
     *            the class name
     * @return the class for the given name
     * @throws IllegalArgumentException
     *             if the class is not found
     */
    public static Class<?> getClass(String name) throws IllegalArgumentException {
        try {
            return Class.forName(name);
        } catch (ClassNotFoundException e) {
            getLogger().debug("Class " + name + " not found", e);
            if (name.contains(".")) {
                int lastDot = name.lastIndexOf('.');
                return getClass(name.substring(0, lastDot) + "$" + name.substring(lastDot + 1));
            } else {
                throw new IllegalArgumentException("Class " + name + " not found", e);
            }
        }
    }

    /**
     * Uses reflection API for finding class name, method and the argument to find
     * whether argument is array.<br>
     * Includes the methods from extended Composite class if it is extended.
     *
     * @param className
     *            full class name of a class.
     * @param methodName
     *            method name
     * @param argumentIndex
     *            argument to look up. It can exceed the method parameters, in that
     *            case the last argument is taken into account
     * @return {@code true} if argument is an array, {@code false} otherwise
     */
    public static boolean isArrayArgument(String className, String methodName, int argumentIndex) {
        try {

            Class<?> clazz = JavaReflectionUtil.getClass(className);
            Set<Method> methods = new HashSet<>(Arrays.stream(clazz.getMethods()).toList());
            // including Composite methods
            java.lang.reflect.Type genericSuperclass = clazz.getGenericSuperclass();
            if (genericSuperclass instanceof ParameterizedType parameterizedType
                    && parameterizedType.getRawType() instanceof Class<?> parameterType
                    && parameterType.getName().equals("com.vaadin.flow.component.Composite")) {
                Class<?> compositeClazz = (Class<?>) parameterizedType.getActualTypeArguments()[0];
                methods.addAll(Arrays.stream(compositeClazz.getMethods()).toList());
            }

            return methods.stream().filter(f -> f.getName().equals(methodName)).anyMatch(method -> {
                Parameter[] parameters = method.getParameters();
                Parameter seekingParam;
                if (argumentIndex < parameters.length) {
                    seekingParam = parameters[argumentIndex];
                } else {
                    seekingParam = parameters[parameters.length - 1];
                }
                return seekingParam.getType().isArray();
            });
        } catch (Exception ex) {
            getLogger().debug("Could not find argument index of {}", className + "/" + methodName, ex);
        }
        return false;
    }

    /**
     * Returns a list of fully qualified class names representing the class
     * hierarchy for the specified class name, starting from the given class up to
     * {@code java.lang.Object}.
     *
     * <p>
     * This method attempts to load the class using {@link #getClass(String)} and
     * then traverses its superclasses, collecting their fully qualified names into
     * a list.
     *
     * @param className
     *            the fully qualified name of the class whose hierarchy is to be
     *            retrieved
     * @return a list of fully qualified class names in the hierarchy, starting with
     *         the specified class
     * @throws RuntimeException
     *             if the class cannot be found by {@link #getClass(String)}
     */
    public static List<String> getClassHierarchy(String className) {
        Class<?> clazz = getClass(className);
        List<String> hierarchy = new ArrayList<>();
        while (clazz != null) {
            hierarchy.add(clazz.getName());
            clazz = clazz.getSuperclass();
        }
        return hierarchy;
    }

    /**
     * Retrieves the value of a specified field from the given target object by
     * searching through the class hierarchy of the provided class.
     *
     * <p>
     * This method attempts to find the field with the specified name in the given
     * class or any of its superclasses. If found, it makes the field accessible and
     * returns its value wrapped in an {@link Optional}. If the field is not found
     * or cannot be accessed, an empty {@code Optional} is returned.
     *
     * @param <T>
     *            the type of the class
     * @param clazz
     *            the class to start the field lookup from (can be a superclass of
     *            {@code target})
     * @param instance
     *            the instance from which to retrieve the field value
     * @param fieldName
     *            the name of the field to retrieve
     * @return an {@code Optional} containing the field value if found and
     *         accessible; otherwise, an empty {@code Optional}
     */
    public static <T> Optional<Object> getFieldValue(Class<? extends T> clazz, T instance, String fieldName) {
        try {
            Optional<Field> declaredFieldsInClassHierarchy = getDeclaredFieldsInClassHierarchy(clazz, fieldName);
            if (declaredFieldsInClassHierarchy.isEmpty()) {
                return Optional.empty();
            }
            Field field = declaredFieldsInClassHierarchy.get();
            field.setAccessible(true);
            Object fieldValue = field.get(instance);
            return Optional.ofNullable(fieldValue);
        } catch (IllegalAccessException e) {
            getLogger().debug("Could not access the field {}", fieldName, e);
        }
        return Optional.empty();
    }

    /**
     * Retrieves the value of a specified field from the given target object.
     *
     * <p>
     * This method searches for the specified field in the class hierarchy of the
     * target object. If the field is found, it is made accessible and its value is
     * retrieved.
     *
     * @param targetInstance
     *            The object from which the field value is retrieved.
     * @param fieldName
     *            The name of the field whose value is to be accessed.
     * @return An {@link Optional} containing the field value if found; otherwise,
     *         an empty {@link Optional}.
     */
    public static Optional<Object> getFieldValue(Object targetInstance, String fieldName) {
        return getFieldValue(targetInstance.getClass(), targetInstance, fieldName);
    }

    /**
     * Recursively searches for a declared field in the given class and its
     * superclass hierarchy.
     *
     * <p>
     * If the field is found in the given class, it is returned. If the field is not
     * found in the current class, the search continues recursively in its
     * superclass.
     *
     * @param clazz
     *            The class in which to search for the field.
     * @param fieldName
     *            The name of the field to search for.
     * @return An {@link Optional} containing the {@link Field} if found; otherwise,
     *         an empty {@link Optional}.
     */
    private static Optional<Field> getDeclaredFieldsInClassHierarchy(Class<?> clazz, String fieldName) {
        try {
            Field field = clazz.getDeclaredField(fieldName);
            return Optional.of(field);
        } catch (NoSuchFieldException ex) {
            getLogger().debug("Could not find field {} in class {}", fieldName, clazz.getName(), ex);
            if (clazz.getSuperclass() != null) {
                return getDeclaredFieldsInClassHierarchy(clazz.getSuperclass(), fieldName);
            }
        }
        return Optional.empty();
    }

    /**
     * Instantiates a new object of the given class using its no-argument
     * constructor.
     *
     * <p>
     * This method uses reflection to create a new instance of the specified class.
     * The class must have a public no-argument constructor, otherwise a
     * {@link NoSuchMethodException} will be thrown. The returned object is cast to
     * the expected type {@code T}, so callers should ensure that the class type
     * matches the expected return type.
     *
     * @param clazz
     *            the {@link Class} object representing the class to instantiate
     * @param <T>
     *            the type of the object to return
     * @return a new instance of the specified class
     * @throws NoSuchMethodException
     *             if the class does not have a public no-argument constructor
     * @throws InvocationTargetException
     *             if the constructor throws an exception
     * @throws InstantiationException
     *             if the class represents an abstract class
     * @throws IllegalAccessException
     *             if the constructor is not accessible
     * @throws ClassCastException
     *             if the created instance cannot be cast to type {@code T}
     */
    public static <T> T instantiateClass(Class<?> clazz)
            throws NoSuchMethodException, InvocationTargetException, InstantiationException, IllegalAccessException {
        Constructor<?> constructor = clazz.getConstructor();
        return (T) constructor.newInstance();
    }

    /**
     * Checks whether the specified class declares a field with the given name.
     *
     * <p>
     * This method attempts to retrieve the field using
     * {@code clazz.getDeclaredField(fieldName)}, which only checks for fields
     * declared directly in the class (not inherited fields). If the field exists,
     * the method returns {@code true}. If the field is not found, it returns
     * {@code false}.
     *
     * @param clazz
     *            the {@code Class} object to inspect
     * @param fieldName
     *            the name of the field to check for
     * @return {@code true} if the field is declared in the class; {@code false}
     *         otherwise
     * @throws NullPointerException
     *             if {@code clazz} or {@code fieldName} is {@code null}
     */
    public static boolean hasField(Class<?> clazz, String fieldName) {
        try {
            clazz.getDeclaredField(fieldName);
            return true;
        } catch (NoSuchFieldException _e) {
            return false;
        }
    }

    /**
     * Checks whether a class with the specified fully qualified name exists and is
     * loadable.
     *
     * <p>
     * This method attempts to load the class using
     * {@link JavaReflectionUtil#getClass(String)}. If the class can be successfully
     * loaded, the method returns {@code true}. If any exception occurs during the
     * loading process (such as {@code ClassNotFoundException}), the method returns
     * {@code false}.
     *
     * @param className
     *            the fully qualified name of the class to check
     * @return {@code true} if the class exists and can be loaded; {@code false}
     *         otherwise
     */
    public static boolean exists(String className) {
        try {
            JavaReflectionUtil.getClass(className);
            return true;
        } catch (Exception _e) {
            return false;
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(JavaReflectionUtil.class);
    }
}
