package com.vaadin.copilot;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.server.VaadinServletContext;
import com.vaadin.flow.server.VaadinSession;

import elemental.json.JsonObject;

/**
 * Base class for all classes that handle Copilot commands from the browser.
 */
public abstract class CopilotCommand {

    public static final String KEY_REQ_ID = "reqId";
    public static final String KEY_CANCEL_REQ_ID = "requestId";
    /**
     * Used to pass the session to all commands without having to add it to all
     * constructors but so it is still available in constructors.
     */
    public static final ThreadLocal<VaadinSession> currentSession = new ThreadLocal<>();

    private final VaadinSession vaadinSession;

    protected CopilotCommand() {
        this.vaadinSession = currentSession.get();
    }

    /**
     * Called whenever there is a message from the browser.
     *
     * Implementations should check if the command is for them, handle it and return
     * true if the command was handled.
     *
     * @param command
     *            the command to handle
     * @param data
     *            the data for the command
     * @param devToolsInterface
     *            the devtools interface for the connection
     * @return true if the command was handled, false otherwise
     */
    public abstract boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface);

    /**
     * Called when a new browser (tab) connects to the server. Override to add logic
     * that should run on every connect.
     *
     * @param devToolsInterface
     *            the devtools interface for the new connection
     */
    public void handleConnect(DevToolsInterface devToolsInterface) {
    }

    /**
     * Checks if given command can be handled asynchronously. If the handler can't
     * be asynchronous at all, it can return false always. But if you want to handle
     * one or more commands asynchronously, you should specify the exact command to
     * prevent handling commands that are not part of the handler as asynchronous.
     * 
     * @param command
     *            the command to check
     * @return true if the command can be handled by this command handler
     */
    public boolean canBeParallelCommand(String command) {
        return false;
    }

    /**
     * Shorthand for getting the ProjectFileManager instance.
     * 
     * @return the ProjectFileManager instance
     */
    public ProjectFileManager getProjectFileManager() {
        return ProjectFileManager.get();
    }

    /**
     * Gets the vaadin session.
     */
    protected VaadinSession getVaadinSession() {
        if (vaadinSession == null) {
            throw new IllegalStateException("Session is not available. This should never happen.");
        }
        return vaadinSession;
    }

    /**
     * Gets the vaadin context.
     * 
     * @return the vaadin context
     */
    protected VaadinServletContext getVaadinContext() {
        return Copilot.getContext(getVaadinSession());
    }
}
