package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;

import elemental.json.Json;
import elemental.json.JsonObject;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility class to create JSON response from Configuration files and save
 * configuration data from request
 */
public final class ConfigurationFileUtil {
    private ConfigurationFileUtil() {

    }

    /**
     * Gets response that contains configuration JSON.
     *
     * @param configFile
     *            file to read config
     * @return JSON object with <code>conf</code> key that has Configuration data
     */
    public static JsonObject getConfigResponse(File configFile) {
        JsonObject response = Json.createObject();
        try {
            if (configFile.exists()) {
                String json = FileUtils.readFileToString(configFile, StandardCharsets.UTF_8);
                getLogger().debug("Reading configuration: {}", json);
                response.put("conf", Json.parse(json));
            } else {
                response.put("conf", Json.parse("{}"));
            }
        } catch (Exception e) {
            getLogger().error("Failed to read configuration from {}", configFile.toPath(), e);
            response.put("error", true);
        }
        return response;
    }

    /**
     * Saves the configuration into given file by extracting configuration data from
     * request
     *
     * @param configFile
     *            file to store configuration
     * @param request
     *            request that has <code>conf</code> object
     */
    public static void saveConfigFromRequest(File configFile, JsonObject request) {
        try {
            String json = request.getString("conf");
            ObjectNode jsonObject = new ObjectMapper().readValue(json, ObjectNode.class);
            String indented = new ObjectMapper().writerWithDefaultPrettyPrinter().writeValueAsString(jsonObject);
            FileUtils.write(configFile, indented, StandardCharsets.UTF_8);
            getLogger().debug("Writing configuration: {}", indented);
        } catch (IOException e) {
            getLogger().error("Failed to write configuration to {}", configFile.toPath(), e);
        }
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ConfigurationFileUtil.class);
    }
}
