package com.vaadin.copilot;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.customcomponent.CustomComponentDetector;
import com.vaadin.copilot.customcomponent.CustomComponentHelper;
import com.vaadin.copilot.customcomponent.CustomComponentResponseData;
import com.vaadin.copilot.javarewriter.ComponentTypeAndSourceLocation;
import com.vaadin.copilot.javarewriter.FlowComponentQuirks;
import com.vaadin.copilot.javarewriter.JavaComponent;
import com.vaadin.flow.component.Component;
import com.vaadin.flow.component.internal.ComponentTracker;
import com.vaadin.flow.internal.JsonUtils;

import elemental.json.Json;
import elemental.json.JsonObject;
import elemental.json.JsonType;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public class ComponentInfoHandler extends CopilotCommand {
    private static final String JAVA_CLASS_NAME_REQ_KEY = "javaClassName";

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("get-component-source-info")) {
            // command that is called to get information about components after page is
            // loaded.
            JsonObject responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));

            int uiId = (int) data.getNumber("uiId");

            Map<Component, ComponentTypeAndSourceLocation> allComponents = FlowUtil
                    .findAllComponents(getVaadinSession(), uiId);
            List<Component> componentsInProject = allComponents.entrySet().stream().filter(entry -> {
                Optional<ComponentTracker.Location> create = entry.getValue().createLocationInProject();
                return create.filter(location -> getProjectFileManager().getSourceFile(location).exists()).isPresent();
            }).map(Map.Entry::getKey).toList();

            componentsInProject.stream().map(Component::getClass).collect(Collectors.toSet())
                    .forEach(clazz -> CustomComponentDetector.detectAndPutIfPresent(getVaadinSession(), clazz));

            CustomComponentResponseData customComponentResponseData = CustomComponentHelper
                    .generateResponse(getVaadinSession(), allComponents, componentsInProject);

            responseData.put("customComponentResponse", JsonUtils.writeValue(customComponentResponseData));
            responseData.put("nodeIdsInProject",
                    JsonUtils.listToJson(componentsInProject.stream().map(FlowUtil::getNodeId).toList()));
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        } else if (command.equals("get-class-hierarchy")) {
            // command that is called to get class hierarchy of a component. One of the
            // usage is to display relative methods in custom component panel.
            JsonObject responseData = Json.createObject();
            responseData.put(KEY_REQ_ID, data.getString(KEY_REQ_ID));
            try {
                List<String> classHierarchy;
                if (data.hasKey(JAVA_CLASS_NAME_REQ_KEY)
                        && data.get(JAVA_CLASS_NAME_REQ_KEY).getType() != JsonType.NULL) {
                    classHierarchy = FlowComponentQuirks.getClassHierarchy(new JavaComponent(null,
                            data.getString(JAVA_CLASS_NAME_REQ_KEY), new HashMap<>(), new ArrayList<>()));
                } else {
                    classHierarchy = FlowComponentQuirks.getClassHierarchy(
                            new JavaComponent(data.getString("reactTag"), null, new HashMap<>(), new ArrayList<>()));
                }
                responseData.put("classHierarchy", JsonUtils.listToJson(classHierarchy));
            } catch (Exception e) {
                getLogger().warn("Unable to get class hierarchy", e);
            }
            devToolsInterface.send(Copilot.PREFIX + command + "-response", responseData);
            return true;
        }
        return false;
    }

    private Logger getLogger() {
        return LoggerFactory.getLogger(getClass());
    }
}
