package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.copilot.ide.IdeUtils;
import com.vaadin.copilot.javarewriter.JavaModifier;
import com.vaadin.flow.component.page.AppShellConfigurator;
import com.vaadin.flow.theme.Theme;

import elemental.json.Json;
import elemental.json.JsonObject;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Helps to initialize a Spring Boot Vaadin application with no views by adding
 * either a Flow or Hilla view + a theme.
 */
public class ApplicationInitializer extends CopilotCommand {
    public static final String SUCCESS_KEY = "success";
    public static final String REASON_KEY = "reason";
    public static final String INIT_APP_RESULT = "copilot-init-app-result";
    public static final String REFRESH_KEY = "refresh";

    @Override
    public boolean handleMessage(String command, JsonObject data, DevToolsInterface devToolsInterface) {
        if (command.equals("init-app")) {
            String framework = data.getString("framework");
            try {
                initApp(framework, devToolsInterface, data.getString(KEY_REQ_ID));
            } catch (IOException e) {
                getLogger().error("Unable to initialize project", e);
            }
            return true;
        }
        return false;
    }

    private synchronized void initApp(String framework, DevToolsInterface devToolsInterface, String reqId)
            throws IOException {
        JsonObject responseData = Json.createObject();
        responseData.put(KEY_REQ_ID, reqId);

        if (!SpringBridge.isSpringAvailable(getVaadinContext())) {
            responseData.put(SUCCESS_KEY, false);
            responseData.put(REASON_KEY, "This only works for Spring Boot projects");
            devToolsInterface.send(INIT_APP_RESULT, responseData);
            return;
        }
        Map<File, String> toWrite = new HashMap<>();

        Class<?> applicationClass = SpringBridge.getApplicationClass(getVaadinContext());
        Theme themeAnnotation = applicationClass.getAnnotation(Theme.class);
        if (themeAnnotation == null) {
            String themeName = "my-theme";
            toWrite.putAll(new JavaModifier().modify(applicationClass, operations -> {
                if (!operations.hasClassAnnotation(Theme.class)) {
                    // The class check above is not enough if somebody clicks multiple time on the
                    // 'create' link
                    operations.addClassAnnotation(Theme.class, themeName);
                    operations.addInterface(AppShellConfigurator.class);
                }
            }));
            toWrite.putAll(createTheme(themeName));
        }

        File toOpen;
        if (framework.equals("flow")) {
            Map<? extends File, String> flowView = addFlowView(applicationClass);
            toWrite.putAll(flowView);
            toOpen = flowView.keySet().iterator().next();
        } else if (framework.equals("hilla")) {
            Map<File, String> hillaView = addHillaView();
            toWrite.putAll(hillaView);
            toOpen = hillaView.keySet().iterator().next();
        } else {
            responseData.put(SUCCESS_KEY, false);
            responseData.put(REASON_KEY, "Unknown framework " + framework);
            devToolsInterface.send(INIT_APP_RESULT, responseData);
            return;
        }

        for (Map.Entry<File, String> entry : toWrite.entrySet()) {
            getProjectFileManager().writeFile(entry.getKey(), "Project generation", entry.getValue());
        }

        if (toOpen != null) {
            try {
                Thread.sleep(1000);
                IdeUtils.openFile(toOpen, 1);
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        }
        responseData.put(SUCCESS_KEY, true);
        responseData.put(REFRESH_KEY, true);
        devToolsInterface.send(INIT_APP_RESULT, responseData);
    }

    private Map<File, String> addHillaView() {
        File frontendFolder = getProjectFileManager().getFrontendFolder();
        File viewFile = new File(new File(frontendFolder, "views"), "@index.tsx");
        String viewTemplate = getViewTemplate(viewFile);

        return Collections.singletonMap(viewFile, viewTemplate);
    }

    private Map<? extends File, String> addFlowView(Class<?> applicationClass) {
        File applicationFile = getProjectFileManager().getFileForClass(applicationClass);
        String basePackage = applicationClass.getPackage().getName();
        File viewFolder = new File(applicationFile.getParentFile(), "views");
        File viewFile = new File(viewFolder, "HomeView.java");
        String relativeViewName = getProjectFileManager().getProjectRelativeName(viewFile);
        String viewPackage = basePackage + ".views";

        String content = String.format("""
                package %s;

                import com.vaadin.flow.component.html.H1;
                import com.vaadin.flow.component.html.Paragraph;
                import com.vaadin.flow.component.orderedlayout.VerticalLayout;
                import com.vaadin.flow.router.Route;

                @Route("")
                public class HomeView extends VerticalLayout {

                    public HomeView() {

                        add(new H1("Welcome to your new application"));
                        add(new Paragraph("This is the home view"));

                        add(new Paragraph("You can edit this view in %s"));

                    }
                }
                """, viewPackage, relativeViewName.replace("\\", "\\\\"));

        return Collections.singletonMap(viewFile, content);
    }

    private String getViewTemplate(File viewFile) {
        String relativeViewName = getProjectFileManager().getProjectRelativeName(viewFile);
        return String.format("""
                export default function HomeView() {
                  return (
                    <div>
                      <h1>Welcome to your new application</h1>
                      <p>This is the home view.</p>
                      <p>
                        You can edit this view in <code>%s</code> or by
                        activating Copilot by clicking the icon in the lower right corner
                      </p>
                    </div>
                  );
                }

                """, relativeViewName);
    }

    private Map<File, String> createTheme(String themeName) {
        File frontendFolder = getProjectFileManager().getFrontendFolder();
        File themeFolder = new File(new File(frontendFolder, "themes"), themeName);
        File stylesCss = new File(themeFolder, "styles.css");
        if (!themeFolder.exists()) {
            return Collections.singletonMap(stylesCss, "");
        }
        return Collections.emptyMap();
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(ApplicationInitializer.class);
    }
}
