package com.vaadin.copilot.ide;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;

import com.vaadin.base.devserver.DevToolsInterface;
import com.vaadin.flow.internal.JacksonUtils;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * IDE plugin support utility
 */
public class CopilotIDEPlugin {

    private static final String UNDO_REDO_PREFIX = "Vaadin Copilot";

    private static final String COPILOT_PLUGIN_DOTFILE = ".copilot-plugin";

    private static final String PLUGIN_FILE_LOCATION_PROPERTY = "vaadin.copilot.pluginDotFilePath";

    public enum Commands {
        WRITE("write"),
        WRITE_BASE64("writeBase64"),
        UNDO("undo"),
        REDO("redo"),
        SHOW_IN_IDE("showInIde"),
        REFRESH("refresh"),
        GET_MODULE_PATHS("getModulePaths"),
        RESTART_APPLICATION("restartApplication"),
        HEARTBEAT("heartbeat");

        final String command;

        Commands(String command) {
            this.command = command;
        }

        Command create(Object data) {
            return new Command(command, data);
        }
    }

    public static class UnsupportedOperationByPluginException extends UnsupportedOperationException {
        public UnsupportedOperationByPluginException(Commands command) {
            super("Used version of Copilot IDE Plugin does not support %s operation, please update plugin to latest version."
                    .formatted(command.command));
        }
    }

    private record Command(String command, Object data) {
    }

    private record RestCommand(String command, String projectBasePath, Object data) {
    }

    private record WriteFileMessage(String file, String undoLabel, String content) {
    }

    private record UndoRedoMessage(List<String> files) {
    }

    private record ShowInIdeMessage(String file, Integer line, Integer column) {
    }

    private record RestartApplicationMessage(String mainClass) {
    }

    private record NoData() { // NOSONAR

    }

    private static Path folderInLaunchedModule;

    private static DevToolsInterface devToolsInterface;

    private static CopilotIDEPlugin instance;

    private final ObjectMapper objectMapper = new ObjectMapper();

    private final IdeUtils.IDE ideThatLaunchedTheApplication;

    CopilotIDEPlugin() {
        this.ideThatLaunchedTheApplication = IdeUtils.findIde().orElse(null);
    }

    /**
     * Gets instance of CopilotIDEPlugin. Project root must be set before.
     *
     * @return gets or create new instance of CopilotIDEPlugin
     */
    public static synchronized CopilotIDEPlugin getInstance() {
        if (instance == null) {
            instance = new CopilotIDEPlugin();
        }
        return instance;
    }

    /**
     * Sets a reference folder used to find the plugin metadata file.
     *
     * @param folderInLaunchedModule
     *            some folder inside the module containing the launched application
     */
    public static void setFolderInLaunchedModule(Path folderInLaunchedModule) {
        CopilotIDEPlugin.folderInLaunchedModule = folderInLaunchedModule;
    }

    /**
     * Sets dev tools interface for user notifications
     *
     * @param devToolsInterface
     */
    public static void setDevToolsInterface(DevToolsInterface devToolsInterface) {
        CopilotIDEPlugin.devToolsInterface = devToolsInterface;
    }

    /**
     * Check if plugin is active based on existing properties file
     *
     * @return true if active, false otherwise
     */
    public boolean isActive() {
        return getDotFile() != null;
    }

    public CopilotIDEPluginProperties getProperties() {
        return new CopilotIDEPluginProperties(getDotFile());
    }

    public IDEPluginInfo getPluginInfo() {
        if (!isActive()) {
            return new IDEPluginInfo(false, IdeUtils.findIde().map(IdeUtils.IDE::getPluginIde).orElse(null));
        }
        CopilotIDEPluginProperties props = new CopilotIDEPluginProperties(getDotFile());

        return new IDEPluginInfo(true, props.getVersion(), props.getSupportedActions(), props.getIde());
    }

    private String getProjectBasePath() throws IOException {
        File dotFile = getDotFile();
        if (dotFile == null) {
            throw new IOException("Dot file not found");
        }
        // assume that project base path is parent of IDE dot dir
        return dotFile.toPath().toRealPath().getParent().getParent().toString();
    }

    private File getDotFile() {
        if (ideThatLaunchedTheApplication != null) {
            // When running from an IDE, only accept connections from that IDE
            if (System.getProperty(PLUGIN_FILE_LOCATION_PROPERTY) != null) {
                File dotFile = new File(System.getProperty(PLUGIN_FILE_LOCATION_PROPERTY));
                if (dotFile.exists()) {
                    return dotFile;
                }
            }
            return getDotFile(ideThatLaunchedTheApplication);
        }

        // When not running from an IDE, try to find one to use
        for (IdeUtils.IDE ide : IdeUtils.IDE.values()) {
            File dotFile = getDotFile(ide);
            if (dotFile != null) {
                return dotFile;
            }
        }

        return null;
    }

    private File getDotFile(IdeUtils.IDE ide) {
        File dotFile;
        if (folderInLaunchedModule == null) {
            throw new IllegalStateException("Use setFolderInLaunchedModule before using the plugin");
        }

        Path dir = folderInLaunchedModule;
        do {
            dotFile = dir.resolve(ide.getMetaDir()).resolve(COPILOT_PLUGIN_DOTFILE).toFile();
            dir = dir.getParent();
        } while (dir != null && !dotFile.exists());
        if (dotFile.exists() && dotFile.canRead()) {
            return dotFile;
        }
        return null;
    }

    /**
     * Calls plugin write file operation
     *
     * @param file
     *            file to be written
     * @param undoLabel
     *            custom undo label
     * @param content
     *            file content
     * @throws IOException
     *             exception if command cannot be serialized
     */
    public Optional<JsonNode> writeFile(File file, String undoLabel, String content) throws IOException {
        if (!supports(Commands.WRITE)) {
            throw new UnsupportedOperationByPluginException(Commands.WRITE);
        }

        WriteFileMessage data = new WriteFileMessage(file.getAbsolutePath(), undoLabel, content);
        return send(Commands.WRITE.create(data));
    }

    /**
     * Calls plugin writeBase64 file operation
     *
     * @param file
     *            file to be written
     * @param undoLabel
     *            custom undo label
     * @param content
     *            file contents as base 64 encoded string
     * @throws IOException
     *             exception if command cannot be serialized
     */
    public Optional<JsonNode> writeBase64File(File file, String undoLabel, String content) throws IOException {
        if (!supports(Commands.WRITE_BASE64)) {
            throw new UnsupportedOperationByPluginException(Commands.WRITE_BASE64);
        }

        WriteFileMessage data = new WriteFileMessage(file.getAbsolutePath(), undoLabel, content);
        return send(Commands.WRITE_BASE64.create(data));
    }

    /**
     * Performs Undo for given files
     *
     * @param files
     *            list of files to perform undo
     * @throws IOException
     *             thrown on exception
     */
    public Optional<JsonNode> undo(List<String> files) throws IOException {
        if (!supports(Commands.UNDO)) {
            throw new UnsupportedOperationByPluginException(Commands.UNDO);
        }

        UndoRedoMessage data = new UndoRedoMessage(files);
        return send(Commands.UNDO.create(data));
    }

    /**
     * Performs Redo for given files
     *
     * @param files
     *            list of files to perform redo
     * @throws IOException
     *             thrown on exception
     */
    public Optional<JsonNode> redo(List<String> files) throws IOException {
        if (!supports(Commands.REDO)) {
            throw new UnsupportedOperationByPluginException(Commands.REDO);
        }

        UndoRedoMessage data = new UndoRedoMessage(files);
        return send(Commands.REDO.create(data));
    }

    /**
     * Opens editor and places caret on given line and column
     *
     * @param line
     *            line number, use 0 as first line
     * @param column
     *            column number to put caret before, 0 as first column
     * @throws IOException
     *             thrown on exception
     */
    public Optional<JsonNode> showInIde(String file, Integer line, Integer column) throws IOException {
        if (!supports(Commands.SHOW_IN_IDE)) {
            throw new UnsupportedOperationByPluginException(Commands.SHOW_IN_IDE);
        }

        ShowInIdeMessage data = new ShowInIdeMessage(file, line, column);
        return send(Commands.SHOW_IN_IDE.create(data));
    }

    /**
     * Sends request to synchronize project files with filesystem
     *
     * @throws IOException
     *             thrown on exception
     */
    public Optional<JsonNode> refresh() throws IOException {
        if (!supports(Commands.REFRESH)) {
            throw new UnsupportedOperationByPluginException(Commands.REFRESH);
        }

        return send(Commands.REFRESH.create(new NoData()));
    }

    public Optional<JsonNode> getModulePaths() {
        if (!supports(Commands.GET_MODULE_PATHS)) {
            throw new UnsupportedOperationByPluginException(Commands.GET_MODULE_PATHS);
        }
        return send(Commands.GET_MODULE_PATHS.create(new NoData()));
    }

    /**
     * Send requests to restart application
     *
     * @param mainClass
     *            name of main class to be matched with runner
     * @return response from IDE plugin
     */
    public Optional<JsonNode> restartApplication(String mainClass) {
        if (!supports(Commands.RESTART_APPLICATION)) {
            throw new UnsupportedOperationByPluginException(Commands.RESTART_APPLICATION);
        }

        RestartApplicationMessage data = new RestartApplicationMessage(mainClass);
        return send(Commands.RESTART_APPLICATION.create(data));
    }

    public Optional<JsonNode> heartbeat() {
        if (!supports(Commands.HEARTBEAT)) {
            throw new UnsupportedOperationByPluginException(Commands.HEARTBEAT);
        }
        return send(Commands.HEARTBEAT.create(new NoData()));
    }

    /**
     * Checks if given command is supported by plugin
     *
     * @param command
     *            command to be checked
     * @return true if supported, false otherwise
     */
    public boolean supports(Commands command) {
        return getProperties().getSupportedActions().contains(command.command);
    }

    private Optional<JsonNode> send(Command command) {
        if (getProperties().getEndpoint() != null) {
            return sendRestSync(command);
        } else {
            throw new IllegalStateException("Your plugin is outdated and needs to be updated");
        }
    }

    // rest client
    private Optional<JsonNode> sendRestSync(Command command) {
        try {
            RestCommand restCommand = new RestCommand(command.command, getProjectBasePath(), command.data);
            byte[] data = objectMapper.writeValueAsBytes(restCommand);
            HttpRequest request = HttpRequest.newBuilder().uri(URI.create(getProperties().getEndpoint()))
                    .header("Content-Type", "application/json").POST(HttpRequest.BodyPublishers.ofByteArray(data))
                    .build();
            HttpResponse<String> response = HttpClient.newHttpClient().send(request,
                    HttpResponse.BodyHandlers.ofString());
            if (response.statusCode() != 200) {
                throw new IOException("Unexpected response (" + response.statusCode()
                        + ") communicating with the IDE plugin: " + response.body());
            }
            if (response.body() != null && !response.body().isEmpty()) {
                JsonNode responseJson = JacksonUtils.readTree(response.body());
                handleIdeNotifications(responseJson);
                return Optional.of(responseJson);
            }
        } catch (IOException e) {
            if (e.getMessage() != null && e.getMessage().contains("429 Too Many Requests")) {
                throw new IdeTooManyRequestException();
            }
            Optional.ofNullable(getDotFile()).ifPresent(File::delete);
            getLogger().warn(
                    "Unable to communicate with IDE plugin, please check if plugin is running. You might need to reopen your project or reinstall plugin.");
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }

        return Optional.empty();
    }

    public static String undoLabel(String operation) {
        return UNDO_REDO_PREFIX + " " + operation;
    }

    private static Logger getLogger() {
        return LoggerFactory.getLogger(CopilotIDEPlugin.class);
    }

    private void handleIdeNotifications(JsonNode response) {
        if (response == null) {
            return;
        }

        if (response.isObject() && response.get("blockingPopup") != null) {
            IdeNotification notification = new IdeNotification(
                    "There is a popup in your IDE waiting for action, please check because it might block next Vaadin Copilot operations",
                    IdeNotification.Type.WARNING, "blocking-popup");
            devToolsInterface.send("copilot-ide-notification", notification.asJson());
            ((ObjectNode) response).remove("blockingPopup");
        }
    }
}
