package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;

import com.vaadin.flow.server.VaadinServletContext;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

/**
 * Utility class for configuring VS Code for hot code replace and Hotswap Agent
 * launch configurations.
 */
public final class VsCodeUtil {
    private static final ObjectMapper objectMapper = new ObjectMapper();

    private static final String HOTCODE_REPLACE_KEY = "java.debug.settings.hotCodeReplace";
    private static final String LAUNCH_CONFIGURATION_NAME = "Debug using Hotswap Agent";

    static {
        objectMapper.configure(JsonParser.Feature.ALLOW_COMMENTS, true);
        objectMapper.configure(SerializationFeature.INDENT_OUTPUT, true);
    }

    private static final String MAIN_CLASS = "mainClass";

    private VsCodeUtil() {
        // Util class
    }

    /**
     * Enables auto hot deploy of code changes for the given project in VS Code by
     * writing a project specific setting.
     *
     * @param projectFolder
     *            The project folder
     * @throws IOException
     *             If an I/O error occurs
     */
    public static void ensureAutoHotDeploy(File projectFolder) throws IOException {
        File settingsJson = new File(getOrCreateVsCodeFolder(projectFolder), "settings.json");
        if (!settingsJson.exists()) {
            Files.writeString(settingsJson.toPath(), "{}");
        }
        ObjectNode tree = (ObjectNode) objectMapper.readTree(settingsJson);
        tree.put(HOTCODE_REPLACE_KEY, "auto");
        objectMapper.writeValue(settingsJson, tree);
    }

    /**
     * Creates a launch configuration for Hotswap Agent in VS Code for the given
     * project.
     *
     * @param projectFolder
     *            The project folder
     * @param javaExecutable
     *            The path to the java executable
     * @param vaadinServletContext
     *            The Vaadin servlet context
     * @throws IOException
     *             If an I/O error occurs
     */
    public static void addHotswapAgentLaunchConfiguration(File projectFolder, File javaExecutable,
            VaadinServletContext vaadinServletContext) throws IOException {
        File launchJson = new File(getOrCreateVsCodeFolder(projectFolder), "launch.json");
        if (!launchJson.exists()) {
            Files.writeString(launchJson.toPath(), """
                    {
                      "version": "0.2.0",
                      "configurations": [
                      ]
                    }
                    """);
        }
        ObjectNode tree = (ObjectNode) objectMapper.readTree(launchJson);
        ArrayNode confArray = (ArrayNode) tree.get("configurations");
        String mainClass = null;
        if (confArray == null) {
            confArray = objectMapper.createArrayNode();
            tree.set("configurations", confArray);
        }

        int existingLaunchConfig = -1;
        for (int i = 0; i < confArray.size(); i++) {
            ObjectNode launchConf = (ObjectNode) confArray.get(i);
            if (mainClass == null && launchConf.has(MAIN_CLASS)) {
                mainClass = launchConf.get(MAIN_CLASS).asText();
            }
            if (launchConf.has("name") && launchConf.get("name").asText().equals(LAUNCH_CONFIGURATION_NAME)) {
                existingLaunchConfig = i;
            }
        }
        if (existingLaunchConfig != -1) {
            // Overwrite it, in case some settings are missing
            confArray.remove(existingLaunchConfig);
        }
        if (mainClass == null) {
            Class<?> appClass = SpringBridge.getApplicationClass(vaadinServletContext);
            if (appClass != null) {
                mainClass = appClass.getName();
            } else {
                mainClass = "com.example.Application";
            }
        }
        confArray.add(createLaunchConf(javaExecutable.getAbsolutePath(), mainClass));
        objectMapper.writeValue(launchJson, tree);
    }

    private static File getOrCreateVsCodeFolder(File projectFolder) throws IOException {
        File vsCodeFolder = new File(projectFolder, ".vscode");
        if (!vsCodeFolder.exists()) {
            if (!vsCodeFolder.mkdirs()) {
                throw new IOException("Unable to create folder " + vsCodeFolder.getAbsolutePath());
            }
        }
        return vsCodeFolder;
    }

    private static JsonNode createLaunchConf(String javaExecutable, String mainClass) {
        ObjectNode launchConf = objectMapper.createObjectNode();
        launchConf.put("type", "java");
        launchConf.put("name", LAUNCH_CONFIGURATION_NAME);
        launchConf.put("javaExec", javaExecutable);
        launchConf.put("request", "launch");
        launchConf.put(MAIN_CLASS, mainClass);

        List<String> vmArgs = new ArrayList<>();
        vmArgs.add("-XX:+AllowEnhancedClassRedefinition");
        vmArgs.add("-XX:+ClassUnloading");
        vmArgs.add("-XX:HotswapAgent=fatjar");
        launchConf.put("vmArgs", String.join(" ", vmArgs));
        return launchConf;
    }
}
