package com.vaadin.copilot;

import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.concurrent.ExecutionException;

import com.vaadin.copilot.exception.CopilotException;
import com.vaadin.copilot.exception.RouteDuplicatedException;
import com.vaadin.copilot.ide.CopilotIDEPlugin;
import com.vaadin.copilot.javarewriter.JavaRewriterUtil;
import com.vaadin.copilot.javarewriter.JavaSource;
import com.vaadin.flow.server.VaadinSession;
import com.vaadin.flow.shared.util.SharedUtil;

import com.github.javaparser.ParseProblemException;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import org.apache.commons.io.FilenameUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Handles creation of views for new routes.
 */
public class RouteCreator {

    private final NewRouteTemplateHandler newRouteTemplateHandler = new NewRouteTemplateHandler();

    private static Logger getLogger() {
        return LoggerFactory.getLogger(RouteCreator.class);
    }

    private final VaadinSession vaadinSession;

    static final String VIEW_FROM_IMAGE_TEMPLATE_NAME = "from-image";

    public RouteCreator(VaadinSession vaadinSession) {
        this.vaadinSession = vaadinSession;
    }

    /**
     * Creates a Flow view for the given route using the given access requirement.
     *
     * @param title
     *            the title of the view
     * @param route
     *            the path to create a view for
     * @param qualifiedClassNamePath
     *            qualified class name path that is used to create the file e.g.
     *            com.vaadin.org.MainView.java
     * @param accessRequirement
     *            the access requirement for the view
     * @param layoutClass
     *            the layout class to use, or {@code null} to not use a layout
     * @param templateFileName
     *            the file name to select the template of view
     * @param templateData
     *            data for the selected template, or {@code null}
     * @throws IOException
     *             if the view file could not be created
     */
    public void createFlowView(String title, String route, Path qualifiedClassNamePath,
            AccessRequirement accessRequirement, Class<?> layoutClass, String templateFileName, String templateData,
            Boolean addToMenu) throws IOException, RouteDuplicatedException {
        getLogger().debug("Creating Flow view for route {}", route);

        throwIfInvalidRoute(route);

        String className = qualifiedClassNamePath.getFileName().toString().replace(".java", "");
        if (!ProjectFileManager.get().sanitizeFilename(className).equals(className)) {
            throw new IllegalArgumentException("Invalid filename " + className);
        }

        File viewFile = qualifiedClassNamePath.toFile();

        String viewsPackage = ProjectFileManager.get().getJavaPackage(viewFile);

        if (existsRoute(route)) {
            throw new RouteDuplicatedException(route);
        }
        if (viewFile.exists()) {
            throw new RouteDuplicatedException(route);
        }
        viewFile.getParentFile().mkdirs();

        NewRouteTemplateHandler.FlowTemplateRequest flowTemplateRequest = new NewRouteTemplateHandler.FlowTemplateRequest(
                viewsPackage, route, title, className, accessRequirement, layoutClass, templateFileName, templateData,
                addToMenu);
        try {
            String content = newRouteTemplateHandler.getFlowTemplate(flowTemplateRequest).get();
            throwIfGeneratedFileSourceIsInvalid(content);
            ProjectFileManager.get().writeFile(viewFile, CopilotIDEPlugin.undoLabel("Add route"), content);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Unable to create route: " + e.getMessage(), e);
        } catch (ExecutionException e) {
            Throwable cause = e.getCause();
            if (cause instanceof IOException ioe) {
                throw ioe;
            }
            throw new IOException("Unable to create route: " + e.getMessage(), cause);
        }
    }

    private void throwIfGeneratedFileSourceIsInvalid(String content) {
        CompilationUnit compilationUnit;
        try {
            compilationUnit = new JavaSource(null, content, true).getCompilationUnit();
        } catch (ParseProblemException e) {
            throw new CopilotException("Failed to parse the file due to invalid user input", e.getCause());
        }

        var classDeclaration = compilationUnit.findFirst(ClassOrInterfaceDeclaration.class)
                .orElseThrow(() -> new CopilotException("Generated content is not valid"));
        String classSimpleName = classDeclaration.getNameAsString();
        boolean expressionSameWithClassNamePresent = classDeclaration.findAll(ObjectCreationExpr.class).stream()
                .anyMatch(objectCreationExpr -> classSimpleName.equals(objectCreationExpr.getType().getNameAsString()));
        if (expressionSameWithClassNamePresent) {
            throw new CopilotException("Invalid view name “" + classSimpleName
                    + "”: this name conflicts with an existing variable in the generated template. Use a unique view name to avoid collisions.");

        }
    }

    private void throwIfInvalidRoute(String route) {
        if (route.isEmpty()) {
            return;
        }
        if (!route.matches("^[a-zA-Z0-9-/]*$")) {
            throw new IllegalArgumentException(
                    "Routes can only contain letters, numbers, dashes and separators (/): " + route);
        }
        if (!route.matches(".*[a-zA-Z0-9]+.*")) {
            throw new IllegalArgumentException("Route must contain at least one letter or number: " + route);
        }
        if (route.contains("//")) {
            throw new IllegalArgumentException("Route must not contain consecutive slashes: " + route);
        }
    }

    static String getFlowViewName(String route) {
        String[] parts = route.split("/");
        String filename = parts[parts.length - 1];

        String identifier;
        if (filename.isEmpty() || filename.endsWith("/")) {
            identifier = "Main";
        } else {
            identifier = SharedUtil.capitalize(JavaRewriterUtil.getJavaIdentifier(filename, 100));
        }

        return identifier;
    }

    /**
     * Creates a Hilla view for the given route using the given access requirement.
     *
     * @param title
     *            the title of the view
     * @param route
     *            the path to create a view for
     * @param filePath
     *            the folder to add view
     * @param accessRequirement
     *            the access requirement for the view
     * @param templateFileName
     *            the selected template for the view
     * @param templateData
     *            data for the selected template, or {@code null}
     * @param addToMenu
     */
    public void createHillaView(String title, String route, Path filePath, AccessRequirement accessRequirement,
            String templateFileName, String templateData, boolean addToMenu)
            throws RouteDuplicatedException, IOException {
        getLogger().debug("Creating Hilla view for route {}", route);

        throwIfInvalidRoute(route);
        // Assumes FS router will set up routing
        String filenameWithPath = filePath.toFile().getAbsolutePath();
        if (filenameWithPath.isEmpty() || filenameWithPath.endsWith("/")) {
            filenameWithPath += "@index";
        }
        if (existsRoute(route)) {
            throw new RouteDuplicatedException(route);
        }
        if (filePath.toFile().exists()) {
            throw new RouteDuplicatedException(filePath.toString());
        }
        String viewName = filePath.toFile().getName().replace(".tsx", "");
        if (viewName.startsWith("@")) {
            viewName = "";
        }
        NewRouteTemplateHandler.HillaTemplateRequest hillaTemplateRequest = new NewRouteTemplateHandler.HillaTemplateRequest(
                viewName, title, accessRequirement, templateFileName, templateData, addToMenu);

        try {
            String content = newRouteTemplateHandler.getHillaTemplate(hillaTemplateRequest).get();
            ProjectFileManager.get().writeFile(filePath.toFile(), CopilotIDEPlugin.undoLabel("Add route"), content);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IOException("Unable to create route: " + e.getMessage(), e);
        } catch (ExecutionException e) {
            throw new IOException("Unable to create route: " + e.getMessage(), e.getCause());
        }
    }

    public static String getNewClassPackageNameRelative(String framework, VaadinSession session) {
        File directoryFolder;
        Path packageFolderPath;
        if ("hilla".equals(framework)) {
            directoryFolder = ProjectFileManager.get().getHillaViewsFolder();
        } else if ("flow".equals(framework)) {
            directoryFolder = ProjectFileManager.get().getFlowNewViewFolder(session);
        } else {
            throw new IllegalArgumentException("Unknown framework: " + framework);
        }
        packageFolderPath = getPackageFolder(framework);
        Path path = directoryFolder.toPath();
        String relativePathStr = packageFolderPath.relativize(path).toString();
        if ("flow".equals(framework)) {
            // replacing slash characters with '.'
            return relativePathStr.replace("\\\\", ".").replace("/", ".");
        }
        // it is more convenient to have file path separator for fs systems
        return relativePathStr;

    }

    public static Path getNewFilePath(String framework, String qualifiedClassNameOrViewFile) {
        String sanitizedFilePath;
        if ("hilla".equals(framework)) {
            String filePathToCreate = qualifiedClassNameOrViewFile;
            if (!FilenameUtils.isExtension(qualifiedClassNameOrViewFile, "tsx")) {
                filePathToCreate = qualifiedClassNameOrViewFile + ".tsx";
            }
            sanitizedFilePath = filePathToCreate;
        } else if ("flow".equals(framework)) {
            String filePathToCreate = qualifiedClassNameOrViewFile;
            if (FilenameUtils.isExtension(qualifiedClassNameOrViewFile, "java")) {
                filePathToCreate = qualifiedClassNameOrViewFile.substring(0,
                        qualifiedClassNameOrViewFile.lastIndexOf('.'));
            }
            filePathToCreate = filePathToCreate.replace(".", File.separator);
            filePathToCreate = filePathToCreate + ".java";
            sanitizedFilePath = filePathToCreate;
        } else {
            throw new IllegalArgumentException("Unknown framework: " + framework);
        }
        return getPackageFolder(framework).resolve(sanitizedFilePath);
    }

    private static Path getPackageFolder(String framework) {
        File packageFolder;
        if ("hilla".equals(framework)) {
            packageFolder = ProjectFileManager.get().getFrontendFolder();
        } else if ("flow".equals(framework)) {
            packageFolder = ProjectFileManager.get().getJavaSourceFolder();
        } else {
            throw new IllegalArgumentException("Unknown framework: " + framework);
        }
        return packageFolder.toPath();
    }

    private boolean existsRoute(String route) {
        try {
            RouteHandler.getServerRoutes(vaadinSession).stream()
                    .filter(routeData -> routeData.getTemplate().equals(route)).findAny().ifPresent(routeData -> {
                        throw new CopilotException("Route already exists: " + route);
                    });
        } catch (RuntimeException e) {
            return true;
        }
        return false;
    }
}
