package com.vaadin.copilot;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;

import com.vaadin.flow.internal.JsonDecodingException;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;

// missing Flow JacksonUtils methods used in Copilot
public class CopilotJacksonUtils {

    private static final ObjectMapper objectMapper = new ObjectMapper();
    static {
        objectMapper.registerModule(new JavaTimeModule());
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    }

    public static ObjectMapper getObjectMapper() {
        return objectMapper;
    }

    public static <T> T readValue(String string, Class<T> clazz) {
        try {
            return objectMapper.readValue(string, clazz);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    public static <T> T readValue(InputStream in, Class<T> clazz) {
        try {
            return objectMapper.readValue(in, clazz);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        } catch (IOException e) {
            throw new IllegalArgumentException("Unable to read JSON data from stream", e);
        }
    }

    public static <T> T readValue(InputStream in, TypeReference<T> typeReference) {
        try {
            return objectMapper.readValue(in, typeReference);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        } catch (IOException e) {
            throw new IllegalArgumentException("Unable to read JSON data from stream", e);
        }
    }

    public static String writeValueAsString(Object value) {
        try {
            return objectMapper.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            throw new IllegalArgumentException("Invalid data", e);
        }
    }

    public static ArrayNode readArray(String json) {
        try {
            return (ArrayNode) objectMapper.readTree(json);
        } catch (JsonProcessingException e) {
            throw new JsonDecodingException("Could not parse json content", e);
        }
    }

    public static HashMap<String, String> toMap(JsonNode data) {
        if (data == null) {
            return null;
        }
        return objectMapper.convertValue(data, new TypeReference<>() {
        });
    }

}
