// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.starter;

import jakarta.annotation.PostConstruct;

import jnr.constants.platform.Signal;
import jnr.posix.POSIX;
import jnr.posix.POSIXFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.availability.AvailabilityChangeEvent;
import org.springframework.boot.availability.ReadinessState;
import org.springframework.context.ApplicationContext;

class ReadinessStateSignalHandler {

    private static final POSIX POSIX = POSIXFactory.getPOSIX();

    private static final Logger LOGGER = LoggerFactory.getLogger(ReadinessStateSignalHandler.class);

    private final ApplicationContext applicationContext;

    ReadinessStateSignalHandler(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    @PostConstruct
    void registerSignalHandler() {
        POSIX.signal(Signal.SIGUSR1, this::handleAcceptingTrafficSignal);
        POSIX.signal(Signal.SIGUSR2, this::handleRefusingTrafficSignal);
    }

    private void handleAcceptingTrafficSignal(int signal) {
        LOGGER.debug("Accepting traffic signal received: {}", signal);
        AvailabilityChangeEvent.publish(applicationContext, ReadinessState.ACCEPTING_TRAFFIC);
    }

    private void handleRefusingTrafficSignal(int signal) {
        LOGGER.debug("Refusing traffic signal received: {}", signal);
        AvailabilityChangeEvent.publish(applicationContext, ReadinessState.REFUSING_TRAFFIC);
    }
}
