// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.starter;

import java.util.List;

import org.springframework.boot.context.properties.ConfigurationProperties;

/// Configuration properties for Control Center.
@ConfigurationProperties(prefix = ControlCenterProperties.PREFIX)
public class ControlCenterProperties {

    static final String PREFIX = "vaadin.control-center";

    /// The client-id to register for OpenID Connect authentication.
    private String clientId;

    /// The client-secret to register for OpenID Connect authentication.
    private String clientSecret;

    /// The scope of the OpenID Connect client registration.
    private List<String> clientScope = List.of("openid");

    /// The OpenID Connect registration-id.
    private String registrationId;

    /// The OpenID Connect issuer backend URI.
    private String issuerBackendUri;

    /// The OpenID Connect issuer frontend URI.
    private String issuerFrontendUri;

    /// Create a new instance.
    public ControlCenterProperties() {
        super();
    }

    /// Gets the client-id to register for OpenID Connect authentication.
    ///
    /// @return the client-id
    public String getClientId() {
        return clientId;
    }

    /// Sets the client-id to register for OpenID Connect authentication.
    ///
    /// @param clientId the client-id
    public void setClientId(String clientId) {
        this.clientId = clientId;
    }

    /// Gets the client-secret to register for OpenID Connect authentication.
    ///
    /// @return the client-secret
    public String getClientSecret() {
        return clientSecret;
    }

    /// Sets the client-secret to register for OpenID Connect authentication.
    ///
    /// @param clientSecret the client-secret
    public void setClientSecret(String clientSecret) {
        this.clientSecret = clientSecret;
    }

    /// Gets the scope of the OpenID Connect client registration.
    ///
    /// @return the scope
    public List<String> getClientScope() {
        return List.copyOf(clientScope);
    }

    /// Sets the scope of the OpenID Connect client registration.
    ///
    /// @param clientScope the scope
    public void setClientScope(List<String> clientScope) {
        this.clientScope = List.copyOf(clientScope);
    }

    /// Gets the OpenID Connect registration-id.
    ///
    /// @return the registration-id
    public String getRegistrationId() {
        return registrationId;
    }

    /// Sets the OpenID Connect registration-id.
    ///
    /// @param registrationId the registration-id
    public void setRegistrationId(String registrationId) {
        this.registrationId = registrationId;
    }

    /// Gets the OpenID Connect issuer backend URI.
    ///
    /// @return the issuer backend URI
    public String getIssuerBackendUri() {
        return issuerBackendUri;
    }

    /// Sets the OpenID Connect issuer backend URI.
    ///
    /// @param issuerBackendUri the issuer backend URI
    public void setIssuerBackendUri(String issuerBackendUri) {
        this.issuerBackendUri = issuerBackendUri;
    }

    /// Gets the OpenID Connect issuer frontend URI.
    ///
    /// @return the issuer frontend URI
    public String getIssuerFrontendUri() {
        return issuerFrontendUri;
    }

    /// Sets the OpenID Connect issuer frontend URI.
    ///
    /// @param issuerFrontendUri the issuer frontend URI
    public void setIssuerFrontendUri(String issuerFrontendUri) {
        this.issuerFrontendUri = issuerFrontendUri;
    }
}
