// Copyright (C) 2025 Vaadin Ltd
// This program is available under Vaadin Commercial License and Service Terms.
// See <https://vaadin.com/commercial-license-and-service-terms> for the full license.
package com.vaadin.controlcenter.starter;

import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;

import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.client.KubernetesClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.kubernetes.commons.PodUtils;
import org.springframework.security.core.Authentication;
import org.springframework.security.web.authentication.logout.LogoutHandler;
import org.springframework.web.client.RestClient;
import org.springframework.web.util.UriComponentsBuilder;

class KubernetesBackChannelLogoutHandler implements LogoutHandler {

    private static final Logger LOGGER = LoggerFactory.getLogger(KubernetesBackChannelLogoutHandler.class);

    private static final String BROADCAST_HEADER = "X-OIDC-BackChannel-Logout-Broadcast";

    private final KubernetesClient kubernetesClient;

    private final PodUtils<Pod> podUtils;

    private final LogoutHandler delegate;

    private final RestClient restClient;

    KubernetesBackChannelLogoutHandler(
            RestClient.Builder restClientBuilder,
            KubernetesClient kubernetesClient,
            PodUtils<Pod> podUtils,
            LogoutHandler delegate) {
        this.kubernetesClient = kubernetesClient;
        this.podUtils = podUtils;
        this.delegate = delegate;
        this.restClient = restClientBuilder.build();
    }

    @Override
    public void logout(HttpServletRequest request, HttpServletResponse response, Authentication authentication) {
        var broadcastHeader = request.getHeader(BROADCAST_HEADER);
        if (podUtils.isInsideKubernetes() && broadcastHeader == null) {
            LOGGER.debug("No broadcast header in OIDC Back-Channel Logout request, broadcasting request to pods");
            var metadata = podUtils.currentPod().get().getMetadata();
            var namespace = metadata.getNamespace();
            var labels = metadata.getLabels();
            var token = request.getParameter("logout_token");
            kubernetesClient
                    .pods()
                    .inNamespace(namespace)
                    .withLabels(labels)
                    .list()
                    .getItems()
                    .forEach(pod -> {
                        var ip = pod.getStatus().getPodIP();
                        LOGGER.debug("Broadcasting OIDC Back-Channel Logout request to Pod with IP {}", ip);
                        var uri = UriComponentsBuilder.fromUriString(request.getRequestURI())
                                .scheme("http")
                                .host(ip)
                                .port(8080)
                                .queryParam("logout_token", token)
                                .toUriString();
                        restClient
                                .post()
                                .uri(uri)
                                .header(BROADCAST_HEADER, "true")
                                .retrieve()
                                .toBodilessEntity();
                    });
        } else if (!podUtils.isInsideKubernetes() || broadcastHeader != null) {
            LOGGER.debug("Found broadcast header in OIDC Back-Channel Logout request (or not inside Kubernetes)");
            delegate.logout(request, response, authentication);
        }
    }
}
